﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using System.Collections.Generic;
using nw.g3d.iflib;
using System.Xml;
using System.Xml.Schema;
using System.IO;
using System.Reflection;
using _3dToolsTestUtility;

namespace _3dIntermediateFileLibraryTest
{
    [TestClass]
    public class SerializerTest
    {
        /// <summary>
        /// original_image を持たないテクスチャー中間ファイルの入出力テストです。
        /// </summary>
        [TestMethod]
        public void WriteReadNoOriginalImageTextureTest()
        {
            G3dParallel.Job = 1;
            nw4f_3difType writeFile = new nw4f_3difType();
            textureType tex = new textureType();
            writeFile.Item = tex;
            var info = new texture_infoType();
            tex.texture_info = info;
            info.width = 1;
            info.height = 1;
            info.depth = 1;
            info.mip_level = 1;
            info.size = 1;
            info.dcc_preset = string.Empty;
            info.hint = string.Empty;
            info.original_image_hash = string.Empty;
            info.mip_gen_filter = string.Empty;
            info.quantize_type = texture_info_quantize_typeType.unorm_8;
            info.linear = new bool[] { false, false, false, false};
            info.comp_sel = new texture_info_comp_selValue[]
            {
                texture_info_comp_selValue.r,
                texture_info_comp_selValue.r,
                texture_info_comp_selValue.r,
                texture_info_comp_selValue.r,
            };
            info.level_offset = new uint[] { 0 };

            string tempFilePath = IoUtility.GetTempFilePath(MethodBase.GetCurrentMethod().Name + ".ftxb");

            try
            {
                IfWriteUtility.Write(writeFile, tempFilePath, IoUtility.GetXsdBasePath());
                nw4f_3difType readFile = IfReadUtility.Read(tempFilePath, IoUtility.GetXsdBasePath());
                Assert.IsNotNull(readFile.Item as textureType);
            }
            finally
            {
                if (System.IO.File.Exists(tempFilePath))
                {
                    System.IO.File.Delete(tempFilePath);
                }
            }
        }

        /// <summary>
        /// コンバイナー中間ファイルの入出力テストです。
        /// </summary>
        [TestMethod]
        public void WriteReadCombinerTest()
        {
            G3dParallel.Job = 1;
            combiner_shaderType writeData = new combiner_shaderType()
            {
                version = combiner_shader_versionType.Item400,
                material = CreateTestMaterial(),
                x = 0,
                y = 0,
                block_array = new block_arrayType()
                {
                    block = new blockType[]
                    {
                        new blockType()
                        {
                            id = "test",
                            guid = "test",
                            @const = new constType()
                            {
                                type = "test",
                                values = "test",
                            }
                        }
                    }
                },
                connection_array = new connection_arrayType()
                {
                    connection = new connectionType[]
                    {
                        new connectionType()
                        {
                            source = new sourceType()
                            {
                                block = "test"
                            },
                            target = new targetType()
                            {
                                block = "test"
                            }
                        }
                    }
                },
            };

            nw4f_3difType writeFile = new nw4f_3difType();
            writeFile.Item = writeData;

            string tempFilePath = IoUtility.GetTempFilePath(MethodBase.GetCurrentMethod().Name + ".fcmb");
            IfWriteUtility.Write(writeFile, tempFilePath, IoUtility.GetXsdBasePath());

            try
            {

                nw4f_3difType readFile = IfReadUtility.Read(tempFilePath, IoUtility.GetXsdBasePath());
                var readData = readFile.Item as combiner_shaderType;
                Assert.AreEqual(writeData.material.name, readData.material.name);
            }
            finally
            {
                if (System.IO.File.Exists(tempFilePath))
                {
                    System.IO.File.Delete(tempFilePath);
                }
            }
        }

        /// <summary>
        /// マテリアル中間ファイルの入出力テストです。
        /// </summary>
        [TestMethod]
        public void WriteReadMaterialTest()
        {
            G3dParallel.Job = 1;
            string tempFilePath = IoUtility.GetTempFilePath(MethodBase.GetCurrentMethod().Name + ".fmta");
            nw4f_3difType writeFile = new nw4f_3difType();
            List<G3dStream> streams = new List<G3dStream>();
            G3dStream stream = new G3dStream();
            stream.type = stream_typeType.@float;
            stream.FloatData.AddRange(new float[] { 0.1f, 0.2f, 0.3f});
            stream.column = 3;
            streams.Add(stream);
            materialType writeMat = CreateTestMaterial();
            writeMat.stream_array = G3dStreamUtility.ToStreamArray(streams);
            writeFile.Item = writeMat;

            IfWriteUtility.Write(writeFile, tempFilePath, IoUtility.GetXsdBasePath());

            try
            {

                nw4f_3difType readFile = IfReadUtility.Read(tempFilePath, IoUtility.GetXsdBasePath());
                var readMat = readFile.Item as materialType;
                Assert.AreEqual(writeMat.name, readMat.name);
            }
            finally
            {
                if (System.IO.File.Exists(tempFilePath))
                {
                    System.IO.File.Delete(tempFilePath);
                }
            }
        }

        /// <summary>
        /// 最新の中間ファイルが読み書きできるかどうかをテストします。
        /// </summary>
        [TestMethod]
        public void ReadWriteLatestVersionTest()
        {
            G3dParallel.Job = 1;
            foreach (string inputFilePath in new string[]
                {
                    "Resources/cube.fmda",
                    "Resources/cube.ftxa",
                    "Resources/cube_anim.fska",
                    "Resources/cube_anim.fmaa",
                    "Resources/cube_anim.fspa",
                    "Resources/cube_anim.ftpa",
                    "Resources/cube_anim.fvba",
                    "Resources/cube_anim.fvma",
                    "Resources/cube_anim.fsha",
                    "Resources/cube_anim.fsna",
                    "Resources/cube.fsca",
                    "Resources/cube.fsva",
                    "Resources/cube.fsda",
                })
            {
                bool isUpdated;
                nw4f_3difType readFile = IfReadUtility.Read(
                        inputFilePath,
                        IoUtility.GetXsdBasePath(),
                        out isUpdated);
                Assert.IsFalse(isUpdated, $"{inputFilePath}");

                string outFilePath = IoUtility.GetTempFilePath(System.IO.Path.GetFileName(inputFilePath));
                IfWriteUtility.Write(readFile, $"{outFilePath}");
                System.IO.File.Delete(outFilePath);
            }
        }

        /// <summary>
        /// 古い中間ファイルを最新の中間ファイルバージョンに更新できるかどうかをテストします。
        /// </summary>
        [TestMethod]
        public void VersionUpdateTest()
        {
            G3dParallel.Job = 1;
            foreach (string inputFilePath in new string[]
                {
                    "Resources/VersionUpdateTestCharacterModel.fmdb",
                    "Resources/VersionUpdateTestFieldModel.fmdb",
                    "Resources/VersionUpdateTest.ftxb",
                    "Resources/VersionUpdateTest.fskb",
                    "Resources/VersionUpdateTest.fmab",
                    "Resources/VersionUpdateTest.fspb",
                    "Resources/VersionUpdateTest.ftpb",
                    "Resources/VersionUpdateTest.fvbb",
                    "Resources/VersionUpdateTest.fvmb",
                    "Resources/VersionUpdateTest.fshb",
                    "Resources/VersionUpdateTest.fsnb",
                    "Resources/VersionUpdateTest.fsca",
                    "Resources/VersionUpdateTest.fsva",
                })
            {
                bool isUpdated;
                nw4f_3difType readFile = IfReadUtility.Read(
                        inputFilePath,
                        IoUtility.GetXsdBasePath(),
                        out isUpdated);
                Assert.IsTrue(isUpdated, $"{inputFilePath}");

                string outFilePath = IoUtility.GetTempFilePath(System.IO.Path.GetFileName(inputFilePath));

                IfWriteUtility.Write(readFile, $"{outFilePath}");

                // 更新後に再度読み書きできるかテスト
                nw4f_3difType updatedFile = IfReadUtility.Read(
                    outFilePath,
                    IoUtility.GetXsdBasePath(),
                    out isUpdated);
                Assert.IsFalse(isUpdated, $"{outFilePath}");
                IfWriteUtility.Write(updatedFile, $"{outFilePath}");

                System.IO.File.Delete(outFilePath);
            }
        }

        /// <summary>
        /// テキスト部分のみを読み込む API が正しく動作するかをテストします。
        /// </summary>
        [TestMethod]
        public void TestReadTextPartOnly()
        {
            string xsdBasePath = "../../../../../../../Tools/Graphics/3dTools/3dIntermediateFileXsd";
            nw4f_3difType readFile = IfBinaryReadUtility.ReadTextPartOnly(
                    "Resources/ReadTextPartOnlyTest.ftxb", xsdBasePath);

            Assert.IsNotNull(readFile);
            var texture = readFile.Item as textureType;
            Assert.IsNotNull(texture);
            Assert.IsTrue(texture.original_image_array.Items.Length == 1);
            Assert.IsTrue(texture.original_image_array.Items[0].format == original_image_formatType.rgba8);
        }

        /// <summary>
        /// カメラアニメーションをデシリアライズ、書き換え、シリアライズするテストです。
        /// </summary>
        [TestMethod]
        public void TestDeserializeSerializeCameraAnim()
        {
            G3dParallel.Job = 1;
            string xsdBasePath = "../../../../../../../Tools/Graphics/3dTools/3dIntermediateFileXsd";
            nw4f_3difType readFile =
                IfReadUtility.Read(
                    "Resources/SceneAnimTest.fsnb",
                    xsdBasePath);
            var readSceneAnim = readFile.Item as scene_animType;

            foreach (camera_animType cameraAnim in readSceneAnim.camera_anim_array.camera_anim)
            {
                Debug.WriteLine(string.Format("frame count: {0}", cameraAnim.frame_count));
                Debug.WriteLine(string.Format("loop: {0}", cameraAnim.loop));
                Debug.WriteLine(string.Format("projection mode: {0}", cameraAnim.projection_mode));
                Debug.WriteLine(string.Format("rotate mode: {0}", cameraAnim.rotate_mode));

                foreach (camera_anim_targetType cameraAnimTarget in cameraAnim.camera_anim_target)
                {
                    Debug.WriteLine(string.Format("base value: {0}", cameraAnimTarget.base_value));
                    if (cameraAnimTarget.Curve != null)
                    {
                        List<G3dStream> streamArray = new List<G3dStream>();
                        G3dStreamUtility.ToStreams(streamArray, readSceneAnim.stream_array);
                        G3dStream stream = streamArray[cameraAnimTarget.Curve.stream_index];

                        Assert.AreEqual(4, stream.column);
                        int curveKeyCount = stream.FloatData.Count / stream.column;

                        for (int keyIndex = 0; keyIndex < curveKeyCount; ++keyIndex)
                        {
                            // 1カラムには順番にフレーム、値、前の傾き、後の傾きが格納されている
                            float frame = stream.FloatData[keyIndex * 4 + 0];
                            float value = stream.FloatData[keyIndex * 4 + 1];
                            float previousKeySlope = stream.FloatData[keyIndex * 4 + 2];
                            float nextKeySlope = stream.FloatData[keyIndex * 4 + 3];
                            Debug.WriteLine(string.Format("frame: {0}\n", frame));
                            Debug.WriteLine(string.Format("value: {0}\n", value));
                            Debug.WriteLine(string.Format("previous key slope: {0}\n", previousKeySlope));
                            Debug.WriteLine(string.Format("next key slope: {0}\n", nextKeySlope));
                        }
                    }
                }
            }

            {
                nw4f_3difType writeFile = new nw4f_3difType();
                var writeSceneAnim = new scene_animType();
                writeFile.Item = writeSceneAnim;

                // アニメ情報の設定
                {
                    var info = new scene_anim_infoType();
                    writeSceneAnim.scene_anim_info = readSceneAnim.scene_anim_info;
                }

                // カメラアニメーションの設定
                {
                    writeSceneAnim.camera_anim_array = new camera_anim_arrayType();

                    camera_animType cameraAnim = new camera_animType();
                    writeSceneAnim.camera_anim_array.camera_anim = new camera_animType[1];
                    writeSceneAnim.camera_anim_array.camera_anim[0] = cameraAnim;
                    cameraAnim.camera_name = "CameraA";
                    cameraAnim.rotate_mode = camera_anim_rotate_modeType.aim;
                    cameraAnim.projection_mode = camera_anim_projection_modeType.persp;
                    cameraAnim.loop = false;

                    camera_anim_targetType cameraAnimTarget = new camera_anim_targetType();
                    cameraAnim.camera_anim_target = new camera_anim_targetType[1];
                    cameraAnim.camera_anim_target[0] = cameraAnimTarget;
                    cameraAnimTarget.target = camera_anim_target_targetType.position_x;
                    cameraAnimTarget.base_value = 19.0f;

                    cameraAnimTarget.Curve = new hermite_curveType();
                    cameraAnimTarget.Curve.stream_index = 0;
                    cameraAnimTarget.Curve.count = 2;
                    cameraAnimTarget.Curve.frame_type = curve_frame_typeType.frame32;
                    cameraAnimTarget.Curve.key_type = curve_key_typeType.key32;
                    cameraAnimTarget.Curve.scale = 1.0f;
                    cameraAnimTarget.Curve.offset = 0.0f;
                    cameraAnimTarget.Curve.pre_wrap = curve_wrapType.clamp;
                    cameraAnimTarget.Curve.post_wrap = curve_wrapType.clamp;
                }

                // ストリームの設定
                List<G3dStream> streamList = new List<G3dStream>();
                {
                    G3dStream stream = new G3dStream();
                    stream.column = 4;

                    stream.FloatData.Add(1.0f); // フレーム
                    stream.FloatData.Add(10.0f); // 値
                    stream.FloatData.Add(0.0f); // 前の傾き
                    stream.FloatData.Add(0.0f); // 後ろの傾き

                    stream.FloatData.Add(100.0f); // フレーム
                    stream.FloatData.Add(30.0f); // 値
                    stream.FloatData.Add(0.0f); // 前の傾き
                    stream.FloatData.Add(0.0f); // 後ろの傾き

                    streamList.Add(stream);
                }

                writeSceneAnim.stream_array = G3dStreamUtility.ToStreamArray(streamList);

                string outFilePath = IoUtility.GetTempFilePath(System.IO.Path.GetFileName("SceneAnimTestOut.fsnb"));
                IfWriteUtility.Write(
                    writeFile,
                    outFilePath);

                // xsd 違反の設定を行ったときにエラーになることをテスト
                writeSceneAnim.scene_anim_info.dcc_magnify = 0.0f;
                bool isExceptionOccurred = false;
                try
                {
                    IfWriteUtility.Write(
                        writeFile,
                        "Resources/SceneAnimTestOut.fsnb",
                        xsdBasePath);
                }
                catch (Exception)
                {
                    isExceptionOccurred = true;
                }

                Assert.IsTrue(isExceptionOccurred);

                System.IO.File.Delete(outFilePath);
            }
        }

        private materialType CreateTestMaterial()
        {
            return new materialType()
            {
                name = "testMat",
                version = material_versionType.Item400,
                versionSpecified = true,
                material_info = new material_infoType()
                {
                },
                render_state = new render_stateType()
                {
                    alpha_test = new alpha_testType(),
                    color_blend = new color_blendType()
                    {
                        const_color = new float[4],
                    },
                    depth_test = new depth_testType(),
                    display_face = new render_state_display_faceType(),
                    logical_blend = new logical_blendType(),
                },
                comment = new commentType()
                {
                    label = "hoge",
                    text = "fuga",
                    color = "0 0 0",
                },
                process_log_array = new process_log_arrayType()
                {
                    process_log = new process_logType[]
                    {
                        new process_logType()
                        {
                            argument = "hoge",
                            process = "fuga",
                        },
                    },
                    length = 1,
                },
                sampler_array = new sampler_arrayType()
                {
                    sampler = new samplerType[]
                    {
                        new samplerType()
                        {
                            name = "hoge",
                            hint = string.Empty,
                            tex_name = string.Empty,
                            wrap = new wrapType()
                            {
                            },
                            filter = new filterType(),
                            lod = new lodType()
                        },
                    },
                    length = 1,
                },
                shader_assign = new shader_assignType()
                {
                    shader_archive = "hoge",
                    shading_model = "fuga",
                    attrib_assign_array = new attrib_assign_arrayType()
                    {
                        attrib_assign = new attrib_assignType[]
                        {
                            new attrib_assignType()
                            {
                                id = "hoge",
                                attrib_name = "hoge",
                            }
                        },
                        length = 1
                    },
                    sampler_assign_array = new sampler_assign_arrayType()
                    {
                        sampler_assign = new sampler_assignType[]
                        {
                            new sampler_assignType()
                            {
                                id = "hoge",
                                sampler_name = "hoge",
                            }
                        },
                        length = 1
                    },
                    render_info_array = new render_info_arrayType()
                    {
                        render_info = new render_infoType[]
                        {
                            new render_infoType()
                            {
                                name = "hoge",
                                Value = "hoge",
                            }
                        },
                        length = 1
                    },
                    shader_option_array = new shader_option_arrayType()
                    {
                        shader_option = new shader_optionType[]
                        {
                            new shader_optionType()
                            {
                                id = "hoge",
                                value = "hoge",
                            }
                        },
                        length = 1
                    },
                    shader_param_array = new shader_param_arrayType()
                    {
                        shader_param = new shader_paramType[]
                        {
                            new shader_paramType()
                            {
                                id = "hoge",
                                type = shader_param_typeType.@float,
                                Value = "0.0",
                                original_hint = string.Empty,
                                depend = string.Empty,
                            }
                        },
                        length = 1
                    },
                },
                user_data_array = new user_data_arrayType()
                {
                    user_data = new user_dataType[]
                    {
                        new user_dataType()
                        {
                            name = "hoge",
                            Item = new user_floatType()
                            {
                                count = 1,
                                Value = "0.1 0.2 0.3",
                            }
                        }
                    },
                    length = 1,
                },
            };
        }
    }
}
