﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.Reflection;
using nw.g3d.toollib;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib.nw3de;

namespace _3dIntermediateFileAssignUtilityTest
{
    /// <summary>
    /// assign-material-reference-behavior のテストです。
    /// </summary>
    [TestClass]
    public class AssignMaterialReferenceBehaviorTest
    {
        /// <summary>
        /// --unassigned-param-only オプションのテストです。
        /// </summary>
        [TestMethod]
        public void UnassignedParamOnly()
        {
            G3dParallel.Job = 1;
            var resourceFolder = GetResourcePath(out var inputFilePath, out var outputFilePath);
            {
                nw.g3d.ifassign.g3difassign.XsdBasePath = GetXsdBasePath();
                string args = $"-j 1 assign-material-reference-behavior {inputFilePath} -o {outputFilePath} --material-reference-behavior-assign \"--type AttribAssign --child-restriction ForceRefer --unassigned-param-only\"";

                try
                {
                    nw.g3d.ifassign.g3difassign tool = new nw.g3d.ifassign.g3difassign();

                    {
                        tool.Arguments = $"{args} --shader-path {resourceFolder}";
                        tool.Run();

                        // 出力後の fmdb が読み込めるかをテスト
                        nw4f_3difType file = IfReadUtility.Read(outputFilePath, nw.g3d.ifassign.g3difassign.XsdBasePath);
                        modelType model = file.Item as modelType;
                        Assert.IsNotNull(model);

                        // シェーダー定義を渡した時は fmdb に含まれない頂点属性割り当て分の参照設定も設定される
                        var material = model.material_array.Items.FirstOrDefault(x => x.name == "ClothBase");
                        Assert.IsNotNull(material);
                        var materialReference = IfToolData.GetMaterialReference(material);

                        // 元から設定されている ForceOverride が維持される
                        int forceOverrideCount = materialReference.AttribAssignBehaviors.Count(x => x.ChildRestriction == ShaderItemRestrictionState.ForceOverride);
                        Assert.AreEqual(1, forceOverrideCount);

                        // その他のアトリビュートは指定した値になっている
                        int forceReferCount = materialReference.AttribAssignBehaviors.Count(x => x.ChildRestriction == ShaderItemRestrictionState.ForceRefer);
                        Assert.AreEqual(12, forceReferCount);
                    }
                }
                finally
                {
                    if (System.IO.File.Exists(outputFilePath))
                    {
                        System.IO.File.Delete(outputFilePath);
                    }
                }
            }
        }

        /// <summary>
        /// 全頂点属性に ForceOverride を割り当てるテストです。
        /// </summary>
        [TestMethod]
        public void AssignForceOverrideToAttributeAssign()
        {
            G3dParallel.Job = 1;
            var resourceFolder = GetResourcePath(out var inputFilePath, out var outputFilePath);
            {
                nw.g3d.ifassign.g3difassign.XsdBasePath = GetXsdBasePath();
                string args = $"-j 1 assign-material-reference-behavior {inputFilePath} -o {outputFilePath} --material-reference-behavior-assign \"--type AttribAssign --child-restriction ForceOverride\"";

                try
                {
                    nw.g3d.ifassign.g3difassign tool = new nw.g3d.ifassign.g3difassign();

                    // --shader-path 指定しない時のテスト
                    {
                        tool.Arguments = args;
                        tool.Run();

                        // 出力後の fmdb が読み込めるかをテスト
                        nw4f_3difType file = IfReadUtility.Read(outputFilePath, nw.g3d.ifassign.g3difassign.XsdBasePath);
                        modelType model = file.Item as modelType;
                        Assert.IsNotNull(model);

                        // 元々の fmdb には頂点属性割り当てが含まれていないので、参照設定も設定されない
                        var materials = model.material_array?.material;
                        Assert.IsNotNull(materials);
                        var materialReferences = materials.Select(IfToolData.GetMaterialReference).Where(x => x?.IsEmpty() == false).ToList();

                        // テストデータに元から 1 マテリアルだけ設定されているものがある
                        Assert.AreEqual(1, materialReferences.Count);
                        foreach (var matRefBehavior in materialReferences)
                        {
                            // テストデータに元から 1 つだけ設定を持つ
                            Assert.AreEqual(1, matRefBehavior.AttribAssignBehaviors.Count);
                        }
                    }

                    // --shader-path 指定時のテスト
                    {
                        tool.Arguments = $"{args} --shader-path {resourceFolder}";
                        tool.Run();

                        // 出力後の fmdb が読み込めるかをテスト
                        nw4f_3difType file = IfReadUtility.Read(outputFilePath, nw.g3d.ifassign.g3difassign.XsdBasePath);
                        modelType model = file.Item as modelType;
                        Assert.IsNotNull(model);

                        // シェーダー定義を渡した時は fmdb に含まれない頂点属性割り当て分の参照設定も設定される
                        var materials = model.material_array?.material;
                        Assert.IsNotNull(materials);
                        var materialReferences = materials.Select(IfToolData.GetMaterialReference).Where(x => x?.IsEmpty() == false).ToList();
                        Assert.IsTrue(materialReferences.Count > 0);

                        foreach (var materialReference in materialReferences)
                        {
                            Assert.IsTrue(materialReference.AttribAssignBehaviors.Count > 0);
                            foreach (var behavior in materialReference.AttribAssignBehaviors)
                            {
                                Assert.AreEqual(ShaderItemRestrictionState.ForceOverride, behavior.ChildRestriction);
                            }
                        }
                    }
                }
                finally
                {
                    if (System.IO.File.Exists(outputFilePath))
                    {
                        System.IO.File.Delete(outputFilePath);
                    }
                }
            }
        }

        /// <summary>
        /// サンプラーにマテリアル参照情報を割り当てるテストです。
        /// </summary>
        string[,] assignInfo =
        {
            {"MetalBase", "_a0", "tex_name",   "Override", "ForceOverride"},
            {"MetalBase", "_a0", "wrap_u",     "Refer",    null},
            {"MetalBase", "_a0", "wrap_v",     "Override", null},
            {"MetalBase", "_a0", "wrap_w",     "Default",  null},
            {"MetalBase", "_a0", "filter_mag", null,       "ForceOverride"},
            {"MetalBase", "_a0", "filter_min", null,       "ForceRefer"},
            {"MetalBase", "_a0", "filter_mip", null,       "None"},
            {"MetalBase", "_n0", null,         "Default",  "ForceRefer"},
            {null,        "_x0", "lod_min",    "Default",  "ForceOverride"},
        };

        [TestMethod]
        public void AssignMaterialReferenceInfoToSampler()
        {
            G3dParallel.Job = 1;
            var resourceFolder = GetResourcePath(out var inputFilePath, out var outputFilePath);
            {
                nw.g3d.ifassign.g3difassign.XsdBasePath = GetXsdBasePath();
                var baseargs = $"-j 1 assign-material-reference-behavior {inputFilePath} -o {outputFilePath}";

                try
                {
                    nw.g3d.ifassign.g3difassign tool = new nw.g3d.ifassign.g3difassign();

                    // --shader-path 指定しない時のテスト
                    {
                        void GetAssignInfo(int i, out string materialName, out string sampler, out string id, out string value, out string restriction)
                        {
                            materialName = assignInfo[i, 0];
                            sampler = assignInfo[i, 1];
                            id = assignInfo[i, 2];
                            value = assignInfo[i, 3];
                            restriction = assignInfo[i, 4];
                        }

                        var args = baseargs;
                        // assignInfo 配列のサイズを確認
                        Assert.IsNotNull(assignInfo);
                        Assert.IsTrue(assignInfo.GetLength(0) > 0);
                        Assert.IsTrue(assignInfo.GetLength(1) == 5);

                        // assignInfo に従ってオプションを指定
                        for (var i = 0; i < assignInfo.GetLength(0); i++)
                        {
                            var arg = string.Empty;
                            GetAssignInfo(i, out var materialName, out var sampler, out var id, out var value, out var restriction);

                            if (!string.IsNullOrEmpty(materialName)) arg += $" --material {materialName} ";
                            if (!string.IsNullOrEmpty(sampler) ) arg += $" --sampler {sampler} ";
                            if (!string.IsNullOrEmpty(id)) arg += $" --id {id} ";
                            if (!string.IsNullOrEmpty(value)) arg += $" --value {value} ";
                            if (!string.IsNullOrEmpty(restriction)) arg += $" --child-restriction {restriction} ";
                            Assert.IsFalse(string.IsNullOrEmpty(arg));
                            args += $" --material-reference-behavior-assign \"{arg}\"";
                        }

                        args +=" --material-reference-sampler-behavior-assign \"--material FaceBase --sampler-structure-restriction DisallowAddOwnSampler\"";
                        args += " --material-reference-sampler-behavior-assign \"--material SkinBase --sampler-structure-restriction None\"";
                        args += " --material-reference-sampler-behavior-assign \"--material FaceBase --sampler _p0 --is-sampler-required true\"";
                        args += " --material-reference-sampler-behavior-assign \"--material FaceBase --sampler _n0 --is-sampler-required false\"";

                        tool.Arguments = args;
                        tool.Run();

                        // 出力後の fmdb が読み込めるかをテスト
                        nw4f_3difType file = IfReadUtility.Read(outputFilePath, nw.g3d.ifassign.g3difassign.XsdBasePath);
                        modelType model = file.Item as modelType;
                        Assert.IsNotNull(model);
                        var materials = model.material_array?.material;
                        Assert.IsNotNull(materials);

                        // assignInfo 通りに指定されているかどうか確認
                        for (var i = 0; i < assignInfo.GetLength(0); i++)
                        {
                            GetAssignInfo(i, out var materialName, out var sampler, out var id, out var value, out var restriction);

                            // null が指定された場合は既定値にする
                            value = string.IsNullOrEmpty(value) ? ShaderItemValueState.Refer.ToString() : value;
                            restriction = string.IsNullOrEmpty(restriction) ? ShaderItemRestrictionState.None.ToString() : restriction;

                            var matchedMaterials = materials.Where(x => string.IsNullOrEmpty(materialName) || x.name == materialName).ToArray();
                            Assert.IsTrue(matchedMaterials.Any());
                            foreach (var material in matchedMaterials)
                            {
                                var materialReference = IfToolData.GetMaterialReference(material);
                                Assert.IsNotNull(materialReference);
                                var samplerBehaviorItems = materialReference.SamplerBehaviors.SamplerBehaviorItems;
                                Assert.IsNotNull(samplerBehaviorItems);
                                var matchedSamplerBehaviorItems = samplerBehaviorItems.Where(x => string.IsNullOrEmpty(sampler) || x.Id == sampler).ToArray();
                                Assert.IsTrue(matchedSamplerBehaviorItems.Any());
                                foreach (var samplerBehaviorItem in matchedSamplerBehaviorItems)
                                {
                                    var matchedBehaviorItems = samplerBehaviorItem.SamplerParamBehaviors.Where(x => string.IsNullOrEmpty(id) || x.Id == id).ToArray();
                                    Assert.IsTrue(matchedBehaviorItems.Any());
                                    foreach (var behaviorItem in matchedBehaviorItems)
                                    {
                                        Assert.IsTrue(string.Compare(value, behaviorItem.Value.ToString(), StringComparison.OrdinalIgnoreCase) == 0);
                                        Assert.IsTrue(string.Compare(restriction, behaviorItem.ChildRestriction.ToString(), StringComparison.OrdinalIgnoreCase) == 0);
                                    }
                                }
                            }
                        }

                        // --is-sampler-required, --sampler-structure-restriction の確認
                        {
                            var faceBaseMaterial = materials.FirstOrDefault(x => x.name == "FaceBase");
                            Assert.IsNotNull(faceBaseMaterial);
                            var skinBaseMaterial = materials.FirstOrDefault(x => x.name == "SkinBase");
                            Assert.IsNotNull(skinBaseMaterial);

                            var faceBaseSamplerBehaviors = IfToolData.GetMaterialReference(faceBaseMaterial)?.SamplerBehaviors;
                            Assert.IsNotNull(faceBaseSamplerBehaviors);
                            var skinBaseSamplerBehaviors = IfToolData.GetMaterialReference(skinBaseMaterial)?.SamplerBehaviors;
                            Assert.IsNotNull(skinBaseSamplerBehaviors);

                            Assert.AreEqual(faceBaseSamplerBehaviors.ChildSamplerStructureRestriction, ChildSamplerStructureRestrictionState.DisallowAddOwnSampler);
                            Assert.AreEqual(skinBaseSamplerBehaviors.ChildSamplerStructureRestriction, ChildSamplerStructureRestrictionState.None);


                            var p0BehaviorItem = faceBaseSamplerBehaviors?.SamplerBehaviorItems?.FirstOrDefault(x => x.Id == "_p0");
                            Assert.IsNotNull(p0BehaviorItem);
                            var n0BehaviorItem = faceBaseSamplerBehaviors?.SamplerBehaviorItems?.FirstOrDefault(x => x.Id == "_n0");
                            Assert.IsNotNull(n0BehaviorItem);

                            Assert.AreEqual(p0BehaviorItem.IsRequiredForChild, true);
                            Assert.AreEqual(n0BehaviorItem.IsRequiredForChild, false);
                        }
                    }
                }
                finally
                {
                    if (System.IO.File.Exists(outputFilePath))
                    {
                        System.IO.File.Delete(outputFilePath);
                    }
                }
            }
        }


        private static string GetResourcePath(out string inputFilePath, out string outputFilePath)
        {
            var assemblyFolder = System.IO.Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            Assert.IsNotNull(assemblyFolder);
            var resourceFolder = System.IO.Path.Combine(assemblyFolder, "Resources");
            inputFilePath = System.IO.Path.Combine(resourceFolder, "TownHumanBase.fmdb");
            outputFilePath = GetTempFilePath($"{System.IO.Path.GetFileNameWithoutExtension(inputFilePath)}.fmdb");
            return resourceFolder;
        }

        private static string GetXsdBasePath()
        {
            return System.IO.Path.Combine(
                System.IO.Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location),
                "../../../../../../../Programs/NintendoWare/Sources/Tools/Graphics/3dTools/3dIntermediateFileXsd");
        }

        public static string GetTempFilePath(string baseFileName)
        {
            return System.IO.Path.Combine(
                System.IO.Path.GetTempPath(),
                $"Nintendo_{Assembly.GetExecutingAssembly().GetName().Name}_{Guid.NewGuid().ToString("N")}_{baseFileName}");
        }
    }
}
