﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d.h>
#include <nw/g3d/g3d_edit.h>
#include <string>
#include <vector>
#include <list>

#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_GfxUtility.h>

#if defined(_DEBUG) || defined(NW_DEBUG)
#define G3DSANDBOX_PRINT( ... )                                                \
    nw::g3d::DebugPrint("G3D:SANDBOX: "), nw::g3d::DebugPrint(__VA_ARGS__)
#else
#define G3DSANDBOX_PRINT( ... ) ((void)0)
#endif

#define G3DSANDBOX_EXECUTE(func)\
    G3DSANDBOX_PRINT(#func "\n"); func

extern nw::g3d::math::Vec3 s_SkyColor;
extern nw::g3d::math::Vec3 s_GroundColor;
extern nw::g3d::math::Vec3 s_DiffuseColor;
extern nw::g3d::math::Vec3 s_SpecularColor;
extern nw::g3d::math::Vec3 s_FogColor;

struct CameraBlock
{
    nw::g3d::math::Mtx44 proj;
    nw::g3d::math::Mtx34 view;
};

class CameraObj
{
public:
    // perspective
    float fovy;
    float aspect;

    // ortho
    float width;
    float height;

    // projection
    float nearZ;
    float farZ;

    // view
    nw::g3d::math::Vec3 pos;
    nw::g3d::math::Vec3 target;
    nw::g3d::math::Vec3 up;

    bool isOrtho;

    void Calc()
    {
        block.view.LookAt(pos, up, target);
        if (isOrtho)
        {
            block.proj.Ortho(-width / 2, width / 2, -height / 2, height / 2, nearZ, farZ);
        }
        else
        {
            block.proj.Perspective(fovy, aspect, nearZ, farZ);
        }
    }

    const nw::g3d::math::Mtx34& GetViewMtx() const
    {
        return block.view;
    }
    const CameraBlock& GetBlock() const
    {
        return block;
    }

private:
    CameraBlock block;
};

struct EnvBlock
{
    // 半球ライト
    nw::g3d::math::Vec4 hemiDir;
    nw::g3d::math::Vec4 skyColor;
    nw::g3d::math::Vec4 groundColor;

    // ディレクショナルライト
    nw::g3d::math::Vec4 lightDir;
    nw::g3d::math::Vec4 diffuseColor;
    nw::g3d::math::Vec4 specularColor;

    // フォグ
    float fogStart;
    float fogStartEndInv;
    float padding[2];
    nw::g3d::math::Vec4 fogColor;
};

enum EnvItem
{
    ENV_HEMISPHERE_DIR,
    ENV_HEMISPHERE_SKY_COLOR,
    ENV_HEMISPHERE_GROUND_COLOR,
    ENV_LIGHT_DIR,
    ENV_DIFFUSE_COLOR,
    ENV_SPECULAR_COLOR,
    ENV_FOG_KIND,
    ENV_FOG_START_Z,
    ENV_FOG_END_Z,
    ENV_FOG_COLOR,
    ENV_WEATHER_KIND,
    NUM_ENV_ITEMS
};

extern const char* ENV_NAMES[NUM_ENV_ITEMS];

enum FogKind
{
    FOG_LINEAR,
    FOG_EXP,
    FOG_EXP2,
    NUM_FOG_KINDS
};

extern const char* FOG_NAMES[NUM_FOG_KINDS];

enum RenderInfoKind
{
    RENDER_PRIORITY,
    CLEAR_COLOR,
    NUM_RENDERINFO_KINDS
};

extern const char* RENDERINFO_NAMES[NUM_RENDERINFO_KINDS];

enum WeatherKind
{
    MOSTLY_SUNNY,
    CLOUDINESS,
    RAIN,
    NUM_WEATHER_KINDS
};

extern const char* WEATHER_NAMES[NUM_WEATHER_KINDS];

struct EnvObj
{
    nw::g3d::math::Vec3 hemiDir;
    nw::g3d::math::Vec3 skyColor;
    nw::g3d::math::Vec3 groundColor;

    nw::g3d::math::Vec3 lightDir;
    nw::g3d::math::Vec3 diffuseColor;
    nw::g3d::math::Vec3 specularColor;

    FogKind fogKind;
    float fogStartZ;
    float fogEndZ;
    nw::g3d::math::Vec3 fogColor;
    float fogDist[2];

    EnvBlock block;

    void CalcHemisphereLight()
    {
        hemiDir.Normalize(hemiDir);
        memcpy(block.hemiDir.a, hemiDir.a, sizeof(hemiDir));
        memcpy(block.skyColor.a, skyColor.a, sizeof(skyColor));
        memcpy(block.groundColor.a, groundColor.a, sizeof(groundColor));
    }

    void CalcDirectionalLight()
    {
        lightDir.Normalize(lightDir);
        memcpy(block.lightDir.a, lightDir.a, sizeof(lightDir));
        memcpy(block.diffuseColor.a, diffuseColor.a, sizeof(diffuseColor));
        memcpy(block.specularColor.a, specularColor.a, sizeof(specularColor));
    }

    void CalcFog()
    {
        if (fogStartZ == fogEndZ)
        {
            block.fogStart = 0.0f;
            block.fogStartEndInv = 0.0f;
            block.fogColor.Zero();
        }
        else
        {
            block.fogStart = fogStartZ / (fogStartZ - fogEndZ);
            block.fogStartEndInv = 1.0f / (fogEndZ - fogStartZ);
            memcpy(block.fogColor.a, fogColor.a, sizeof(fogColor));
        }
    }
};

extern CameraObj s_CameraObj;
extern EnvObj s_EnvObj;

nw::g3d::math::Vec3&
VEC3Transform(nw::g3d::math::Vec3* pDst, const nw::g3d::math::Mtx34& m);

// 描画オブジェクトとシェーダの組み合わせ
class ShaderClip
{
public:
    ShaderClip()
        : vertex(NULL)
        , fShader(NULL)
    {}

    nw::g3d::res::ResVertex* vertex;
    nw::g3d::fnd::GfxFetchShader* fShader;
};

class UniformBlockClip
{
public:
    UniformBlockClip()
        : name()
        , uBlock(NULL)
    {}

    // あえて名前引きにしておく
    std::string name;
    nw::g3d::fnd::GfxBuffer* uBlock;
};

class SamplerClip
{
public:
    SamplerClip()
        : shaderSamplerIdx(-1)
        , sampler(NULL)
    {}

    int shaderSamplerIdx;
    nw::g3d::res::ResSampler* sampler;
};

class ShaderAssign
{
    public:
        ShaderAssign()
            : shadingModel(NULL)
            , program(NULL)
        {}

        nw::g3d::res::ResShadingModel* shadingModel;
        nw::g3d::res::ResShaderProgram* program;
};

class RenderObj
{
public:
    RenderObj()
        : shape(NULL)
        , mat(NULL)
        , shadingModel(NULL)
        , program(NULL)
        , shaderAssign(NULL)
        , clip(NULL)
        , samplerTable()
        , renderPriority(0)
    {}

    nw::g3d::ShapeObj* shape;
    nw::g3d::MaterialObj* mat;
    ShaderAssign* shaderAssign;
    nw::g3d::res::ResShadingModel* shadingModel;
    nw::g3d::res::ResShaderProgram* program;
    ShaderClip* clip;

    int renderPriority;

    std::vector<SamplerClip*> samplerTable;
};

class RenderModel
{
public:
    RenderModel()
        : mdl(NULL)
        , objArray()
    {}

    nw::g3d::ModelObj* mdl;
    std::vector<RenderObj*> objArray;
};

// アニメーションとモデルの組み合わせを扱うインスタンス
class AnimModelObj
{
public:
    AnimModelObj()
        : scale(nw::g3d::math::Vec3::Make(1.0f, 1.0f, 1.0f))
        , rotate(nw::g3d::math::Vec3::Make(0.0f, 0.0f, 0.0f))
        , translate(nw::g3d::math::Vec3::Make(0.0f, 0.0f, 0.0f))
        , constraintModel(NULL)
        , constraintBoneIndex(-1)
        , mdl(NULL)
        , sklAnim(NULL)
        , boneVisAnim(NULL)
        , texAnim(NULL)
        , shapeAnim(NULL)
        , paramAnim(NULL)
    {
    }

    nw::g3d::math::Vec3 scale;
    nw::g3d::math::Vec3 rotate;
    nw::g3d::math::Vec3 translate;

    nw::g3d::ModelObj* constraintModel;
    int constraintBoneIndex;

    nw::g3d::ModelObj* mdl;
    nw::g3d::SkeletalAnimObj* sklAnim;
    nw::g3d::VisibilityAnimObj* boneVisAnim;
    nw::g3d::TexPatternAnimObj* texAnim;
    nw::g3d::ShapeAnimObj* shapeAnim;
    nw::g3d::ShaderParamAnimObj* paramAnim;
};

class UBOFinder
{
public:
    explicit UBOFinder(std::string name)
        : mName(name)
    {}

    bool operator()(UniformBlockClip& clip)
    {
        return clip.name == mName;
    }

    std::string mName;
};

class ShaderArchiveFinder
{
public:
    explicit ShaderArchiveFinder(const char* archiveName)
        : mArchiveName(archiveName)
    {}

    bool operator()(nw::g3d::res::ResShaderArchive*& rArchive)
    {
        if (mArchiveName != NULL && strcmp(rArchive->GetName(), mArchiveName) == 0)
        {
            return true;
        }
        return false;
    }

    const char* mArchiveName;
};

template<typename Model>
class ModelFinder
{
public:
    explicit ModelFinder(const char* modelName)
        : mModelName(modelName)
        , mModelObj(NULL)
    {}

    explicit ModelFinder(const nw::g3d::ModelObj* modelObj)
        : mModelName(NULL)
        , mModelObj(modelObj)
    {}

    bool operator()(const Model model)
    {
        if (mModelObj)
        {
            if (model->mdl == mModelObj)
            {
                return true;
            }
        }
        else
        {
            if (mModelName != NULL && strcmp(model->mdl->GetResource()->GetName(), mModelName) == 0)
            {
                return true;
            }
        }
        return false;
    }

    const char* mModelName;
    const nw::g3d::ModelObj* mModelObj;
};

class GreaterPriority
{
public:

    bool operator()(const RenderObj* lhs, const RenderObj* rhs) const
    {
        return lhs->renderPriority > rhs->renderPriority;
    }
};

extern void BuildModel(nw::g3d::res::ResModel* rMdl,
                       std::vector<AnimModelObj*>& models);

extern void BuildSklAnim(nw::g3d::res::ResSkeletalAnim* rSklAnim,
                         int maxSklCount,
                         std::vector<nw::g3d::SkeletalAnimObj*>& sklAnims);

extern void BuildBoneVisAnim(nw::g3d::res::ResVisibilityAnim* rVisAnim,
                             int maxSklCount,
                             std::vector<nw::g3d::VisibilityAnimObj*>& boneVisAnims);

extern void BuildSceneAnim(nw::g3d::res::ResSceneAnim* rScnAnim,
                           std::vector<nw::g3d::CameraAnimObj*>& cameraAnims,
                           std::vector<nw::g3d::LightAnimObj*>& lightAnims,
                           std::vector<nw::g3d::FogAnimObj*>& fogAnims);

extern void BuildTexPatternAnim(nw::g3d::res::ResTexPatternAnim* rTexAnim,
                                int maxMatCount,
                                std::vector<nw::g3d::TexPatternAnimObj*>& texAnims);

extern void BuildShapeAnim(nw::g3d::res::ResShapeAnim* rShapeAnim,
                           int maxShapeCount,
                           std::vector<nw::g3d::ShapeAnimObj*>& shapeAnims);

extern void BuildParamAnim(nw::g3d::res::ResShaderParamAnim* rParamAnim,
                           int maxMatCount,
                           std::vector<nw::g3d::ShaderParamAnimObj*>& paramAnims);

template<typename AnimType>
inline void DeleteAnim(AnimType* anim)
{
    if (anim == NULL)
    {
        return;
    }

    void* buffer = anim->GetBufferPtr();
    if (buffer != NULL)
    {
        nw::g3d::demo::FreeMem2(buffer);
    }

    nw::g3d::demo::FreeMem2(anim);
}

extern void DeleteModel(AnimModelObj* animModel);

extern void SetupFile(const char* filename, std::vector<nw::g3d::res::ResFile*>& files);

extern void CleanupFile(nw::g3d::res::ResFile* res);

extern void SetupShaderArchive(const char* filename, std::vector<nw::g3d::res::ResShaderArchive*>& shaderArchives);

extern void CleanupShaderArchive(nw::g3d::res::ResShaderArchive* res);

extern void BindSklAnim(AnimModelObj* animModel, std::vector<nw::g3d::SkeletalAnimObj*>& sklAnims);

extern void BindBoneVisAnim(AnimModelObj* animModel, std::vector<nw::g3d::VisibilityAnimObj*>& boneVisAnims);

extern void BindTexAnim(AnimModelObj* animModel, std::vector<nw::g3d::TexPatternAnimObj*>& texAnims);

extern void BindShapeAnim(AnimModelObj* animModel, std::vector<nw::g3d::ShapeAnimObj*>& shapeAnims);

extern void BindParamAnim(AnimModelObj* animModel, std::vector<nw::g3d::ShaderParamAnimObj*>& paramAnims);

extern void DeleteRenderModel(RenderModel* renderModel);

extern void DeleteUniformBlockClip(UniformBlockClip& clip);



extern void SetupShaderAssign(nw::g3d::ResModel* pResModel, nw::g3d::ResShaderArchive** pShaderArchiveArray, std::vector<nw::g3d::ResShaderArchive*>& shaderArchives);

extern void BindShader(RenderModel* renderModel, bool initBlock);

extern void SetupShaderBinding(RenderModel* renderModel);
