﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "SandboxNwG3dEditor.h"
#include "SandboxUtility.h"

#include <nw/g3d/edit/g3d_EditDefs.h>
#include <nw/g3d/g3d_res.h>

#include <vector>

using namespace G3dSandbox;

#if NW_G3D_CONFIG_USE_HOSTIO
SandboxNwG3dEditor* SandboxNwG3dEditor::s_Instance = NULL;

void SandboxNwG3dEditor::CreateInstance(
    std::vector<nw::g3d::ResFile*>* files,
    std::vector<nw::g3d::ResShaderArchive*>* shaderArchives,
    std::vector<AnimModelObj*>* models,
    std::vector<RenderModel*>* renderModels,
    float clearColorR, float clearColorG, float clearColorB)
{
    if (!SandboxNwG3dEditor::s_Instance)
    {
        SandboxNwG3dEditor::s_Instance =
            new (nw::g3d::demo::AllocMem2(sizeof(SandboxNwG3dEditor))) SandboxNwG3dEditor();

        Instance()->m_Files = files;
        Instance()->m_ShaderArchives = shaderArchives;
        Instance()->m_Models = models;
        Instance()->m_RenderModels = renderModels;
        Instance()->m_ClearColor[0] = clearColorR;
        Instance()->m_ClearColor[1] = clearColorG;
        Instance()->m_ClearColor[2] = clearColorB;
    }
}

void SandboxNwG3dEditor::DeleteInstance()
{
    if(SandboxNwG3dEditor::s_Instance)
    {
        nw::g3d::demo::FreeMem2(SandboxNwG3dEditor::s_Instance);
        SandboxNwG3dEditor::s_Instance = NULL;
    }
}

SandboxNwG3dEditor* SandboxNwG3dEditor::Instance()
{
    return s_Instance;
}

void SandboxNwG3dEditor::CalcEditor()
{
    nw::g3d::edit::EditServer::Instance().CalcAnimations();
}

// GPU が直接リードするメモリを書き換えます。
// CalcBlock などの GPU がメモリリードしないタイミングで呼び出してください。
void SandboxNwG3dEditor::CalcBlock()
{
    nw::g3d::edit::EditServer::Instance().Poll();
#if 0// edit の動作確認
    {
        float frame = nw::g3d::edit::EditServer::Instance().GetFrame();
        G3DSANDBOX_PRINT("--- GlobalFrame %f --- \n", frame);
    }
#if 0
    int modelAnimCount = nw::g3d::edit::EditServer::Instance().GetModelAnimCount();
    for (int i = 0; i < modelAnimCount; ++i)
    {
        float frame = nw::g3d::edit::EditServer::Instance().GetModelAnimFrameCount(i);
        const char* name = nw::g3d::edit::EditServer::Instance().GetModelAnimName(i);
        G3DSANDBOX_PRINT("%d : Frame[%f] > %s\n", i, frame, name);
    }
#endif
#endif
}

void SandboxNwG3dEditor::EditShadingModel(const nw::g3d::edit::EditShadingModelArg& arg)
{
    (void)arg;
    G3DSANDBOX_PRINT("EditShadingModel");
}

nw::g3d::res::ResFile* SandboxNwG3dEditor::LoadFile(const nw::g3d::edit::LoadFileArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    void* fFile = nw::g3d::demo::AllocMem2(arg.fileSize, arg.align);
    memcpy(fFile, arg.resFile, arg.fileSize);

    NW_G3D_ASSERT(nw::g3d::res::ResFile::IsValid(fFile));
    nw::g3d::res::ResFile* rFile =
        nw::g3d::res::ResFile::ResCast(fFile);

    (*m_Files).push_back(rFile);

    nw::g3d::res::BindResult result =
        rFile->Bind(rFile, nw::g3d::res::BIND_TEXTURE);

    NW_G3D_ASSERT(result.IsComplete());

    rFile->Setup();

    // モデルインスタンスの構築
    for (int i = 0; i < rFile->GetModelCount(); ++i)
    {
        nw::g3d::ResModel* rModel = rFile->GetModel(i);
        NW_G3D_ASSERT_NOT_NULL(rModel);
        BuildModel(rModel, (*m_Models));

        RenderModel* renderModel = new (nw::g3d::demo::AllocMem2(sizeof(RenderModel))) RenderModel();
        renderModel->mdl = (*m_Models).back()->mdl;
        (*m_RenderModels).push_back(renderModel);

        SetupShaderAssign(renderModel->mdl->GetResource(), NULL, *m_ShaderArchives);
        BindShader(renderModel, true);
        SetupShaderBinding(renderModel);

        nw::g3d::edit::EditServer::AttachModelArg attachArg;
        attachArg.modelObj = renderModel->mdl;
        nw::g3d::edit::EditServer::Instance().AttachModel(attachArg);
    }

    return rFile;
}

void SandboxNwG3dEditor::UnloadFile(const nw::g3d::edit::UnloadFileArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    nw::g3d::ResFile* rFile = arg.resFile;

    if (rFile->GetModelCount())
    {
        nw::g3d::ResModel* rModel = rFile->GetModel(0);

        for (std::vector<RenderModel*>::iterator iter = (*m_RenderModels).begin();
            iter != (*m_RenderModels).end();
            ++iter)
        {
            if ((*iter)->mdl->GetResource() == rModel)
            {
                RenderModel* renderModel = *iter;
                DeleteRenderModel(renderModel);
                (*m_RenderModels).erase(iter);
                break;
            }
        }

        for (std::vector<AnimModelObj*>::iterator iter = (*m_Models).begin();
            iter != (*m_Models).end();
            ++iter)
        {
            if ((*iter)->mdl->GetResource() == rModel)
            {
                AnimModelObj* animModel = (*iter);
                (*m_Models).erase(iter);
                DeleteModel(animModel);
                break;
            }
        }

        for (std::vector<nw::g3d::ResFile*>::iterator iter = (*m_Files).begin();
            iter != (*m_Files).end();
            ++iter)
        {
            if (*iter == rFile)
            {
                (*m_Files).erase(iter);
                break;
            }
        }

        rFile->Cleanup();
        nw::g3d::demo::FreeMem2(rFile);
    }
}

void SandboxNwG3dEditor::UpdateShaderAssign(const nw::g3d::edit::UpdateShaderAssignArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    SetupShaderAssign(arg.newResModel, arg.shaderArchivePtrs, *m_ShaderArchives);

    for (int i = 0; i < arg.newResModel->GetShapeCount(); ++i)
    {
        nw::g3d::ResShape* rShape = arg.newResModel->GetShape(i);
        nw::g3d::ResMaterial* rMat = arg.newResModel->GetMaterial(rShape->GetMaterialIndex());

        ShaderAssign* pShaderAssign = rShape->GetUserPtr<ShaderAssign>();
        NW_G3D_ASSERT_NOT_NULL(pShaderAssign);
        nw::g3d::ResShadingModel* rShadingModel = pShaderAssign->shadingModel;

        // UniformBlock のセットアップ
        int blockIndex = rShadingModel->GetSystemBlockIndex(nw::g3d::ResUniformBlockVar::TYPE_MATERIAL);
        if (blockIndex != -1)
        {
            nw::g3d::ResUniformBlockVar* rBlock = rShadingModel->GetUniformBlock(blockIndex);
            rMat->SetRawParamSize(rBlock->GetSize());

            // shader param
            for (int paramIndex = 0; paramIndex < rMat->GetShaderParamCount(); ++paramIndex)
            {
                nw::g3d::ResShaderParam* rParam = rMat->GetShaderParam(paramIndex);
                nw::g3d::ResUniformVar* rUniform = rBlock->GetUniform(rParam->GetId());

                NW_G3D_ASSERT_NOT_NULL(rUniform);

                rParam->SetOffset(rUniform->GetOffset());
            }
        }
    }
}

void SandboxNwG3dEditor::UpdateMaterials(const nw::g3d::edit::UpdateMaterialsArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);
    PrintUpdateMaterialState(arg.state);

    // 古い RenderModel を削除する。
    for (std::vector<RenderModel*>::iterator iter = (*m_RenderModels).begin();
        iter != (*m_RenderModels).end();
        ++iter)
    {
        if ((*iter)->mdl == arg.modelObj)
        {
            RenderModel* renderModel = *iter;
            DeleteRenderModel(renderModel);
            (*m_RenderModels).erase(iter);
            break;
        }
    }

    // 新しい RenderModel を構築する。
    {
        RenderModel* renderModel = new (nw::g3d::demo::AllocMem2(sizeof(RenderModel))) RenderModel();
        renderModel->mdl = arg.modelObj;
        (*m_RenderModels).push_back(renderModel);

        BindShader(renderModel, false);
        SetupShaderBinding(renderModel);
    }
}

void SandboxNwG3dEditor::UpdateRenderInfo(const nw::g3d::edit::RenderInfoArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    const nw::g3d::ModelObj* mdl = arg.modelObj;
    std::vector<RenderModel*>::iterator comp =
        std::find_if((*m_RenderModels).begin(), (*m_RenderModels).end(), ModelFinder<RenderModel*>(mdl));
#if 0
    {
        const nw::g3d::ModelObj* pModel = arg.modelObj;
        const nw::g3d::MaterialObj* pMaterial = pModel->GetMaterial(arg.materialIndex);
        const nw::g3d::ResMaterial* pResMaterial = pMaterial->GetResource();
        const nw::g3d::ResRenderInfo* pResRenderInfo = pResMaterial->GetRenderInfo(arg.name);

        switch(pResRenderInfo->GetType())
        {
        case nw::g3d::ResRenderInfo::INT:
            {
                const int* values = pResRenderInfo->GetInt();
                for (int idxRenderInfo = 0, numRenderInfo = pResRenderInfo->ref().arrayLength;
                    idxRenderInfo < numRenderInfo; ++idxRenderInfo)
                {
                    int value = values[idxRenderInfo];
                    G3DSANDBOX_PRINT("value : %d\n", value);
                }
            }
            break;
        case nw::g3d::ResRenderInfo::FLOAT:
            {
                const float* values = pResRenderInfo->GetFloat();
                for (int idxRenderInfo = 0, numRenderInfo = pResRenderInfo->ref().arrayLength;
                    idxRenderInfo < numRenderInfo; ++idxRenderInfo)
                {
                    float value = values[idxRenderInfo];
                    G3DSANDBOX_PRINT("value : %f\n", value);
                }
            }
            break;
        case nw::g3d::ResRenderInfo::STRING:
            {
                for (int idxRenderInfo = 0, numRenderInfo = pResRenderInfo->ref().arrayLength;
                    idxRenderInfo < numRenderInfo; ++idxRenderInfo)
                {
                    const char* value = pResRenderInfo->GetString(idxRenderInfo);
                    G3DSANDBOX_PRINT("value : %s\n", value);
                }
            }
            break;
        default:
            break;
        }
    }
#endif
    if (comp != (*m_RenderModels).end())
    {
        const nw::g3d::MaterialObj* mat = mdl->GetMaterial(arg.materialIndex);

        const nw::g3d::res::ResRenderInfo* resRenderInfo = mat->GetResource()->GetRenderInfo(arg.name);
        if (resRenderInfo != NULL)
        {
            if (strcmp(arg.name, RENDERINFO_NAMES[RENDER_PRIORITY]) == 0)
            {
                // 描画優先度の設定
                NW_G3D_ASSERT(resRenderInfo->GetType() == nw::g3d::res::ResRenderInfo::INT);
                NW_G3D_ASSERT(resRenderInfo->GetArrayLength() > 0);

                for (int i = 0; i < static_cast<int>((*comp)->objArray.size()); ++i)
                {
                    RenderObj* obj = (*comp)->objArray[i];
                    if (obj->mat == mat)
                    {
                        obj->renderPriority = resRenderInfo->GetInt()[0];
                    }
                }
            }
            else if (strcmp(arg.name, RENDERINFO_NAMES[CLEAR_COLOR]) == 0)
            {
                // クリアカラー（float のテスト用）
                NW_G3D_ASSERT(resRenderInfo->GetType() == nw::g3d::res::ResRenderInfo::FLOAT);
                NW_G3D_ASSERT(resRenderInfo->GetArrayLength() >= 0 && resRenderInfo->GetArrayLength() <= 3);

                for (int i = 0; i < resRenderInfo->GetArrayLength(); ++i)
                {
                    m_ClearColor[i] = resRenderInfo->GetFloat()[i];
                }
            }

#if 0 // デバッグ用途
            {
                G3DSANDBOX_PRINT("UpdateRenderInfo() called.\n");

                switch(resRenderInfo->GetType())
                {
                case nw::g3d::res::ResRenderInfo::STRING:
                    {
                        G3DSANDBOX_PRINT("-> Name:%s (Type:STRING)\n", resRenderInfo->GetName());
                        for (int i = 0; i < resRenderInfo->GetArrayLength(); ++i)
                        {
                            G3DSANDBOX_PRINT("   %s [%d]\n", resRenderInfo->GetString(i), i );
                        }
                    }
                    break;
                case nw::g3d::res::ResRenderInfo::INT:
                    {
                        G3DSANDBOX_PRINT("-> Name:%s (Type:INT)\n", resRenderInfo->GetName());
                        for (int i = 0; i < resRenderInfo->GetArrayLength(); ++i)
                        {
                            G3DSANDBOX_PRINT("   %d [%d]\n", resRenderInfo->GetInt()[i], i );
                        }
                    }
                    break;
                case nw::g3d::res::ResRenderInfo::FLOAT:
                    {
                        G3DSANDBOX_PRINT("-> Name:%s (Type:FLOAT)\n", resRenderInfo->GetName());
                        for (int i = 0; i < resRenderInfo->GetArrayLength(); ++i)
                        {
                            G3DSANDBOX_PRINT("   %f [%d]\n", resRenderInfo->GetFloat()[i], i );
                        }
                    }
                    break;
                default:
                    break;
                }

                G3DSANDBOX_PRINT("\n");
            }
#endif
        }
    }
} // NOLINT (readability/fn_size)

void SandboxNwG3dEditor::UpdateRenderState(nw::g3d::ModelObj* modelObj)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);
    nw::g3d::ResModel* pResModel = modelObj->GetResource();
    G3DSANDBOX_PRINT("Model Name: %s\n", pResModel->GetName());

    int material_count = pResModel->GetMaterialCount();
    for (int i = 0; i < material_count; ++i)
    {
        nw::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(i);
        G3DSANDBOX_PRINT("Material Name: %s\n", pResMaterial->GetName());
        nw::g3d::ResRenderState* pResRenderState = pResMaterial->GetRenderState();
        PrintRenderState(pResRenderState);
    }
}

void SandboxNwG3dEditor::SendRenderInfo(const nw::g3d::edit::SendRenderInfoArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

#if 0 // デバッグ用途
    for (int i = 0; i < arg.numSetupRenderInfo; ++i)
    {
        const nw::g3d::edit::SetupRenderInfo* renderInfo = arg.setupRenderInfoPtrs[i];
        G3DSANDBOX_PRINT("%d : Name > %s\n", i, renderInfo->GetName());

        const nw::g3d::edit::SetupRenderInfoChoice* choice = renderInfo->GetChoice();
        if (renderInfo->GetType() == nw::g3d::res::ResRenderInfo::STRING)
        {
            for (int j = 0; j < choice->GetChoiceCount(); ++j)
            {
                G3DSANDBOX_PRINT("\t%d : Choice > %s\n", j, choice->GetString(j));
            }
        }
        else if (renderInfo->GetType() == nw::g3d::res::ResRenderInfo::INT)
        {
            for (int j = 0; j < choice->GetChoiceCount(); ++j)
            {
                G3DSANDBOX_PRINT("\t%d : Choice > %d\n", j, choice->GetInt()[j]);
            }
        }
        else if (renderInfo->GetType() == nw::g3d::res::ResRenderInfo::FLOAT)
        {
            for (int j = 0; j < choice->GetChoiceCount(); ++j)
            {
                G3DSANDBOX_PRINT("\t%d : Choice > %f\n", j, choice->GetFloat()[j]);
            }
        }

        const nw::g3d::edit::SetupRenderInfoDefault* defaultValue = renderInfo->GetDefault();
        if (renderInfo->GetType() == nw::g3d::res::ResRenderInfo::STRING)
        {
            for (int j = 0; j < defaultValue->GetDefaultCount(); ++j)
            {
                G3DSANDBOX_PRINT("\t%d : Default > %s\n", j, defaultValue->GetString(j));
            }
        }
        else if (renderInfo->GetType() == nw::g3d::res::ResRenderInfo::INT)
        {
            for (int j = 0; j < defaultValue->GetDefaultCount(); ++j)
            {
                G3DSANDBOX_PRINT("\t%d : Default > %d\n", j, defaultValue->GetInt()[j]);
            }
        }
        else if (renderInfo->GetType() == nw::g3d::res::ResRenderInfo::FLOAT)
        {
            for (int j = 0; j < defaultValue->GetDefaultCount(); ++j)
            {
                G3DSANDBOX_PRINT("\t%d : Default > %f\n", j, defaultValue->GetFloat()[j]);
            }
        }
    }
#endif
#if 0//簡易検証用
    // 0 から 255 段階の描画優先度を設定する。choice 範囲をテストなのでありえない値にしておく。
    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[RENDER_PRIORITY], 32, 192);

    // 動作検証用に三個デフォルトを送る。
    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 128);
    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 64);
    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 32);
#endif
    if (strcmp(arg.modelObj->GetResource()->GetName(), "EnvModel") == 0)
    {
#if 1
        const nw::g3d::MaterialObj* materialObj = arg.modelObj->GetMaterial(arg.materialIndex);
        NW_G3D_ASSERT_NOT_NULL(materialObj);

        // ランタイムから choice、default を送信するテスト。
        const nw::g3d::res::ResRenderInfo* renderInfoWeather = materialObj->GetResource()->GetRenderInfo(ENV_NAMES[ENV_WEATHER_KIND]);
        if (renderInfoWeather != NULL)
        {
            NW_G3D_ASSERT(renderInfoWeather->GetType() == nw::g3d::res::ResRenderInfo::STRING);

            {
                nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_WEATHER_KIND], WEATHER_NAMES[MOSTLY_SUNNY], "快晴");
                nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_WEATHER_KIND], WEATHER_NAMES[CLOUDINESS], "曇り");
                nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_WEATHER_KIND], WEATHER_NAMES[RAIN], "雨");
                nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_WEATHER_KIND], WEATHER_NAMES[MOSTLY_SUNNY]);
            }

            if (renderInfoWeather->GetArrayLength() > 0)
            {
                if (strcmp(renderInfoWeather->GetString(0), WEATHER_NAMES[CLOUDINESS]) == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP], "指数フォグ");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP2], "指数フォグ２");

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP2]);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR]);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR]);

                    // int型のRenderInfo

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[RENDER_PRIORITY], 32, 192);

                    // 動作検証用に三個デフォルトを送る。
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 128);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 64);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 32);

                    // float型のRenderInfo

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[CLEAR_COLOR], 0.3f, 128.f);

                    // 動作検証用に三個デフォルトを送る。
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 1.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 0.7f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 0.5f);
                }
                else if (strcmp(renderInfoWeather->GetString(0), WEATHER_NAMES[RAIN]) == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP2]);

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP2]);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR]);

                    // int型のRenderInfo

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[RENDER_PRIORITY], 16, 96);

                    // 動作検証用に二個デフォルトを送る。
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 64);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 32);

                    // float型のRenderInfo

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[CLEAR_COLOR], 0.1f, 32.f);

                    // 動作検証用に二個デフォルトを送る。
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 0.7f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 0.5f);
                }
                else
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR], "線形フォグ");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP]);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP2], "指数フォグ２");

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR]);

                    // int型のRenderInfo

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[RENDER_PRIORITY], 0, 128);

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 64);

                    // float型のRenderInfo

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[CLEAR_COLOR], 0.5f, 64.f);

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 32.f);
                }
            }
            else
            {
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR], "線形フォグ");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP]);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP2], "指数フォグ２");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR]);

                    // int型のRenderInfo

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[RENDER_PRIORITY], 32, 192);

                    // 動作検証用に三個デフォルトを送る。
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 128);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 64);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[RENDER_PRIORITY], 32);

                    // float型のRenderInfo

                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[CLEAR_COLOR], 0.3f, 128.f);

                    // 動作検証用に三個デフォルトを送る。
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 1.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 0.7f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 0.5f);
                }
            }
        }

#else //簡易検証用
        nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR], "線形フォグ");
        nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP]);
        nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP2], "指数フォグ２");

        // 動作検証用に三個デフォルトを送る。
        nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_EXP2]);
        nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR]);
        nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(ENV_NAMES[ENV_FOG_KIND], FOG_NAMES[FOG_LINEAR]);

        // クリアカラー（float のテスト用）choice 範囲をテストなのでありえない値にしておく。
        nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice(RENDERINFO_NAMES[CLEAR_COLOR], 0.3f, 128.f);
        nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 1.f);
        nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 0.7f);
        nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault(RENDERINFO_NAMES[CLEAR_COLOR], 0.5f);
#endif
    }

    {
        // シェーダアノテーションで choice を記述して、その選択に基づいてランタイムから choice、default を送信するテスト。
        const nw::g3d::MaterialObj* materialObj = arg.modelObj->GetMaterial(arg.materialIndex);
        NW_G3D_ASSERT_NOT_NULL(materialObj);

        const nw::g3d::res::ResRenderInfo* renderInfoFsd = materialObj->GetResource()->GetRenderInfo("string_fsd");
        if (renderInfoFsd != NULL)
        {
            NW_G3D_ASSERT(renderInfoFsd->GetType() == nw::g3d::res::ResRenderInfo::STRING);

            if (renderInfoFsd->GetArrayLength() >= 1)
            {
                if (strcmp(renderInfoFsd->GetString(0), "0") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("int_edit", 1, 10);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 1);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 5);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 10);
                }
                else if (strcmp(renderInfoFsd->GetString(0), "1") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("int_edit", 11, 99);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 11);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 55);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 99);
                }
                else if (strcmp(renderInfoFsd->GetString(0), "2") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("int_edit", 100, 255);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 100);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 123);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("int_edit", 255);
                }
            }
            if (renderInfoFsd->GetArrayLength() >= 2)
            {
                if (strcmp(renderInfoFsd->GetString(1), "0") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("float_edit", 1.f, 10.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 1.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 5.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 10.f);
                }
                else if (strcmp(renderInfoFsd->GetString(1), "1") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("float_edit", 11.f, 99.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 11.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 55.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 99.f);
                }
                else if (strcmp(renderInfoFsd->GetString(1), "2") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("float_edit", 100.f, 255.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 100.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 123.f);
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("float_edit", 255.f);
                }
            }
        }

        const nw::g3d::res::ResRenderInfo* renderInfoMulti = materialObj->GetResource()->GetRenderInfo("multi_category");
        if (renderInfoMulti != NULL)
        {
            NW_G3D_ASSERT(renderInfoMulti->GetType() == nw::g3d::res::ResRenderInfo::STRING);

            if (renderInfoMulti->GetArrayLength() >= 1)
            {
                if (strcmp(renderInfoMulti->GetString(0), "0") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "chara_player",  "プレーヤー");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "chara_npc",  "NPC");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("multi_list", "chara_player");
                }
                else if (strcmp(renderInfoMulti->GetString(0), "1") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "obj_ground",  "地面");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "obj_obstacle",  "障害物");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "obj_sphere",  "天球");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("multi_list", "obj_obstacle");
                }
                else if (strcmp(renderInfoMulti->GetString(0), "2") == 0)
                {
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "other_effect",  "エフェクト");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "other_hemisphere",  ":ref{group/group_hemisphere}:color{white}:background-color{black}");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "other_directional_light",  ":ref{group/group_directional_light}:color{red}:background-color{black}");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoChoice("multi_list", "other_group_fog",  ":ref{group/group_fog}:color{yellow}:background-color{black}");
                    nw::g3d::edit::EditServer::Instance().PushRenderInfoDefault("multi_list", "other_hemisphere");
                }
            }
        }
    }
} // NOLINT (readability/fn_size)

void SandboxNwG3dEditor::UpdateBoneBind(const nw::g3d::edit::UpdateBoneBindArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    std::vector<AnimModelObj*>::iterator comp =
        std::find_if((*m_Models).begin(), (*m_Models).end(), ModelFinder<AnimModelObj*>(arg.modelObj));

    if (comp != (*m_Models).end())
    {
        (*comp)->constraintModel = arg.parentModelObj;
        (*comp)->constraintBoneIndex = arg.parentBoneIndex;
    }
}

bool SandboxNwG3dEditor::SendModelLayout(nw::g3d::edit::SendModelLayoutData* outputData, const nw::g3d::edit::SendModelLayoutArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    std::vector<AnimModelObj*>::iterator comp =
        std::find_if((*m_Models).begin(), (*m_Models).end(), ModelFinder<AnimModelObj*>(arg.modelObj));

    if (comp != (*m_Models).end())
    {
        memcpy(outputData->scale.a, (*comp)->scale.a, sizeof(f32) * 3);
        memcpy(outputData->rotate.a, (*comp)->rotate.a, sizeof(f32) * 3);
        memcpy(outputData->translate.a, (*comp)->translate.a, sizeof(f32) * 3);

        return true;
    }

    return false;
}

void SandboxNwG3dEditor::UpdateModelLayout(const nw::g3d::edit::UpdateModelLayoutArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    const nw::g3d::ModelObj* modelObj = arg.modelObj;
    std::vector<AnimModelObj*>::iterator comp =
        std::find_if((*m_Models).begin(), (*m_Models).end(), ModelFinder<AnimModelObj*>(modelObj));

    if (comp != (*m_Models).end())
    {
        memcpy((*comp)->scale.a, arg.scale.a, sizeof(f32) * 3);
        memcpy((*comp)->rotate.a, arg.rotate.a, sizeof(f32) * 3);
        memcpy((*comp)->translate.a, arg.translate.a, sizeof(f32) * 3);
    }
}

void SandboxNwG3dEditor::BindSceneAnim(const nw::g3d::edit::BindSceneAnimArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);
    (void)arg;
}

void SandboxNwG3dEditor::UnbindSceneAnim(const nw::g3d::edit::UnbindSceneAnimArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);
    (void)arg;
}

void SandboxNwG3dEditor::ApplySceneAnim(const nw::g3d::edit::ApplySceneAnimArg& arg)
{
    // カメラアニメが一つ以上あれば、最初のものを適当に使う
    if (arg.numCameraAnimObj > 0)
    {
        nw::g3d::CameraAnimObj* camera = arg.cameraAnimObjPtrs[0];
        nw::g3d::CameraAnimResult* result = camera->GetResult();
        s_CameraObj.pos.Set(result->view.pos[0], result->view.pos[1], result->view.pos[2]);
        s_CameraObj.up.Set(0.0f, 1.0f, 0.0f);
        s_CameraObj.target.Set(result->view.aim[0], result->view.aim[1], result->view.aim[2]);
        s_CameraObj.Calc();
    }
}

void SandboxNwG3dEditor::SelectTarget(const nw::g3d::edit::SelectTargetArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    switch (arg.targetKind)
    {
    case nw::g3d::edit::SelectTargetArg::TARGET_MODEL:
        G3DSANDBOX_PRINT("MODEL Selected. (%s)\n", arg.modelObj->GetResource()->GetName());
        break;
    case nw::g3d::edit::SelectTargetArg::TARGET_MATERIAL:
        G3DSANDBOX_PRINT("MATERIAL Selected. (%s)\n", arg.modelObj->GetResource()->GetName());
        break;
    case nw::g3d::edit::SelectTargetArg::TARGET_BONE:
        G3DSANDBOX_PRINT("BONE Selected. (%s)\n", arg.modelObj->GetResource()->GetName());
        break;
    case nw::g3d::edit::SelectTargetArg::TARGET_SHAPE:
        G3DSANDBOX_PRINT("SHAPE Selected. (%s)\n", arg.modelObj->GetResource()->GetName());
        break;
    default:
        break;
    }

    for (u32 i = 0; i < arg.indexSize; ++i)
    {
        G3DSANDBOX_PRINT(" => Selected index: %d\n", arg.index[i]);
    }
}

void SandboxNwG3dEditor::AttachModel(const nw::g3d::edit::AttachModelArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);
    G3DSANDBOX_PRINT("%s Attached\n", arg.modelObj->GetResource()->GetName());
}

void SandboxNwG3dEditor::DetachModel(const nw::g3d::edit::DetachModelArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);
    G3DSANDBOX_PRINT("%s Detached\n", arg.modelObj->GetResource()->GetName());
}

void SandboxNwG3dEditor::ExecuteModelUserScript(const nw::g3d::edit::ExecuteModelUserScriptArg& arg)
{
    G3DSANDBOX_PRINT("%s\n", __FUNCTION__);

    G3DSANDBOX_PRINT("ScriptText: \n%s\n", arg.scriptText);

    nw::g3d::ResModel* pResModel = arg.targetModelObj->GetResource();
    G3DSANDBOX_PRINT("Target Model Name: %s\n", pResModel->GetName());

    G3DSANDBOX_PRINT("selectedBoneCount = %d\n", arg.selectedBoneCount);
    for (int i = 0; i < arg.selectedBoneCount; ++i)
    {
        int boneIndex = arg.selectedBoneIndices[i];
        G3DSANDBOX_PRINT("selectedBoneIndices[%d] = %d\n", i, boneIndex);
    }

    G3DSANDBOX_PRINT("selectedShapeCount = %d\n", arg.selectedShapeCount);
    for (int i = 0; i < arg.selectedShapeCount; ++i)
    {
        int shapeIndex = arg.selectedShapeIndices[i];
        G3DSANDBOX_PRINT("selectedShapeIndices[%d] = %d\n", i, shapeIndex);
    }

    G3DSANDBOX_PRINT("selectedMaterialCount = %d\n", arg.selectedMaterialCount);
    for (int i = 0; i < arg.selectedMaterialCount; ++i)
    {
        int materialIndex = arg.selectedMaterialIndices[i];
        G3DSANDBOX_PRINT("selectedMaterialIndices[%d] = %d\n", i, materialIndex);
    }
}

void SandboxNwG3dEditor::PrintUpdateMaterialState(nw::g3d::edit::UpdateMaterialsArg::State state)
{
    switch (state)
    {
    case nw::g3d::edit::UpdateMaterialsArg::STATE_UPDATE:
        G3DSANDBOX_PRINT("state = UPDATE\n");
        break;
    case nw::g3d::edit::UpdateMaterialsArg::STATE_ERROR:
        G3DSANDBOX_PRINT("state = ERROR\n");
        break;
    case nw::g3d::edit::UpdateMaterialsArg::STATE_END:
        G3DSANDBOX_PRINT("state = END\n");
        break;
    case nw::g3d::edit::UpdateMaterialsArg::STATE_BEGIN:
        G3DSANDBOX_PRINT("state = BEGIN\n");
        break;
    default:
        break;
    }
}

void SandboxNwG3dEditor::PrintRenderState(nw::g3d::ResRenderState* pResRenderState)
{
    nw::g3d::ResRenderState::Mode mode = pResRenderState->GetMode();
    switch (mode)
    {
    case nw::g3d::ResRenderState::MODE_TRANSLUCENT: G3DSANDBOX_PRINT("Mode = Translucent\n"); break;
    case nw::g3d::ResRenderState::MODE_ALPHAMASK: G3DSANDBOX_PRINT("Mode = AlphaMask\n"); break;
    case nw::g3d::ResRenderState::MODE_OPAQUE: G3DSANDBOX_PRINT("Mode = Opaque\n"); break;
    case nw::g3d::ResRenderState::MODE_CUSTOM: G3DSANDBOX_PRINT("Mode = Custom\n"); break;
    default: G3DSANDBOX_PRINT("Mode = Unknown\n"); break;
    }

    nw::g3d::GfxAlphaTest& alphaTest = pResRenderState->GetAlphaTest();
    G3DSANDBOX_PRINT("AlphaTestEnable: %d\n", alphaTest.GetAlphaTestEnable());
    G3DSANDBOX_PRINT("AlphaFunc: %d\n", alphaTest.GetAlphaFunc());
}

#endif
