﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SandboxAllocator.h"
#include "SandboxUtility.h"

#include <g3ddemo_DemoUtility.h>

#ifdef _WIN32
#include <windows.h>
#include <imagehlp.h>

#pragma comment(lib, "imagehlp.lib")
#endif

#include <string>

#define NN_CONFIG_G3D_SANDBOX_ENABLED_MEMORY_ALLOCATION_LOG 0

using namespace G3dSandbox;

namespace {
    const int DefaultAlignment = 4;

#ifdef _WIN32
    const int FunctionNameLengthMax = 256;
    const int TraceBackStackDepth = 3;

    std::string GetCallstackOutputString( char* pOutSymbolNames, int nameLengthMax, int maxSymbolCount )
    {
        std::string outCallstack;
        for (int i = 0; i < maxSymbolCount; ++i)
        {
            char* pFuncName = pOutSymbolNames + nameLengthMax * i;
            outCallstack += std::string("    ") + std::string(pFuncName) + std::string("\n");
        }

        return outCallstack;
    }
#endif
}

void* G3dSandbox::Allocate(size_t size, size_t alignment, void* pUserData)
{
    SandboxAllocator* pAllocator = static_cast<SandboxAllocator*>(pUserData);
    return pAllocator->Alloc(size, alignment);
}

void G3dSandbox::Free(void* ptr, void* pUserData)
{
    SandboxAllocator* pAllocator = static_cast<SandboxAllocator*>(pUserData);
    pAllocator->Free(ptr);
}

void* G3dSandbox::Allocate(size_t size)
{
    return nw::g3d::demo::AllocMem2(size, DefaultAlignment);
}

void G3dSandbox::Free(void* ptr, size_t)
{
    nw::g3d::demo::FreeMem2(ptr);
}

void* SandboxAllocator::Alloc(size_t size, size_t alignment)
{
    void* pBuffer = nw::g3d::demo::AllocMem2(size, alignment);
    AllocationInfo info;
    info.address = reinterpret_cast<uintptr_t>(pBuffer);
#if NN_CONFIG_G3D_SANDBOX_ENABLED_MEMORY_ALLOCATION_LOG
#ifdef _WIN32
    char calledFunctionNames[FunctionNameLengthMax * TraceBackStackDepth];
    GetCalledSymbolName(calledFunctionNames, FunctionNameLengthMax, TraceBackStackDepth);
    strcpy(info.allocatedFunctionName, GetCallstackOutputString(calledFunctionNames, FunctionNameLengthMax, TraceBackStackDepth).c_str());
    G3DSANDBOX_PRINT("allocated: %08x\n%s\n", info.address, info.allocatedFunctionName);
#endif
#endif
    m_AllocatedAddressList.push_back(info);
    return pBuffer;
}

void SandboxAllocator::Free(void* pBuffer)
{
    uintptr_t address = reinterpret_cast<uintptr_t>(pBuffer);
#if NN_CONFIG_G3D_SANDBOX_ENABLED_MEMORY_ALLOCATION_LOG
#ifdef _WIN32
    AllocationInfo* pInfo = FindAllocationInfo(address);
    NW_G3D_UNUSED(pInfo);
    char calledFunctionNames[FunctionNameLengthMax * TraceBackStackDepth];
    GetCalledSymbolName(calledFunctionNames, FunctionNameLengthMax, TraceBackStackDepth);
    std::string callstacks = GetCallstackOutputString(calledFunctionNames, FunctionNameLengthMax, TraceBackStackDepth);
    G3DSANDBOX_PRINT("freed: %08x\n%s  allocated at\n%s\n",
        address,
        callstacks.c_str(),
        pInfo != NULL ? pInfo->allocatedFunctionName : "Not Found");
#endif
#endif
    bool removed = RemoveAddress(address);
    NW_G3D_UNUSED(removed);
    NW_G3D_ASSERT(removed);
    nw::g3d::demo::FreeMem2(pBuffer);
}



void SandboxAllocator::GetCalledSymbolName( char* pOutSymbolNames, int nameLengthMax, int maxSymbolCount )
{
#ifdef _WIN32
    HANDLE process = GetCurrentProcess();
    SymInitialize( process, NULL, TRUE );

    void* stack[ 100 ];
    CaptureStackBackTrace( 0, 100, stack, NULL );
    SYMBOL_INFO* pSymbol  = ( SYMBOL_INFO * )calloc( sizeof( SYMBOL_INFO ) + 256 * sizeof( char ), 1 );
    pSymbol->MaxNameLen   = 255;
    pSymbol->SizeOfStruct = sizeof( SYMBOL_INFO );

    std::string symbolName;
    for (int i = 0; i < maxSymbolCount; ++i)
    {
        int calledStackLevel = 2 + i;
        SymFromAddr( process, ( DWORD64 )( stack[ calledStackLevel ] ), 0, pSymbol );

        char* pDst = pOutSymbolNames + nameLengthMax * i;
        strcpy(pDst, pSymbol->Name);
    }

    free( pSymbol );
#else
    strcpy(pOutSymbolNames, "");
#endif
}

void SandboxAllocator::PrintMemoryLeaks()
{
    if (m_AllocatedAddressList.size() > 0)
    {
        G3DSANDBOX_PRINT("Detected Memory Leaks:\n");
        for (std::vector<AllocationInfo>::iterator iter = m_AllocatedAddressList.begin(), end = m_AllocatedAddressList.end();
            iter != end; ++iter)
        {
#if NN_CONFIG_G3D_SANDBOX_ENABLED_MEMORY_ALLOCATION_LOG
#ifdef _WIN32
            G3DSANDBOX_PRINT("  0x%016x(%s)\n", (*iter).address, (*iter).allocatedFunctionName);
#else
            G3DSANDBOX_PRINT("  0x%016x\n", (*iter).address);
#endif
#else
            G3DSANDBOX_PRINT("  0x%016x\n", (*iter).address);
#endif
        }
    }
    else
    {
        G3DSANDBOX_PRINT("No Memory Leaks\n");
    }
}


bool SandboxAllocator::RemoveAddress(uintptr_t address)
{
    for (std::vector<AllocationInfo>::iterator iter = m_AllocatedAddressList.begin(), end = m_AllocatedAddressList.end();
        iter != end; ++iter)
    {
        if ((*iter).address == address)
        {
            m_AllocatedAddressList.erase(iter);
            return true;
        }
    }

    return false;
}

SandboxAllocator::AllocationInfo* SandboxAllocator::FindAllocationInfo(uintptr_t address)
{
    for (std::vector<AllocationInfo>::iterator iter = m_AllocatedAddressList.begin(), end = m_AllocatedAddressList.end();
        iter != end; ++iter)
    {
        if ((*iter).address == address)
        {
            return &(*iter);
        }
    }

    return NULL;
}

