﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_DayOfWeek.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>

// nn::time が動作保証する西暦の期間
const int TestBeginYear = 2000;
const int TestEndYear = 2100;


/**
 * @brief   テスト向け TimeZoneRule のローダー
 */
class RuleLoader
{
    NN_DISALLOW_COPY(RuleLoader);
    NN_DISALLOW_MOVE(RuleLoader);
public:
    explicit RuleLoader(const char* zoneName):
        m_Name(zoneName)
    {
        m_pRule = static_cast<nn::time::TimeZoneRule*>( std::malloc(sizeof(nn::time::TimeZoneRule)));
        NN_ABORT_UNLESS_NOT_NULL(m_pRule);
    }
    ~RuleLoader()
    {
        std::free(m_pRule);
    }

    const char* ToString() const
    {
        static char buffer[256];
        nn::util::TSNPrintf(buffer, sizeof(buffer), "[%s]", m_Name);
        return buffer;
    }

    nn::Result Load()
    {
        nn::time::LocationName name;
        nn::util::Strlcpy(name._value, m_Name, nn::time::LocationName::Size);
        return nn::time::LoadTimeZoneRule(m_pRule, name);
    }

    const nn::time::TimeZoneRule& operator()() const
    {
        return *m_pRule;
    }

private:
    RuleLoader();
    nn::time::TimeZoneRule *m_pRule;
    const char* m_Name;
};


/**
 * @brief 曜日を決定する方法の種類
 */
enum class GettingWeekType
{
    First,  //!< 第一○曜日
    Second, //!< 第二○曜日
    Last,   //!< 最終○曜日
};

/**
 * @brief 年/月/GettingWeekType を指定して、所望の曜日を取得する
 * @param[in]   year
 * @param[in]   month
 * @param[in]   type        第一日曜日とかの、第一、の部分を enum で指定
 * @param[in]   dayOfWeek   曜日
 * @retval      曜日
 */
int GetDayOfSpecific(int year, int month, GettingWeekType type, nn::time::DayOfWeek dayOfWeek) NN_NOEXCEPT;


/**
 * @brief   テスト向け現地時刻.
 *          Shift() で指定秒数進めることができる.
 */
class TestCalendarType
{
public:
    TestCalendarType(int year, int month ,int day, int hour, int minute, int second)
    {
        c.year = static_cast<uint16_t>(year);
        c.month = static_cast<uint8_t>(month);
        c.day = static_cast<uint8_t>(day);
        c.hour = static_cast<uint8_t>(hour);
        c.minute = static_cast<uint8_t>(minute);
        c.second = static_cast<uint8_t>(second);
    }

    const char* ToString() const
    {
        static char buffer[256];
        nn::util::TSNPrintf(buffer, sizeof(buffer),
            "(%04d-%02d-%02d %02d:%02d:%02d)",
            c.year, c.month, c.day, c.hour, c.minute, c.second);
        return buffer;
    }

    bool operator == (const TestCalendarType& rhs) const
    {
        return c == rhs.c;
    }
    bool operator != (const TestCalendarType& rhs) const
    {
        return !( *this == rhs );
    }

    bool operator > (const TestCalendarType& rhs) const
    {
        if(this->c.year > rhs.c.year) return true;
        if(this->c.year < rhs.c.year) return false;

        if(this->c.month > rhs.c.month) return true;
        if(this->c.month < rhs.c.month) return false;

        if(this->c.day > rhs.c.day) return true;
        if(this->c.day < rhs.c.day) return false;

        if(this->c.hour > rhs.c.hour) return true;
        if(this->c.hour < rhs.c.hour) return false;

        if(this->c.minute > rhs.c.minute) return true;
        if(this->c.minute < rhs.c.minute) return false;

        if(this->c.second > rhs.c.second) return true;
        if(this->c.second < rhs.c.second) return false;

        return false;
    }

    bool operator < (const TestCalendarType& rhs) const
    {
        return !(*this > rhs) && (*this != rhs);
    }

    nn::time::CalendarTime c;

    nn::time::CalendarTime Shift(int64_t addSecond)
    {
        static bool first = true;
        static nn::time::TimeZoneRule utcRule;
        if(first)
        {
            nn::time::LocationName utc = {"UTC"};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::LoadTimeZoneRule(&utcRule, utc));
            first = false;
        }

        nn::time::PosixTime posix;
        int count;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToPosixTime(&count, &posix, 1, c, utcRule) );
        NN_ABORT_UNLESS_EQUAL(count, 1);
        posix.value += addSecond;

        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&c, nullptr, posix, utcRule) );

        return c;
    }
};

/**
 * @brief   テスト向けタイムゾーン情報
 */
class TimeZoneType
{
public:
    TimeZoneType():
        isDst(false),
        standardTimeName(""),
        saveHour(0)
    {
    }
    TimeZoneType(const char* standardTimeName_, bool isDst_, int saveHour_ = 0):
        isDst(isDst_),
        standardTimeName(standardTimeName_),
        saveHour(saveHour_)
    {
    }

    bool isDst;
    const char *standardTimeName;
    int saveHour; // 切り替わり時にスキップされる時間。1だと1時間進む。
};



/**
 * @brief   テスト向けタイムゾーンルールの定義
 * @details
 *  yearBegin <= year <= yearEnd の西暦がテスト範囲.
 *
 *  tz database 側と同様に、
 *  4月第一日曜日の2時にサマータイムに突入、
 *  10月最終日曜日の2時にサマータイム終了、
 *  といったようなルールを定義できます。
 */
struct TestRuleType
{
    int yearBegin;  //!< ルール開始の西暦
    int yearEnd;    //!< ルール終了の西暦

    // ○月、第○、○曜日、○時から所定の TimeZoneType 開始
    int month;
    GettingWeekType weekType;
    nn::time::DayOfWeek dayOfWeek;
    int startHour;
    TimeZoneType timeZoneType;
};

/**
 * @brief   入力の TestCalendarType の値のバリデーション
 */
#define NNT_TIME_ASSERT_CALENDAR_RANGE(testCalendar) \
    do \
    { \
        ASSERT_TRUE(1 <= testCalendar.c.month  && testCalendar.c.month  <= 12) << testCalendar.ToString(); \
        ASSERT_TRUE(1 <= testCalendar.c.day    && testCalendar.c.day    <= 31) << testCalendar.ToString(); \
        ASSERT_TRUE(0 <= testCalendar.c.hour   && testCalendar.c.hour   <= 23) << testCalendar.ToString(); \
        ASSERT_TRUE(0 <= testCalendar.c.minute && testCalendar.c.minute <= 59) << testCalendar.ToString(); \
        ASSERT_TRUE(0 <= testCalendar.c.second && testCalendar.c.second <= 59) << testCalendar.ToString(); \
    } \
    while(NN_STATIC_CONDITION(false))

/**
 * @brief   入力の TestCalendarType が存在しない日付であることをテストします
 */
#define NNT_TIME_NOT_FOUND(testCalendar) \
    do \
    { \
        nn::time::CalendarTime nntTimeLoopbackTestCalendar = testCalendar.c; \
        int nntTimeLoopbackTestOutCount; \
        nn::time::PosixTime nntTimeLoopbackTestOutPosixList[2] = {}; \
        NNT_TIME_ASSERT_CALENDAR_RANGE(testCalendar); \
        NNT_ASSERT_RESULT_SUCCESS( nn::time::ToPosixTime(&nntTimeLoopbackTestOutCount, nntTimeLoopbackTestOutPosixList, 2, nntTimeLoopbackTestCalendar, rule()) ) << rule.ToString() << testCalendar.ToString(); \
        ASSERT_EQ(0, nntTimeLoopbackTestOutCount) << rule.ToString() << testCalendar.ToString(); \
    } \
    while(NN_STATIC_CONDITION(false))

/**
 * @brief   入力の CalendarTime を PosixTime に変換後、
 *          その PosixTime を CalendarTime に変換して、
 *          入力値と一致しているかテストする
 *
 * @param[in]   expectCount         現地時刻を絶対時刻に変換したときの答えの数
 * @param[in]   testCalendar        テスト対象の TestCalendarType
 * @param[in]   standardTimeList    答えとして存在する TimeZoneType クラスの配列. 1つしかない場合は2つめの要素を nullptr に.
 *
 * @details
 *  テストには ASSERT_XXXX を利用します。
 *  よって、1つでも失敗した場合は以降のテストは実行されません。
 *
 *  直接使わず、以下のマクロを基本的に使用することを推奨します.
 *  - NNT_TIME_LOOPBACK_0
 *  - NNT_TIME_LOOPBACK_1
 *  - NNT_TIME_LOOPBACK_2
 */
#define NNT_TIME_LOOPBACK_TEST_BASE(expectCount, testCalendar, standardTimeList) \
    do \
    { \
        nn::time::CalendarTime nntTimeLoopbackTestCalendar = testCalendar.c; \
        int nntTimeLoopbackTestOutCount; \
        nn::time::PosixTime nntTimeLoopbackTestOutPosixList[2] = {}; \
        NNT_TIME_ASSERT_CALENDAR_RANGE(testCalendar); \
        NNT_ASSERT_RESULT_SUCCESS( nn::time::ToPosixTime(&nntTimeLoopbackTestOutCount, nntTimeLoopbackTestOutPosixList, 2, nntTimeLoopbackTestCalendar, rule()) ) << rule.ToString() << testCalendar.ToString(); \
        ASSERT_EQ(expectCount, nntTimeLoopbackTestOutCount) << rule.ToString() << testCalendar.ToString(); \
        for(int i = 0 ; i < expectCount ; ++i) \
        { \
            nn::time::CalendarTime nntTimeLoopbackTestTempCalendar; \
            nn::time::CalendarAdditionalInfo nntTimeLoopbackTestTempAdditional; \
            NNT_ASSERT_RESULT_SUCCESS( nn::time::ToCalendarTime(&nntTimeLoopbackTestTempCalendar, &nntTimeLoopbackTestTempAdditional, nntTimeLoopbackTestOutPosixList[i], rule()) ) << rule.ToString() << testCalendar.ToString(); \
            ASSERT_EQ(nntTimeLoopbackTestCalendar, nntTimeLoopbackTestTempCalendar) << rule.ToString() << testCalendar.ToString(); \
            ASSERT_STREQ(standardTimeList[i]->standardTimeName, nntTimeLoopbackTestTempAdditional.timeZone.standardTimeName) << rule.ToString() << testCalendar.ToString(); \
            ASSERT_EQ(standardTimeList[i]->isDst, nntTimeLoopbackTestTempAdditional.timeZone.isDaylightSavingTime) << rule.ToString() << testCalendar.ToString(); \
        } \
    } \
    while(NN_STATIC_CONDITION(false))

/*
 * @brief   PosixTime に変換したときに答えが0個になる現地時刻のテスト
 * @param[in]   rule    RuleLoader
 * @param[in]   begin   テスト対象となる現地時刻の始点
 * @param[in]   end     テスト対象となる現地時刻の終点
 * @param[in]   skip    現地時刻を何秒ずつ足してテストするか
 *
 * @details
 *  skip に何を設定しても、begin と end と同じ値の現地時刻が必ずテストされます。
 */
#define NNT_TIME_LOOPBACK_0(rule, begin, end, skip) \
    do \
    { \
        TestCalendarType nntTimeLoopback0Begin = begin; \
        TestCalendarType nntTimeLoopback0End = end; \
        TimeZoneType *nntTimeLoopback0StList[] = {nullptr}; \
        while(nntTimeLoopback0End > nntTimeLoopback0Begin) \
        { \
            NNT_TIME_LOOPBACK_TEST_BASE(0, nntTimeLoopback0Begin, nntTimeLoopback0StList); \
            nntTimeLoopback0Begin.Shift(skip); \
        } \
        { /* end のテストは確実に行う */ \
            NNT_TIME_LOOPBACK_TEST_BASE(0, nntTimeLoopback0End, nntTimeLoopback0StList); \
        } \
    } \
    while(NN_STATIC_CONDITION(false))

/*
 * @brief   PosixTime に変換したときに答えが1個になる現地時刻のテスト
 * @param[in]   rule        RuleLoader
 * @param[in]   begin       テスト対象となる現地時刻の始点
 * @param[in]   end         テスト対象となる現地時刻の終点
 * @param[in]   timezone    答えになるはずのTimeZoneType
 * @param[in]   skip        現地時刻を何秒ずつ足してテストするか
 *
 * @details
 *  skip に何を設定しても、begin と end と同じ値の現地時刻が必ずテストされます。
 */
#define NNT_TIME_LOOPBACK_1(rule, begin, end, timezone, skip) \
    do \
    { \
        TestCalendarType nntTimeLoopback1Begin = begin; \
        TestCalendarType nntTimeLoopback1End = end; \
        TimeZoneType nntTimeLoopback1TimeZoneType = timezone; \
        TimeZoneType *nntTimeLoopback1StList[] = {&nntTimeLoopback1TimeZoneType , nullptr}; \
        while(nntTimeLoopback1End > nntTimeLoopback1Begin) \
        { \
            NNT_TIME_LOOPBACK_TEST_BASE(1, nntTimeLoopback1Begin, nntTimeLoopback1StList); \
            nntTimeLoopback1Begin.Shift(skip); \
        } \
        { /* end のテストは確実に行う */ \
            NNT_TIME_LOOPBACK_TEST_BASE(1, nntTimeLoopback1End, nntTimeLoopback1StList); \
        } \
    } \
    while(NN_STATIC_CONDITION(false))

/*
 * @brief   PosixTime に変換したときに答えが2個になる現地時刻のテスト
 * @param[in]   rule        RuleLoader
 * @param[in]   begin       テスト対象となる現地時刻の始点
 * @param[in]   end         テスト対象となる現地時刻の終点
 * @param[in]   timezone1   答えになるはずの TimeZoneType の1つ目
 * @param[in]   timezone2   答えになるはずの TimeZoneType の2つ目
 * @param[in]   skip        現地時刻を何秒ずつ足してテストするか
 *
 * @details
 *  skip に何を設定しても、begin と end と同じ値の現地時刻が必ずテストされます。
 */
#define NNT_TIME_LOOPBACK_2(rule, begin, end, timezone1, timezone2, skip) \
    do \
    { \
        TestCalendarType nntTimeLoopback2Begin = begin; \
        TestCalendarType nntTimeLoopback2End = end; \
        TimeZoneType nntTimeLoopback2TimeZoneType1 = timezone1; \
        TimeZoneType nntTimeLoopback2TimeZoneType2 = timezone2; \
        TimeZoneType *nntTimeLoopback2StList[] = {&nntTimeLoopback2TimeZoneType1 , &nntTimeLoopback2TimeZoneType2 }; \
        while(nntTimeLoopback2End > nntTimeLoopback2Begin) \
        { \
            NNT_TIME_LOOPBACK_TEST_BASE(2, nntTimeLoopback2Begin, nntTimeLoopback2StList); \
            nntTimeLoopback2Begin.Shift(skip); \
        } \
        { /* end のテストは確実に行う */ \
            NNT_TIME_LOOPBACK_TEST_BASE(2, nntTimeLoopback2End, nntTimeLoopback2StList); \
        } \
    } \
    while(NN_STATIC_CONDITION(false))
