﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nnt.h>

#include "testTime_TimeZoneTestUtil.h"


using namespace nn::time;

namespace
{
    const char* TestLocationName = "America/Los_Angeles";
    const TimeZoneType StartTimeZone("PST", false, 0);

    // tz database のルールに合わせてテストケースを作る
    const TestRuleType TestRule[] =
    {
        //
        // 月日が昇順になるよう定義すること！
        // 年は順番に関係なく、そのルールが適応される西暦を指定してください。
        //

        {
            // 4月第一日曜日の 01:59:59(PST) の1秒後が 03:00:00(PDT) に
            2000, 2006,
            4, GettingWeekType::First, DayOfWeek_Sunday, 2, // 2時に saveHour が1に
            TimeZoneType("PDT", true, 1)
        },
        {
            // 10月最終日曜日の 01:59:59(PDT) の1秒後が 01:00:00(PST) に
            2000, 2006,
            10, GettingWeekType::Last, DayOfWeek_Sunday, 2, // 2時に saveHourが0に
            TimeZoneType("PST", false, 0)
        },

        // 2007年にルール改定
        {
            // 3月第二日曜日の 01:59:59(PST) の1秒後が 03:00:00(PDT) に
            2007, 2100,
            3, GettingWeekType::Second, DayOfWeek_Sunday, 2, // 2時に saveHour が1に
            TimeZoneType("PDT", true, 1)
        },
        {
            // EDT -> EST
            // 11月第一日曜日の 01:59:59(PDT) の1秒後が 01:00:00(PST) に
            2007, 2100,
            11, GettingWeekType::First, DayOfWeek_Sunday, 2, // 2時に saveHourが0に
            TimeZoneType("PST", false, 0)
        }
    };

    std::vector<std::pair<TestCalendarType, TimeZoneType>> g_TestCase;

    // America/Los_Angeles
    const TestCalendarType TestCalendarAnswer[] = // 夏時間の開始・終了、の順にカレンダーを決め打ち
    {
        // New_York と同じ

        // 2006年まで
        // 4月第一日曜日 PDT 開始
        // 10月最終日曜日 PST 開始
        TestCalendarType(2000,  4,  2, 2, 0, 0),
        TestCalendarType(2000, 10, 29, 2, 0, 0),

        TestCalendarType(2001,  4,  1, 2, 0, 0),
        TestCalendarType(2001, 10, 28, 2, 0, 0),

        TestCalendarType(2002,  4,  7, 2, 0, 0),
        TestCalendarType(2002, 10, 27, 2, 0, 0),

        TestCalendarType(2003,  4,  6, 2, 0, 0),
        TestCalendarType(2003, 10, 26, 2, 0, 0),

        TestCalendarType(2004,  4,  4, 2, 0, 0),
        TestCalendarType(2004, 10, 31, 2, 0, 0),

        TestCalendarType(2005,  4,  3, 2, 0, 0),
        TestCalendarType(2005, 10, 30, 2, 0, 0),

        TestCalendarType(2006,  4,  2, 2, 0, 0),
        TestCalendarType(2006, 10, 29, 2, 0, 0),

        // 2007年から
        // 3月の8日以上の日曜日に PDT 開始. 2時台スキップ
        // 11月の1日以上の日曜日に PST 開始. 1時台重複
        TestCalendarType(2007,  3, 11, 2, 0, 0),
        TestCalendarType(2007, 11,  4, 2, 0, 0),

        TestCalendarType(2008,  3, 9, 2, 0, 0),
        TestCalendarType(2008, 11, 2, 2, 0, 0),

        TestCalendarType(2009,  3, 8, 2, 0, 0),
        TestCalendarType(2009, 11, 1, 2, 0, 0),

        TestCalendarType(2010,  3, 14, 2, 0, 0),
        TestCalendarType(2010, 11,  7, 2, 0, 0),

        TestCalendarType(2011,  3, 13, 2, 0, 0),
        TestCalendarType(2011, 11,  6, 2, 0, 0),

        TestCalendarType(2012,  3, 11, 2, 0, 0),
        TestCalendarType(2012, 11,  4, 2, 0, 0),

        TestCalendarType(2013,  3, 10, 2, 0, 0),
        TestCalendarType(2013, 11,  3, 2, 0, 0),

        TestCalendarType(2014,  3, 9, 2, 0, 0),
        TestCalendarType(2014, 11, 2, 2, 0, 0),

        TestCalendarType(2015,  3, 8, 2, 0, 0),
        TestCalendarType(2015, 11, 1, 2, 0, 0),

        TestCalendarType(2016,  3, 13, 2, 0, 0),
        TestCalendarType(2016, 11,  6, 2, 0, 0),

        TestCalendarType(2017,  3, 12, 2, 0, 0),
        TestCalendarType(2017, 11,  5, 2, 0, 0),

        TestCalendarType(2018,  3, 11, 2, 0, 0),
        TestCalendarType(2018, 11,  4, 2, 0, 0),

        TestCalendarType(2019,  3, 10, 2, 0, 0),
        TestCalendarType(2019, 11,  3, 2, 0, 0),

        TestCalendarType(2020,  3, 8, 2, 0, 0),
        TestCalendarType(2020, 11, 1, 2, 0, 0),

        TestCalendarType(2021,  3, 14, 2, 0, 0),
        TestCalendarType(2021, 11,  7, 2, 0, 0),

        TestCalendarType(2022,  3, 13, 2, 0, 0),
        TestCalendarType(2022, 11,  6, 2, 0, 0),

        TestCalendarType(2023,  3, 12, 2, 0, 0),
        TestCalendarType(2023, 11,  5, 2, 0, 0),

        TestCalendarType(2024,  3, 10, 2, 0, 0),
        TestCalendarType(2024, 11,  3, 2, 0, 0),

        TestCalendarType(2025,  3, 9, 2, 0, 0),
        TestCalendarType(2025, 11, 2, 2, 0, 0),

        TestCalendarType(2026,  3, 8, 2, 0, 0),
        TestCalendarType(2026, 11, 1, 2, 0, 0),

        TestCalendarType(2027,  3, 14, 2, 0, 0),
        TestCalendarType(2027, 11,  7, 2, 0, 0),

        TestCalendarType(2028,  3, 12, 2, 0, 0),
        TestCalendarType(2028, 11,  5, 2, 0, 0),

        TestCalendarType(2029,  3, 11, 2, 0, 0),
        TestCalendarType(2029, 11,  4, 2, 0, 0),

        TestCalendarType(2030,  3, 10, 2, 0, 0),
        TestCalendarType(2030, 11,  3, 2, 0, 0),

        TestCalendarType(2031,  3, 9,  2, 0, 0),
        TestCalendarType(2031, 11, 2,  2, 0, 0),

        TestCalendarType(2032,  3, 14, 2, 0, 0),
        TestCalendarType(2032, 11,  7, 2, 0, 0),

        TestCalendarType(2033,  3, 13, 2, 0, 0),
        TestCalendarType(2033, 11,  6, 2, 0, 0),

        TestCalendarType(2034,  3, 12, 2, 0, 0),
        TestCalendarType(2034, 11,  5, 2, 0, 0),

        TestCalendarType(2035,  3, 11, 2, 0, 0),
        TestCalendarType(2035, 11,  4, 2, 0, 0),

        TestCalendarType(2036,  3, 9, 2, 0, 0),
        TestCalendarType(2036, 11, 2, 2, 0, 0),

        TestCalendarType(2037,  3, 8, 2, 0, 0),
        TestCalendarType(2037, 11, 1, 2, 0, 0),

        TestCalendarType(2038,  3, 14, 2, 0, 0),
        TestCalendarType(2038, 11,  7, 2, 0, 0),

        TestCalendarType(2039,  3, 13, 2, 0, 0),
        TestCalendarType(2039, 11,  6, 2, 0, 0),

        TestCalendarType(2040,  3, 11, 2, 0, 0),
        TestCalendarType(2040, 11,  4, 2, 0, 0),

        TestCalendarType(2041,  3, 10, 2, 0, 0),
        TestCalendarType(2041, 11,  3, 2, 0, 0),

        TestCalendarType(2042,  3, 9, 2, 0, 0),
        TestCalendarType(2042, 11, 2, 2, 0, 0),

        TestCalendarType(2043,  3, 8, 2, 0, 0),
        TestCalendarType(2043, 11, 1, 2, 0, 0),

        TestCalendarType(2044,  3, 13, 2, 0, 0),
        TestCalendarType(2044, 11,  6, 2, 0, 0),

        TestCalendarType(2045,  3, 12, 2, 0, 0),
        TestCalendarType(2045, 11,  5, 2, 0, 0),

        TestCalendarType(2046,  3, 11, 2, 0, 0),
        TestCalendarType(2046, 11,  4, 2, 0, 0),

        TestCalendarType(2047,  3, 10, 2, 0, 0),
        TestCalendarType(2047, 11,  3, 2, 0, 0),

        TestCalendarType(2048,  3, 8, 2, 0, 0),
        TestCalendarType(2048, 11, 1, 2, 0, 0),

        TestCalendarType(2049,  3, 14, 2, 0, 0),
        TestCalendarType(2049, 11,  7, 2, 0, 0),

        TestCalendarType(2050,  3, 13, 2, 0, 0),
        TestCalendarType(2050, 11,  6, 2, 0, 0),
    };
}

TEST(TimeZone_AmericaLosAngeles, GenerateTestCase)
{
    for(int y = TestBeginYear ; y <= TestEndYear ; ++y)
    {
        for(auto& rule : TestRule)
        {
            if(rule.yearBegin <= y && y <= rule.yearEnd)
            {
                int day = GetDayOfSpecific(y, rule.month, rule.weekType, rule.dayOfWeek);
                g_TestCase.push_back(
                    std::make_pair(
                        TestCalendarType(y, rule.month, day, rule.startHour, 0, 0),
                        rule.timeZoneType)
                    );
            }
        }
    }
    // TestCalendarAnswer を手動で2050年までWindowsのカレンダーを目視で作ったので一致するか確認
    int i = 0;
    for(auto& answer : TestCalendarAnswer)
    {
        EXPECT_EQ(answer.c, g_TestCase[ i ].first.c) << answer.ToString() << " " << g_TestCase[ i ].first.ToString();
        ++i;
    }
}

TEST(TimeZone_AmericaLosAngeles, ExecutionTestCase)
{
    NNT_ASSERT_RESULT_SUCCESS( Initialize() );
    RuleLoader rule(TestLocationName);
    NNT_ASSERT_RESULT_SUCCESS(rule.Load());

    const int64_t testSkipSeconds = nn::TimeSpan::FromHours(18).GetSeconds() + 1;

    TestCalendarType begin(TestBeginYear, 1, 1, 0, 0, 0);

    TimeZoneType currentTimeZone = StartTimeZone;

    for(auto &test : g_TestCase)
    {
        const auto& calendar = test.first;
        const auto& timeZone = test.second;

        if(currentTimeZone.saveHour == timeZone.saveHour)
        {
            // 時刻はずれない. 表示の名称とかだけが変わる、など。

            // 切り替わり直前まで
            TestCalendarType end = calendar;
            end.Shift(-1);
            NNT_TIME_LOOPBACK_1(rule, begin, end, currentTimeZone, testSkipSeconds);

            // 切り替わり後からテスト続行
            begin = calendar;
        }
        else if(currentTimeZone.saveHour == 0 && timeZone.saveHour != 0) // 時間が進むとき
        {
            // スキップ直前まで
            TestCalendarType end = calendar;
            end.Shift(-1);
            NNT_TIME_LOOPBACK_1(rule, begin, end, currentTimeZone, testSkipSeconds);

            begin = calendar; // スキップ開始時刻
            end = begin;
            end.Shift(nn::TimeSpan::FromHours(timeZone.saveHour).GetSeconds() - 1); // スキップ終了の1秒前
            NNT_TIME_LOOPBACK_0(rule, begin, end, 1);

            // 次のテストはスキップ終わってから
            begin = calendar;
            begin.Shift(nn::TimeSpan::FromHours(timeZone.saveHour).GetSeconds());

            // スキップ終わってから少しは細かいskipでテストしておく
            end = begin;
            end.Shift(60 * 30);
            NNT_TIME_LOOPBACK_1(rule, begin, end, timeZone, 1);
        }
        else if(currentTimeZone.saveHour != 0 && timeZone.saveHour == 0) // 時間が巻き戻るとき
        {
            // 重複直前まで
            TestCalendarType end = calendar;
            end.Shift(nn::TimeSpan::FromHours(-currentTimeZone.saveHour).GetSeconds() - 1); //重複1秒前
            NNT_TIME_LOOPBACK_1(rule, begin, end, currentTimeZone, testSkipSeconds);

            begin = calendar; // 重複終了時刻
            begin.Shift(nn::TimeSpan::FromHours(-currentTimeZone.saveHour).GetSeconds()); // 重複開始時刻
            end = calendar;
            end.Shift(-1); // 重複終了の1秒前
            NNT_TIME_LOOPBACK_2(
                rule, begin, end,
                currentTimeZone, timeZone,
                1);

            // 次のテストは重複終わってから
            begin = calendar;
            begin.Shift(nn::TimeSpan::FromHours(currentTimeZone.saveHour).GetSeconds());

            // 重複終わってから少しは細かいskipでテストしておく
            end = begin;
            end.Shift(nn::TimeSpan::FromMinutes(30).GetSeconds());
            NNT_TIME_LOOPBACK_1(rule, begin, end, timeZone, 1);
        }
        currentTimeZone = timeZone;

    }

    NNT_ASSERT_RESULT_SUCCESS( Finalize() );
}
