﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/timesrv/detail/core/timesrv_StandardSteadyClockCore.h>

// 初回起動、および電池切れによるRTCリセット時を想定
TEST(StandardSteadyClockCoreSuite, InitialStateTest)
{
    nn::timesrv::detail::core::StandardSteadyClockCore clock;

    nn::time::SourceId latestSourceId;
    clock.SetupSourceId(&latestSourceId, nn::util::InvalidUuid);
    EXPECT_NE(nn::util::InvalidUuid, latestSourceId);

    // GetCurrentTimePoint 経由で取得した SourceId 確認
    nn::time::SteadyClockTimePoint timePoint;
    NNT_ASSERT_RESULT_SUCCESS( clock.GetCurrentTimePoint(&timePoint) );
    EXPECT_NE(nn::util::InvalidUuid, timePoint.sourceId);
    EXPECT_EQ(latestSourceId, timePoint.sourceId);
}

TEST(StandardSteadyClockCoreSuite, NormalStateTest)
{
    nn::timesrv::detail::core::StandardSteadyClockCore clock;

    nn::time::SourceId currentSourceId;
    currentSourceId.FromString("12345678-1234-1234-1234-123456789012"); // 適当に有効な UUID を作る

    nn::time::SourceId latestSourceId;
    clock.SetupSourceId(&latestSourceId, currentSourceId);
    EXPECT_EQ(latestSourceId, currentSourceId);

    // GetCurrentTimePoint 経由で取得した SourceId 確認
    nn::time::SteadyClockTimePoint timePoint;
    NNT_ASSERT_RESULT_SUCCESS( clock.GetCurrentTimePoint(&timePoint) );
    EXPECT_EQ(currentSourceId, timePoint.sourceId);
}

TEST(StandardSteadyClockCoreSuite, TestOffset)
{
    nn::timesrv::detail::core::StandardSteadyClockCore clock;
    clock.SetupSourceId(nullptr, nn::util::InvalidUuid);

    EXPECT_EQ(nn::TimeSpan(), clock.GetTestOffset()); // Initial value is zero.

    // 現在時刻
    nn::time::SteadyClockTimePoint timePoint;
    NNT_ASSERT_RESULT_SUCCESS( clock.GetCurrentTimePoint(&timePoint) );

    // 10秒のテストオフセットを仕込む
    clock.SetTestOffset(nn::TimeSpan::FromSeconds(10));
    EXPECT_EQ(nn::TimeSpan::FromSeconds(10), clock.GetTestOffset());

    // テストオフセット分ずれて現在時刻が取得できるはず
    nn::time::SteadyClockTimePoint timePointWithOffset;
    NNT_EXPECT_RESULT_SUCCESS( clock.GetCurrentTimePoint(&timePointWithOffset) );
    // 実行時間の誤差もあるので1秒以内の誤差でチェック
    EXPECT_NEAR(
        static_cast<double>(timePoint.value + 10),
        static_cast<double>(timePointWithOffset.value), 1);
}

TEST(StandardSteadyClockCoreSuite, InternalOffset)
{
    nn::timesrv::detail::core::StandardSteadyClockCore clock;
    clock.SetupSourceId(nullptr, nn::util::InvalidUuid);
    nn::TimeSpan previousInternalOffset = clock.GetInternalOffset();

    nn::time::SteadyClockTimePoint timePoint;
    NNT_EXPECT_RESULT_SUCCESS( clock.GetCurrentTimePoint(&timePoint) );

    clock.SetInternalOffset(previousInternalOffset + nn::TimeSpan::FromSeconds(100));

    // InternalOffset分ずれて現在時刻が取得できるはず
    nn::time::SteadyClockTimePoint timePointWithOffset;
    NNT_EXPECT_RESULT_SUCCESS( clock.GetCurrentTimePoint(&timePointWithOffset) );
    // 実行時間の誤差もあるので1秒以内の誤差でチェック
    EXPECT_NEAR(
        static_cast<double>(timePoint.value + 100),
        static_cast<double>(timePointWithOffset.value), 1);
}
