﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/mem.h>
#include <nn/fs.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_TimeZoneApi.h>

#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_StandardUserSystemClock.h>

#include <nn/time/time_StandardNetworkSystemClockPrivilegeApi.h>
#include <nn/time/time_StandardUserSystemClockPrivilegeApi.h>

#include <nn/time/time_ResultPrivate.h>

#include <nn/time/detail/time_TimeZonePrivateApi.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>

//
// time:a だけもっているテストプログラム
//

TEST(TimeOnlyAdminCapability, Initialize)
{
    auto result = nn::time::Initialize();
#if NN_BUILD_CONFIG_OS_WIN
    EXPECT_TRUE(result.IsSuccess()); // win だと権限チェックないので成功してしまう
#else
    EXPECT_TRUE(result.IsFailure());
#endif
    if(result.IsSuccess())
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }
}

TEST(TimeOnlyAdminCapability, StandardNetworkSystemClockPrivilege)
{
    // time:s もってないのでネットワーク時計はいじれない

    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() ); // まだ time:s 向け初期化はない

    nn::time::SystemClockContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&context));

    // 適当な時刻をセットしてみる
    {
        nn::time::PosixTime current = {12345678};
        auto result = nn::time::SetStandardNetworkSystemClockCurrentTime(current);
#if NN_BUILD_CONFIG_OS_WIN
        EXPECT_TRUE(result.IsSuccess()); // win だと権限チェックないので成功してしまう
#else
        EXPECT_TRUE(result.IsFailure());
#endif
    }

    // 他に影響しないよう戻す
    {
    auto result = nn::time::SetStandardNetworkSystemClockContext(context);
#if NN_BUILD_CONFIG_OS_WIN
        EXPECT_TRUE(result.IsSuccess()); // win だと権限チェックないので成功してしまう
#else
        EXPECT_TRUE(result.IsFailure());
#endif
    }

    NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeOnlyAdminCapability, StandardUserSystemClockPrivilegeSystemClockContext)
{
    // time:a はあるので StandardUserSystemClock の操作はOK

    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );

    nn::time::SystemClockContext context;
    NNT_ASSERT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&context) );

    // 適当な時刻をセットしてみる
    {
        nn::time::PosixTime current = {12345678};
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardLocalSystemClockCurrentTime(current) );
    }

    // 他に影響しないよう戻す
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardLocalSystemClockContext(context) );
    }
    NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeOnlyAdminCapability, StandardUserSystemClockPrivilegeAutomaticCorrectionEnabled)
{
    // time:a はあるので StandardUserSystemClock の操作はOK
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );

    bool origin = nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled();

    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(false);
    EXPECT_EQ(false, nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled());

    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(true);
    EXPECT_EQ(true, nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled());

    // 他に影響しないよう戻す
    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(origin);
    EXPECT_EQ(origin, nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled());

    NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeOnlyAdminCapability, SetDeviceLocationName)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );

    nn::time::LocationName backup;
    nn::time::GetDeviceLocationName(&backup);
    ASSERT_STRNE(backup._value, ""); // 空ではないことだけ確認しておく

    {
        nn::time::LocationName name = {"Asia/Tokyo"};
        nn::time::SetDeviceLocationName(name);

        nn::time::LocationName outName;
        nn::time::GetDeviceLocationName(&outName);

        EXPECT_STREQ("Asia/Tokyo", outName._value);
        {
            // Asia/Tokyo が適用されているか確認
            nn::time::CalendarTime calendar;
            nn::time::CalendarAdditionalInfo additional;
            nn::time::PosixTime inPosixTime = {123456789};
            NNT_ASSERT_RESULT_SUCCESS( ToCalendarTime(&calendar, &additional, inPosixTime) );
            EXPECT_STREQ("JST", additional.timeZone.standardTimeName); // 標準時だけとりあえず一致テストしておく
        }
    }

    {
        nn::time::LocationName name = {"UTC"};
        nn::time::SetDeviceLocationName(name);

        nn::time::LocationName outName;
        nn::time::GetDeviceLocationName(&outName);

        EXPECT_STREQ("UTC", outName._value);
        {
            // UTC が適用されているか確認
            nn::time::CalendarTime calendar;
            nn::time::CalendarAdditionalInfo additional;
            nn::time::PosixTime inPosixTime = {123456789};
            NNT_ASSERT_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendar, &additional, inPosixTime) );
            EXPECT_STREQ("UTC", additional.timeZone.standardTimeName); // 標準時だけとりあえず一致テストしておく
        }
    }

    nn::time::SetDeviceLocationName(backup); // 元に戻す

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeOnlyAdminCapability, GetDeviceLocationNameAndUpdatedTime)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );

    nn::time::SteadyClockTimePoint current;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&current));

    // LocationName更新. updatedTime がほぼ current と同じになっているはず
    {
        nn::time::LocationName name = {"UTC"};
        nn::time::SetDeviceLocationName(name);
    }

    nn::time::LocationName name;
    nn::time::SteadyClockTimePoint updatedTime;
    nn::time::detail::GetDeviceLocationNameAndUpdatedTime(&name, &updatedTime);

    int64_t elapsedSeconds;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::GetSpanBetween(&elapsedSeconds, current, updatedTime));

    EXPECT_TRUE(0 <= elapsedSeconds && elapsedSeconds <= 5) << "elapsedSeconds:" << elapsedSeconds;

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

