﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_SdkLog.h>
#include <nn/nifm.h>
#include <nn/ntc/shim/ntc_Shim.h>
#include <nn/os.h>
#include <nn/time.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_EphemeralNetworkSystemClock.h>
#include <nn/time/time_EphemeralNetworkSystemClockPrivilegeApi.h>
#include <nn/time/time_EphemeralNetworkSystemClockTestApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Uuid.h>

#include <nn/timesrv/detail/core/timesrv_TickBasedSteadyClockCore.h>

class EphemeralNetworkSystemClockTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize()); // win版だと SuspendAutonomicTimeCorrection() に必要
        NNT_ASSERT_RESULT_SUCCESS(nn::time::InitializeForSystem());

        nn::time::SuspendAutonomicTimeCorrection();
    }

    static void TearDownTestCase()
    {
        nn::time::ResumeAutonomicTimeCorrection();

        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
    }
};

TEST_F(EphemeralNetworkSystemClockTest, Basic)
{
    nn::time::PosixTime posixTime;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    NNT_EXPECT_RESULT_SUCCESS(nn::time::EphemeralNetworkSystemClock::GetCurrentTime(&posixTime));
#else
    auto result = nn::time::EphemeralNetworkSystemClock::GetCurrentTime(&posixTime);
    EXPECT_TRUE(result.IsSuccess() || nn::time::ResultOffsetInvalid::Includes(result));
#endif
}

TEST_F(EphemeralNetworkSystemClockTest, EnsureClock)
{
    // コンテキストを無効化する
    nn::time::SystemClockContext context = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::time::GetEphemeralNetworkSystemClockContext(&context));
    context.timeStamp.sourceId = nn::util::InvalidUuid;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::SetEphemeralNetworkSystemClockContext(context));

    // コンテキストが無効なので失敗するはず
    nn::time::PosixTime posixTime;
    NNT_ASSERT_RESULT_FAILURE(nn::time::ResultOffsetInvalid, nn::time::EphemeralNetworkSystemClock::GetCurrentTime(&posixTime));

    nn::nifm::SubmitNetworkRequestAndWait();
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    // 強制補正
    nn::ntc::shim::CorrectionNetworkClockAsyncTask
        ensureTask(nn::os::EventClearMode_AutoClear, nn::ntc::EnsureNetworkClockAvailabilityMode_ForcibleDownload);
    NNT_ASSERT_RESULT_SUCCESS(ensureTask.StartTask());
    ensureTask.GetFinishNotificationEvent().Wait();
    NNT_ASSERT_RESULT_SUCCESS(ensureTask.GetResult());

    nn::time::PosixTime posixTime1, posixTime2;
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::time::EphemeralNetworkSystemClock::GetCurrentTime(&posixTime1));
        const nn::time::CalendarTime c = nn::time::ToCalendarTimeInUtc(posixTime1);
        NN_SDK_LOG("ServerTime : %04d/%02d/%02d %02d:%02d:%02d (UTC)\n",
            c.year, c.month, c.day, c.hour, c.minute, c.second);
    }
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::time::EphemeralNetworkSystemClock::GetCurrentTime(&posixTime2));
        const nn::time::CalendarTime c = nn::time::ToCalendarTimeInUtc(posixTime2);
        NN_SDK_LOG("ServerTime : %04d/%02d/%02d %02d:%02d:%02d (UTC)\n",
            c.year, c.month, c.day, c.hour, c.minute, c.second);
    }

    auto diff = posixTime2 - posixTime1;
    EXPECT_TRUE(nn::TimeSpan::FromSeconds(3) <= diff && diff <= nn::TimeSpan::FromSeconds(4)) << diff.GetSeconds();

    // 補正後コンテキストが変わるはず
    nn::time::SystemClockContext context2 = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::time::GetEphemeralNetworkSystemClockContext(&context2));
    EXPECT_NE(context, context2);
}

TEST(TickBasedSteadyClockCore, TickBasedSteadyClockCore)
{
    nn::timesrv::detail::core::TickBasedSteadyClockCore clock1;
    nn::time::SteadyClockTimePoint tp1;
    clock1.GetCurrentTimePoint(&tp1);

    nn::timesrv::detail::core::TickBasedSteadyClockCore clock2;
    nn::time::SteadyClockTimePoint tp2;
    clock2.GetCurrentTimePoint(&tp2);

    EXPECT_NE(tp1.sourceId, nn::util::InvalidUuid);
    EXPECT_NE(tp2.sourceId, nn::util::InvalidUuid);
    EXPECT_NE(tp1.sourceId, tp2.sourceId);
}
