﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/mem.h>
#include <nn/fs.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_StandardUserSystemClock.h>

#include <nn/time/time_StandardNetworkSystemClockPrivilegeApi.h>
#include <nn/time/time_StandardUserSystemClockPrivilegeApi.h>

#include <nn/time/time_ResultPrivate.h>
#include <nn/timesrv/timesrv_TimeZoneRuleInner.h>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/os/os_Tick.h>

#if NN_BUILD_CONFIG_OS_WIN

// Windows.h に GetCurrentTime があるので、衝突しないかの確認

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif

#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <nn/nn_Windows.h>

#endif // NN_BUILD_CONFIG_OS_WIN

// 時刻セット後に時刻取得して値が近いかテストするサブ関数.
// IPCや設定の永続化で誤差が生じるため一定範囲の誤差を許容.
// currentValue : 直前に取得した時刻の値
// setValue : 時計にセットした時刻の値
#define NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(currentValue, setValue) \
    do { \
        const int64_t allowedRange = 5; \
        EXPECT_LE((setValue), (currentValue)); \
        auto diff = (currentValue) - (setValue); \
        EXPECT_LE(diff, allowedRange) << "diff:" << diff << " currentValue:" << (currentValue) << " setValue:" << (setValue); \
    } while(NN_STATIC_CONDITION(0))

namespace
{
    // ユーザー時計の自動補正ON/OFF
    // time:a 必須
    void SetUserClockAutomaticCorrectionEnabled(bool value) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
        nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(value);
        EXPECT_EQ(value, nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled());
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }

    // ユーザ時計の補正
    // time:a 必須
    void SetUserClockCurrentTime(const nn::time::PosixTime& posixTime) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardLocalSystemClockCurrentTime(posixTime) );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }

    // ネットワーク時計の補正
    // time:s 必須
    void SetNetworkClockCurrentTime(const nn::time::PosixTime& posixTime) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockCurrentTime(posixTime) );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }

    // ネットワーク時計の補正を解除
    // time:s 必須
    void UnadjustedNetworkClock() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::InitializeForSystem() );
        nn::time::SystemClockContext context = {};
        context.timeStamp.sourceId = nn::util::InvalidUuid;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockContext(context) );
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
    }

    /*
    // ネットワーク時計が補正されているかどうか
    bool IsNetworkClockAdjusted() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS( nn::time::Initialize() );

        // SystemClockContext 取得してみて、ネットワーク時計が補正されているかチェック
        nn::time::SystemClockContext networkContext;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardNetworkSystemClock::GetSystemClockContext(&networkContext) );

        nn::time::SteadyClockTimePoint steadyClockTimePoint;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardSteadyClock::GetCurrentTimePoint(&steadyClockTimePoint) );

        NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );

        // source Id が一致していれば補正済
        return networkContext.timeStamp.sourceId == steadyClockTimePoint.sourceId;
    }
    */

#if !defined (NN_BUILD_CONFIG_OS_WIN)
    template <typename T>
    void GetTimeFwdbgSettings(T* pOut, const char* key) NN_NOEXCEPT
    {
        auto outBytes = nn::settings::fwdbg::GetSettingsItemValue(
            pOut,
            sizeof(T),
            "time",
            key);
        NN_ABORT_UNLESS_EQUAL(outBytes, sizeof(T));
    }
#endif
}

TEST(TimeBasic, InitializeFinalize)
{
    EXPECT_FALSE( nn::time::IsInitialized() );
    EXPECT_FALSE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
    EXPECT_FALSE( nn::time::IsInitialized() );
    EXPECT_FALSE( nn::time::IsInitialized() );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
    EXPECT_FALSE( nn::time::IsInitialized() );
}

TEST(TimeBasic, InitializeFinalizeForMenu)
{
    EXPECT_FALSE( nn::time::IsInitialized() );
    EXPECT_FALSE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
    EXPECT_FALSE( nn::time::IsInitialized() );
    EXPECT_FALSE( nn::time::IsInitialized() );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
    EXPECT_TRUE( nn::time::IsInitialized() );
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
    EXPECT_FALSE( nn::time::IsInitialized() );
}

TEST(TimeBasic, GetSpanBetweenBasic)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );


    int64_t seconds;

    {
        nn::time::SteadyClockTimePoint tp0 = {0, nn::util::InvalidUuid};
        nn::time::SteadyClockTimePoint tp1 = {1, nn::util::InvalidUuid};
        nn::time::SteadyClockTimePoint tp2 = {2, nn::util::InvalidUuid};

        NNT_ASSERT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tp0, tp0) );
        EXPECT_EQ(0, seconds);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tp1, tp1) );
        EXPECT_EQ(0, seconds);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tp2, tp2) );
        EXPECT_EQ(0, seconds);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tp0, tp1) );
        EXPECT_EQ(1, seconds);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tp1, tp2) );
        EXPECT_EQ(1, seconds);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tp1, tp0) );
        EXPECT_EQ(-1, seconds);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tp2, tp1) );
        EXPECT_EQ(-1, seconds);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, GetSpanBetweenOverflow)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    int64_t seconds;
    nn::time::SteadyClockTimePoint tpMin = {INT64_MIN, nn::util::InvalidUuid};
    nn::time::SteadyClockTimePoint tpMax = {INT64_MAX, nn::util::InvalidUuid};
    nn::time::SteadyClockTimePoint tpPlus1 = {1, nn::util::InvalidUuid};
    nn::time::SteadyClockTimePoint tpMinus1 = {-1, nn::util::InvalidUuid};
    nn::time::SteadyClockTimePoint tpMinPlus1 = {INT64_MIN + 1, nn::util::InvalidUuid};
    nn::time::SteadyClockTimePoint tpMaxMinus1 = {INT64_MAX - 1, nn::util::InvalidUuid};

    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOverflowed,  nn::time::GetSpanBetween(&seconds, tpMax, tpMin) );
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOverflowed,  nn::time::GetSpanBetween(&seconds, tpMin, tpMax) );

    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOverflowed,  nn::time::GetSpanBetween(&seconds, tpPlus1, tpMin ) );
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOverflowed,  nn::time::GetSpanBetween(&seconds, tpMinus1, tpMax ) );

    // ぎりぎりOK
    NNT_EXPECT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tpPlus1, tpMinPlus1) );
    NNT_EXPECT_RESULT_SUCCESS( nn::time::GetSpanBetween(&seconds, tpMinus1, tpMaxMinus1) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, GetSpanBetweenNotComparable)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    int64_t seconds;

    nn::time::SourceId sourceId1 = nn::util::InvalidUuid;
    nn::time::SourceId sourceId2 = nn::util::InvalidUuid;

    // 適当に変える
    sourceId1.data[0] = 1;
    sourceId2.data[0] = 2;

    nn::time::SteadyClockTimePoint tp1 = {1, sourceId1};
    nn::time::SteadyClockTimePoint tp2 = {2, sourceId2};

    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultNotComparable,  nn::time::GetSpanBetween(&seconds, tp1, tp2) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, StandardSteadyClock)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    char buffer[nn::util::Uuid::StringSize];
    nn::time::SteadyClockTimePoint val;
    nn::TimeSpan offset(0);

    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardSteadyClock::GetCurrentTimePoint(&val) );
    NN_LOG("StandardSteadyClock val:%lld, resourceId:%s\n", val.value, val.sourceId.ToString(buffer, sizeof(buffer)) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// ネットワーク時計未補正
TEST(TimeBasic, StandardNetworkSystemClock1)
{
    {
        SCOPED_TRACE("");
        UnadjustedNetworkClock();
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    // 補正前なので取得できない
    // TODO:補正前に返る値の仕様で EXPECT_EQ
    nn::time::PosixTime pt;
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOffsetInvalid, nn::time::StandardNetworkSystemClock::GetCurrentTime(&pt) );
    NN_LOG("StandardNetworkSystemClock PosixTime:%lld\n", pt.value);


    // SystemClockContext は取得だけできる
    nn::time::SystemClockContext context;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardNetworkSystemClock::GetSystemClockContext(&context) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// ネットワーク時計補正済
TEST(TimeBasic, StandardNetworkSystemClock2)
{
    {
        SCOPED_TRACE("");
        nn::time::PosixTime posix = {123456};
        SetNetworkClockCurrentTime(posix);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::PosixTime pt;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardNetworkSystemClock::GetCurrentTime(&pt) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        pt.value,
        123456);

    nn::time::SystemClockContext context;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardNetworkSystemClock::GetSystemClockContext(&context) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// 自動補正off, ネットワーク未補正
TEST(TimeBasic, StandardUserSystemClock1)
{
    {
        SCOPED_TRACE("");
        SetUserClockAutomaticCorrectionEnabled(false);
        UnadjustedNetworkClock();

        nn::time::PosixTime posix = {654321};
        SetUserClockCurrentTime(posix);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::PosixTime pt;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&pt) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        pt.value,
        654321);

    nn::time::SystemClockContext context;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&context) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// 自動補正on, ネットワーク未補正
TEST(TimeBasic, StandardUserSystemClock2)
{
    {
        SCOPED_TRACE("");
        SetUserClockAutomaticCorrectionEnabled(true);
        UnadjustedNetworkClock();

        nn::time::PosixTime posix = {123456};
        SetUserClockCurrentTime(posix);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::PosixTime pt;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&pt) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        pt.value,
        123456);

    nn::time::SystemClockContext context;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&context) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// 自動補正off, ネットワーク補正済
TEST(TimeBasic, StandardUserSystemClock3)
{
    {
        SCOPED_TRACE("");
        SetUserClockAutomaticCorrectionEnabled(false);

        {
            nn::time::PosixTime posix = {1234567};
            SetUserClockCurrentTime(posix);
        }
        {
            nn::time::PosixTime posix = {7654321};
            SetNetworkClockCurrentTime(posix);
        }
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::PosixTime pt;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&pt) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        pt.value,
        1234567);

    // 自動補正 on 後にネットワーク時刻に追従しているか
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
    SetUserClockAutomaticCorrectionEnabled(true);
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&pt) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        pt.value,
        7654321);

    nn::time::SystemClockContext context;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&context) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// 自動補正on, ネットワーク補正済
TEST(TimeBasic, StandardUserSystemClock4)
{
    {
        SCOPED_TRACE("");
        SetUserClockAutomaticCorrectionEnabled(true);

        {
            nn::time::PosixTime posix = {1234567};
            SetUserClockCurrentTime(posix);
        }
        {
            nn::time::PosixTime posix = {7654321};
            SetNetworkClockCurrentTime(posix);
        }
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::PosixTime pt;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&pt) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        pt.value,
        7654321);

    // 自動補正 off 後もネットワーク時刻を引き継いでいるか
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
    SetUserClockAutomaticCorrectionEnabled(false);
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&pt) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        pt.value,
        7654321);

    nn::time::SystemClockContext context;
    NNT_EXPECT_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&context) );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// 時刻セット、コンテキストセットの特権持ってないエラーが出るかテスト
template <typename ClockType>
void TestCapabilityErrorOfSetter(
    nn::Result(*pFuncOfSetSystemClockContext)(const nn::time::SystemClockContext&),
    nn::Result(*pFuncOfSetCurrentTime)(const nn::time::PosixTime&))
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() ); // 一般権限で初期化->セット系APIが失敗するはず

    nn::time::SystemClockContext context;
    NNT_EXPECT_RESULT_SUCCESS( ClockType::GetSystemClockContext(&context) );

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // windows だと権限チェックがないので成功する
    context.offset = 0;
    NNT_EXPECT_RESULT_SUCCESS(pFuncOfSetSystemClockContext(context) );

    nn::time::PosixTime pt = { 1234567 };
    NNT_EXPECT_RESULT_SUCCESS(pFuncOfSetCurrentTime(pt) );
#else
    context.offset = 0;
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultNoCapability, pFuncOfSetSystemClockContext(context) );

    nn::time::PosixTime pt = { 1234567 };
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultNoCapability, pFuncOfSetCurrentTime(pt) );
#endif

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, NetworkSystemClockPrivilegeCapabilityError)
{
    SCOPED_TRACE("StandardNetworkSystemClock");
    TestCapabilityErrorOfSetter<nn::time::StandardNetworkSystemClock>(
        nn::time::SetStandardNetworkSystemClockContext,
        nn::time::SetStandardNetworkSystemClockCurrentTime);
}

TEST(TimeBasic, UserSystemClockPrivilegeCapabilityError)
{
    SCOPED_TRACE("StandardUserSystemClock");

    SetUserClockAutomaticCorrectionEnabled(false); // SetStandardLocalSystemClockCurrentTime が ResultNoCapability 以外で失敗しないように

    TestCapabilityErrorOfSetter<nn::time::StandardUserSystemClock>(
        nn::time::SetStandardLocalSystemClockContext,
        nn::time::SetStandardLocalSystemClockCurrentTime);
}

// SystemClockContext を変えたときの細かい動作チェック
template <typename ClockType, int64_t CurrentClockValue, bool IsInvalidSourceIdTestRequired>
void TestSystemClockContextDetail(
    nn::Result(*pFuncOfInitialize)(),
    nn::Result(*pFuncOfSetSystemClockContext)(const nn::time::SystemClockContext&),
    nn::Result(*pFuncOfSetCurrentTime)(const nn::time::PosixTime&))
{
    {
        // ユーザ時計の自動補正off, すべての時計を CurrentClockValue に補正済でテストスタート.

        SCOPED_TRACE("");
        SetUserClockAutomaticCorrectionEnabled(false);
        {
            nn::time::PosixTime posix = {CurrentClockValue};
            SetUserClockCurrentTime(posix);
        }

        {
            nn::time::PosixTime posix = {CurrentClockValue};
            SetNetworkClockCurrentTime(posix);
        }
    }

    NNT_ASSERT_RESULT_SUCCESS( pFuncOfInitialize() );

    {
        nn::time::PosixTime pt;
        NNT_EXPECT_RESULT_SUCCESS( ClockType::GetCurrentTime(&pt) );
        NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
            pt.value,
            CurrentClockValue);
    }

    nn::time::SystemClockContext context;
    NNT_EXPECT_RESULT_SUCCESS( ClockType::GetSystemClockContext(&context) );

    // テストのためオフセットを0に
    const auto preOffset = context.offset;
    context.offset = 0;
    NNT_EXPECT_RESULT_SUCCESS( pFuncOfSetSystemClockContext(context) );

    nn::time::PosixTime pt;
    NNT_EXPECT_RESULT_SUCCESS( ClockType::GetCurrentTime(&pt) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        pt.value,
        CurrentClockValue - preOffset); // preOffset 分なくなった時刻が取得できる

    // pt+100秒をセットする
    NNT_EXPECT_RESULT_SUCCESS( pFuncOfSetCurrentTime( pt + nn::TimeSpan::FromSeconds(100) ) );

    // 現在時刻がptより100秒以上進んでいるはず
    {
        nn::time::PosixTime pt2;
        NNT_EXPECT_RESULT_SUCCESS( ClockType::GetCurrentTime(&pt2) );
        EXPECT_LE(nn::TimeSpan::FromSeconds(100), pt2 - pt);
    }

    // コンテキストオフセットを直接書き換える
    context.offset = 200;
    NNT_EXPECT_RESULT_SUCCESS( pFuncOfSetSystemClockContext(context) );

    // 内部オフセットが0から200になる
    {
        nn::time::SystemClockContext contextOut;
        NNT_EXPECT_RESULT_SUCCESS( ClockType::GetSystemClockContext(&contextOut) );
        EXPECT_EQ(context, contextOut);
    }

    // pt より200秒以上進んでいるはず
    {
        nn::time::PosixTime ptOut;
        NNT_EXPECT_RESULT_SUCCESS( ClockType::GetCurrentTime(&ptOut) );
        EXPECT_LE(nn::TimeSpan::FromSeconds(200), ptOut - pt);
    }

    // リソースIDを変えると時刻取得ができなくなる
    if(NN_STATIC_CONDITION(IsInvalidSourceIdTestRequired))
    {
        context.timeStamp.sourceId.data[0] = context.timeStamp.sourceId.data[0] + 1;    // 適当に変える
        NNT_EXPECT_RESULT_SUCCESS( pFuncOfSetSystemClockContext(context) );
        {
            nn::time::PosixTime ptOut;
            NNT_EXPECT_RESULT_FAILURE( nn::time::ResultOffsetInvalid, ClockType::GetCurrentTime(&ptOut) );
        }
    }

    // 補正されると時刻取得できるようになる
    NNT_EXPECT_RESULT_SUCCESS( pFuncOfSetCurrentTime(pt) );
    {
        nn::time::PosixTime ptOut;
        NNT_EXPECT_RESULT_SUCCESS( ClockType::GetCurrentTime(&ptOut) );
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// SystemClockContext を変えたときの細かい動作チェック
TEST(TimeBasic, StandardNetworkSystemClockOffsetCheck)
{
    const bool IsInvalidSourceIdTestRequired = true;

    SCOPED_TRACE("StandardNetworkSystemClock");
    TestSystemClockContextDetail<nn::time::StandardNetworkSystemClock, 123456789LL, IsInvalidSourceIdTestRequired>(
        nn::time::InitializeForSystem,
        nn::time::SetStandardNetworkSystemClockContext,
        nn::time::SetStandardNetworkSystemClockCurrentTime);

    TestSystemClockContextDetail<nn::time::StandardNetworkSystemClock, -123456789LL, IsInvalidSourceIdTestRequired>(
        nn::time::InitializeForSystem,
        nn::time::SetStandardNetworkSystemClockContext,
        nn::time::SetStandardNetworkSystemClockCurrentTime);

    TestSystemClockContextDetail<nn::time::StandardNetworkSystemClock, 0LL, IsInvalidSourceIdTestRequired>(
        nn::time::InitializeForSystem,
        nn::time::SetStandardNetworkSystemClockContext,
        nn::time::SetStandardNetworkSystemClockCurrentTime);
}

// SystemClockContext を変えたときの細かい動作チェック
TEST(TimeBasic, StandardUserSystemClockOffsetCheck)
{
    // StandardUserSystemClock は常に時刻を返せる時計である位置づけであり、
    // 常駐している SA やプロセスへの影響があるので、SourceId を変えて時刻が取得できなくなることのテストはしない
    const bool IsInvalidSourceIdTestRequired = false;

    SCOPED_TRACE("StandardUserSystemClock");
    TestSystemClockContextDetail<nn::time::StandardUserSystemClock, 123456789LL, IsInvalidSourceIdTestRequired>(
        nn::time::InitializeForMenu,
        nn::time::SetStandardLocalSystemClockContext,
        nn::time::SetStandardLocalSystemClockCurrentTime);

    TestSystemClockContextDetail<nn::time::StandardUserSystemClock, -123456789LL, IsInvalidSourceIdTestRequired>(
        nn::time::InitializeForMenu,
        nn::time::SetStandardLocalSystemClockContext,
        nn::time::SetStandardLocalSystemClockCurrentTime);

    TestSystemClockContextDetail<nn::time::StandardUserSystemClock, 0LL, IsInvalidSourceIdTestRequired>(
        nn::time::InitializeForMenu,
        nn::time::SetStandardLocalSystemClockContext,
        nn::time::SetStandardLocalSystemClockCurrentTime);
}

// C++標準の仕様が想定内かどうかの確認
TEST(TimeBasic, StandardTimeDefinitionSpecCheck)
{
    // sizeof は NN_STATIC_ASSERT でチェックできるが、
    // ビルドが通らないと他モジュールの開発で迷惑がかかる可能性があるので
    // ランタイムでチェックしておき、ビルドは必ず通るようにしておく.

    // NX32 は time_t が 32bit , それ以外では 64bit, であることの確認.
    {
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)

#if defined(NN_BUILD_CONFIG_ADDRESS_32)
        // time_t が 32bit の場合、(NXAddon0.4.0 の時点では) to_time_t で
        // 2038年問題が発生するのでリファレンスへの明記が必要になる.
        // nn::time にある API が std::time_t を返さなくなった(影響がなくなった)場合には、
        // time_t の bit 幅が nn::time の機能へ影響がない場合はこのテストは不要となる.
        EXPECT_EQ(4, sizeof(std::time_t));
#else
        EXPECT_EQ(8, sizeof(std::time_t));
#endif

#else
        EXPECT_EQ(8, sizeof(std::time_t));
#endif
    }

    // duration::rep の bit 幅が想定通りか
    EXPECT_EQ(8, sizeof(std::chrono::seconds::rep));

    // time_point への変換後、time_point::time_since_epoch() で 内部の epoch が意図せず環境依存でずらされていないか
    {
        {
            // SystemClockTraits::duration は秒の分解能
            auto ret = nn::time::SystemClockTraits::time_point(nn::time::SystemClockTraits::duration(12345));
            EXPECT_EQ(12345, std::chrono::duration_cast<std::chrono::seconds>( ret.time_since_epoch() ).count());
        }
        {
            // StandardUserSystemClock::duration は秒の分解能
            auto ret = nn::time::StandardUserSystemClock::time_point(nn::time::StandardUserSystemClock::duration(12345));
            EXPECT_EQ(12345, std::chrono::duration_cast<std::chrono::seconds>( ret.time_since_epoch() ).count());
            // to_time_t でも同じ値が取得できるはず
            EXPECT_EQ(12345, nn::time::StandardUserSystemClock::to_time_t(ret));
        }
        {
            // StandardNetworkSystemClock::duration は秒の分解能
            auto ret = nn::time::StandardNetworkSystemClock::time_point(nn::time::StandardNetworkSystemClock::duration(12345));
            EXPECT_EQ(12345, std::chrono::duration_cast<std::chrono::seconds>( ret.time_since_epoch() ).count());
            // to_time_t でも同じ値が取得できるはず
            EXPECT_EQ(12345, nn::time::StandardNetworkSystemClock::to_time_t(ret));
        }
    }

    // システム標準の time_t が秒単位か確認
    {
        // std::chrono::system_clock::now() から取得する time_t が秒単位か確認
        auto now = std::chrono::system_clock::now();
        // duration_cast で now を秒単位にして、 to_time と一致確認(切り上げの関係で to_time_t で1秒ずれる可能性があるので1秒以内の誤差とする
        {
            auto diff = std::abs(
                static_cast<int64_t>(std::chrono::duration_cast<std::chrono::seconds>( now.time_since_epoch() ).count()) -
                static_cast<int64_t>(std::chrono::system_clock::to_time_t(now)));
            EXPECT_LE(diff, 1);
        }

        // time(time_t* t) で取得する time_t が秒単位か確認
        std::time_t t;
        time(&t);

        // タイミングや切り上げによってずれる可能性があるので1秒以内の誤差としてテストしておく
        {
            auto diff = std::abs(
                static_cast<int64_t>(std::chrono::system_clock::to_time_t(now)) -
                static_cast<int64_t>(t));
            EXPECT_LE(diff, 1);
        }
    }
}

TEST(TimeBasic, StandardApi_StandardSteadyClock)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::StandardSteadyClock::time_point tp = nn::time::StandardSteadyClock::now();

    {
        // C++標準でないAPIで取得した値と同じかチェック
        nn::time::SteadyClockTimePoint steadyClockTimePoint;
        NNT_ASSERT_RESULT_SUCCESS( nn::time::StandardSteadyClock::GetCurrentTimePoint(&steadyClockTimePoint) );

        // IPC等で誤差が生じるため1秒以内の誤差としてテストしておく
        auto diff = std::abs(
            static_cast<int64_t>( tp.time_since_epoch().count() ) -
            static_cast<int64_t>( steadyClockTimePoint.value ));
        EXPECT_LE(diff, 1);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// 自動補正off,　ネットワーク時計補正済
TEST(TimeBasic, StandardApi_UserSystemClock1)
{
    {
        SCOPED_TRACE("");

        SetUserClockAutomaticCorrectionEnabled(false);

        {
            nn::time::PosixTime posix = {8765432};
            SetUserClockCurrentTime(posix);
        }
        {
            nn::time::PosixTime posix = {2345678};
            SetNetworkClockCurrentTime(posix);
        }
    }
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::StandardUserSystemClock::time_point tp = nn::time::StandardUserSystemClock::now();

    std::time_t stdTime = nn::time::StandardUserSystemClock::to_time_t(tp);

    nn::time::StandardUserSystemClock::time_point tp2 = nn::time::StandardUserSystemClock::from_time_t(stdTime);

    EXPECT_EQ(tp, tp2);

    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        tp.time_since_epoch().count(),
        8765432 );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// 自動補正on,　ネットワーク時計補正済
TEST(TimeBasic, StandardApi_UserSystemClock2)
{
    {
        SCOPED_TRACE("");

        SetUserClockAutomaticCorrectionEnabled(true);

        {
            nn::time::PosixTime posix = {654321};
            SetUserClockCurrentTime(posix);
        }
        {
            nn::time::PosixTime posix = {123456};
            SetNetworkClockCurrentTime(posix);
        }
    }
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::StandardUserSystemClock::time_point tp = nn::time::StandardUserSystemClock::now();

    std::time_t stdTime = nn::time::StandardUserSystemClock::to_time_t(tp);

    nn::time::StandardUserSystemClock::time_point tp2 = nn::time::StandardUserSystemClock::from_time_t(stdTime);

    EXPECT_EQ(tp, tp2);

    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        tp.time_since_epoch().count(),
        123456 );

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// ネットワーク時計 未補正
TEST(TimeBasic, StandardApi_NetworkSystemClock1)
{
    {
        SCOPED_TRACE("");
        UnadjustedNetworkClock();
    }
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::StandardNetworkSystemClock::time_point tp1 = nn::time::StandardNetworkSystemClock::now();
    EXPECT_EQ(0, tp1.time_since_epoch().count()); // 未補正だと 0 になる仕様

    std::time_t stdTime = nn::time::StandardNetworkSystemClock::to_time_t(tp1);
    nn::time::StandardNetworkSystemClock::time_point tp2 = nn::time::StandardNetworkSystemClock::from_time_t(stdTime);

    EXPECT_EQ(tp1, tp2);

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// ネットワーク時計 補正済
TEST(TimeBasic, StandardApi_NetworkSystemClock2)
{
    {
        SCOPED_TRACE("");

        nn::time::PosixTime posix = {1234567};
        SetNetworkClockCurrentTime(posix);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::StandardNetworkSystemClock::time_point tp1 = nn::time::StandardNetworkSystemClock::now();
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        tp1.time_since_epoch().count(),
        1234567);

    std::time_t stdTime = nn::time::StandardNetworkSystemClock::to_time_t(tp1);
    nn::time::StandardNetworkSystemClock::time_point tp2 = nn::time::StandardNetworkSystemClock::from_time_t(stdTime);

    EXPECT_EQ(tp1, tp2);

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

template <class SystemClockClass>
void TestSystemClockParameter(const nn::time::SystemClockContext& expectContext, int64_t posixTimeValue) NN_NOEXCEPT
{
    nn::time::SystemClockContext context;
    SystemClockClass::GetSystemClockContext(&context);
    EXPECT_EQ(expectContext, context);

    nn::time::PosixTime posix;
    NNT_ASSERT_RESULT_SUCCESS(SystemClockClass::GetCurrentTime(&posix));
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(posix.value, posixTimeValue);
}
TEST(TimeBasic, AdjustableUserSystemClock1)
{
    {
        SetUserClockAutomaticCorrectionEnabled(false);
        nn::time::PosixTime posix = {1498202629};
        SetUserClockCurrentTime(posix);
    }

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // 最初のライブラリ初期化でしか Adjust されないので、このテストでセットした 1498202629 をまず反映
        nn::time::AdjustableUserSystemClock::Adjust();

        // StandardUserSystemClock の補正が走ってない限りは StandardUserSystemClock と同じ
        nn::time::SystemClockContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetSystemClockContext(&context));
        TestSystemClockParameter<nn::time::AdjustableUserSystemClock>(context, 1498202629);
    }

    nn::time::SystemClockContext adjustableSystemClockContext;
    nn::time::AdjustableUserSystemClock::GetSystemClockContext(&adjustableSystemClockContext);

    // ユーザー時計を適当に補正
    {
        nn::time::PosixTime posix = {1498201629}; // 適当にずらす
        SetUserClockCurrentTime(posix);
    }

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // AdjustableUserSystemClock は変化ないはず
        TestSystemClockParameter<nn::time::AdjustableUserSystemClock>(adjustableSystemClockContext, 1498202629);

        // StandardUserSystemClock 側に追従
        nn::time::AdjustableUserSystemClock::Adjust();

        nn::time::SystemClockContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetSystemClockContext(&context));
        TestSystemClockParameter<nn::time::AdjustableUserSystemClock>(context, 1498201629);
    }
}

TEST(TimeBasic, AdjustableUserSystemClock2)
{
    {
        SetUserClockAutomaticCorrectionEnabled(true);
        nn::time::PosixTime posix = {1498202629};
        SetNetworkClockCurrentTime(posix);
    }

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // 最初のライブラリ初期化でしか Adjust されないので、このテストでセットした 1498202629 をまず反映
        nn::time::AdjustableUserSystemClock::Adjust();

        // StandardUserSystemClock の補正が走ってない限りは StandardUserSystemClock と同じ
        nn::time::SystemClockContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetSystemClockContext(&context));
        TestSystemClockParameter<nn::time::AdjustableUserSystemClock>(context, 1498202629);
    }

    nn::time::SystemClockContext adjustableSystemClockContext;
    nn::time::AdjustableUserSystemClock::GetSystemClockContext(&adjustableSystemClockContext);

    // ネットワーク時計を適当に補正(ユーザー時計も追従する)
    {
        nn::time::PosixTime posix = {1498204629}; // 適当にずらす
        SetNetworkClockCurrentTime(posix);
    }

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // AdjustableUserSystemClock は変化ないはず
        TestSystemClockParameter<nn::time::AdjustableUserSystemClock>(adjustableSystemClockContext, 1498202629);

        // StandardUserSystemClock 側に追従
        nn::time::AdjustableUserSystemClock::Adjust();

        nn::time::SystemClockContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetSystemClockContext(&context));
        TestSystemClockParameter<nn::time::AdjustableUserSystemClock>(context, 1498204629);
    }
}

TEST(TimeBasic, AdjustableNetworkSystemClock1)
{
    // 未補正から補正されたときのテスト

    UnadjustedNetworkClock();

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // 最初のライブラリ初期化でしか Adjust されないのでまず Adjust
        nn::time::AdjustableNetworkSystemClock::Adjust();

        // 未補正
        nn::time::PosixTime posix;
        NNT_EXPECT_RESULT_FAILURE( nn::time::ResultOffsetInvalid, nn::time::StandardNetworkSystemClock::GetCurrentTime(&posix) );
    }

    {
        nn::time::PosixTime posix = {1498101234};
        SetNetworkClockCurrentTime(posix);
    }

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // 自動的に GetCurrentTime が成功するようになってるはず
        nn::time::SystemClockContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&context));

        TestSystemClockParameter<nn::time::AdjustableNetworkSystemClock>(context, 1498101234);
    }

}

TEST(TimeBasic, AdjustableNetworkSystemClock2)
{
    {
        nn::time::PosixTime posix = {1498102629};
        SetNetworkClockCurrentTime(posix);
    }

    nn::time::SystemClockContext adjustableSystemClockContext;

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // 最初のライブラリ初期化でしか Adjust されないので、このテストでセットした 1498102629 をまず反映
        nn::time::AdjustableNetworkSystemClock::Adjust();

        // StandardNetworkSystemClock の補正が走ってない限りは StandardNetworkSystemClock と同じ
        nn::time::SystemClockContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&context));
        TestSystemClockParameter<nn::time::AdjustableNetworkSystemClock>(context, 1498102629);

        nn::time::AdjustableNetworkSystemClock::GetSystemClockContext(&adjustableSystemClockContext);
    }

    // ネットワーク時計を適当に補正(ユーザー時計も追従する)
    {
        nn::time::PosixTime posix = {1498104629}; // 適当にずらす
        SetNetworkClockCurrentTime(posix);
    }

    {
        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize()); };

        // AdjustableNetworkSystemClock は変化ないはず
        TestSystemClockParameter<nn::time::AdjustableNetworkSystemClock>(adjustableSystemClockContext, 1498102629);

        // StandardNetworkSystemClock 側に追従
        nn::time::AdjustableNetworkSystemClock::Adjust();

        nn::time::SystemClockContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&context));
        TestSystemClockParameter<nn::time::AdjustableNetworkSystemClock>(context, 1498104629);
    }
}

TEST(TimeBasic, LoadLocationNameListWithEnoughBuffer)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    const int TotalCount = nn::time::GetTotalLocationNameCount();
    EXPECT_LE(0, TotalCount);

    auto *pLocationNameList = new nn::time::LocationName[TotalCount];
    ASSERT_NE(nullptr, pLocationNameList);
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pLocationNameList;
    };

    for(int offset = 0 ; offset <= TotalCount + 10 ; ++offset)
    {
        std::memset(pLocationNameList, 0, sizeof(nn::time::LocationName) * TotalCount);

        int outCount;
        nn::time::LoadLocationNameList(&outCount, pLocationNameList, TotalCount, offset);

        if(offset >= TotalCount)
        {
            EXPECT_EQ(0, outCount);
        }
        else
        {
            EXPECT_EQ(TotalCount - offset, outCount);
        }

        // ダブりがないことを確認
        for(int i = 0 ; i < outCount ; ++i)
        {
            const auto& locationName1 = pLocationNameList[i];
            for(int j = i + 1 ; j < outCount ; ++j)
            {
                const auto& locationName2 = pLocationNameList[j];
                EXPECT_NE(locationName1, locationName2);
            }
        }

        // すべてルールが読み込めることの確認
        if(offset == 0)
        {
            static nn::time::TimeZoneRule rule;
            for(int i = 0 ; i < outCount ; ++i)
            {
                const auto& locationName = pLocationNameList[i];

                testing::Message msg;
                msg << "LocationName:" << locationName._value << "/index:" << i << "/outcount:" << outCount;
                SCOPED_TRACE(msg);

                NN_SDK_LOG("Try LoadTimeZoneRule(%s)\n", locationName._value);
                NNT_EXPECT_RESULT_SUCCESS( nn::time::LoadTimeZoneRule(&rule, locationName) );

                // ルールが正常に利用できることも確認しておく
                nn::time::PosixTime posix = {123456789};
                nn::time::CalendarTime calendar;
                NNT_EXPECT_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendar, nullptr, posix, rule) );
            }
        }
    }

    NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, LoadLocationNameListWithNotEnoughBuffer)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    const int TotalCount = nn::time::GetTotalLocationNameCount();
    EXPECT_LE(0, TotalCount);

    auto *pTotalLocationNameList = new nn::time::LocationName[TotalCount];
    int totalLocationNameListIndex = 0;
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pTotalLocationNameList;
    };

    const int ListCount = TotalCount / 10;
    auto *pLocationNameList = new nn::time::LocationName[ListCount];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pLocationNameList;
    };

    int sumLoadCount = 0;

    while(sumLoadCount < TotalCount)
    {
        int outCount;
        const int offset = sumLoadCount;
        nn::time::LoadLocationNameList(&outCount, pLocationNameList, ListCount, offset);

        sumLoadCount += outCount;
        if(sumLoadCount < TotalCount)
        {
            EXPECT_EQ(ListCount, outCount);
        }
        else
        {
            EXPECT_EQ(TotalCount, sumLoadCount);
        }

        // 今までの累積の地域名とダブりがないことを確認
        for(int totalIndex = 0 ; totalIndex < totalLocationNameListIndex ; ++totalIndex )
        {
            for(int i = 0 ; i < outCount ; ++i)
            {
                EXPECT_NE(pTotalLocationNameList[totalIndex], pLocationNameList[i]);
            }
        }

        // 累積に入れつつ、すべてルールが読み込めることの確認
        for(int i = 0 ; i < outCount ; ++i)
        {
            pTotalLocationNameList[totalLocationNameListIndex++] = pLocationNameList[i];
            static nn::time::TimeZoneRule rule;
            NNT_EXPECT_RESULT_SUCCESS( nn::time::LoadTimeZoneRule(&rule, pLocationNameList[i]) );

            // ルールが正常に利用できることも確認しておく
            nn::time::PosixTime posix = {123456789};
            nn::time::CalendarTime calendar;
            NNT_EXPECT_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendar, nullptr, posix, rule) );
        }
    }
    NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, DateTimeConvertWithAllTimeZone)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    const int TotalCount = nn::time::GetTotalLocationNameCount();
    EXPECT_LE(0, TotalCount);

    auto *pLocationNameList = new nn::time::LocationName[TotalCount];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pLocationNameList;
    };

    int outCount;
    nn::time::LoadLocationNameList(&outCount, pLocationNameList, TotalCount, 0);
    EXPECT_EQ(outCount, TotalCount);

    for(int i = 0 ; i < outCount ; ++i)
    {
        const auto TestLocationName = pLocationNameList[i];
        static nn::time::TimeZoneRule rule;
        NNT_EXPECT_RESULT_SUCCESS( nn::time::LoadTimeZoneRule(&rule, TestLocationName) );

        // 許容する PosixTime の最小値、最大値をいくつかに分割して、網羅的に ToCalendarTime(), ToPosixTime() を実行
        const nn::time::PosixTime TestSpan ={ (nn::time::InputPosixTimeMax.value + nn::time::InputPosixTimeMin.value) / 1000LL };

        for(nn::time::PosixTime posix = nn::time::InputPosixTimeMin ; posix <= nn::time::InputPosixTimeMax ; posix.value += TestSpan.value)
        {
            SCOPED_TRACE(testing::Message() << TestLocationName._value << "/Posix:" << posix.value);

            nn::time::CalendarTime calendar;
            nn::time::CalendarAdditionalInfo additional;
            NNT_EXPECT_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendar, &additional, posix, rule) );

            int stLen = nn::util::Strnlen(additional.timeZone.standardTimeName, static_cast<int>(additional.timeZone.StandardTimeNameSize));
            EXPECT_LT(0, stLen);
            EXPECT_LT(stLen, static_cast<int>(additional.timeZone.StandardTimeNameSize));

            int outPosixCount;
            nn::time::PosixTime outPosix[2];
            NNT_EXPECT_RESULT_SUCCESS( nn::time::ToPosixTime(&outPosixCount, outPosix, 2, calendar, rule) );
            EXPECT_LE(0, outPosixCount);
            EXPECT_LE(outPosixCount, 2);
        }
    }

    NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, UtcApi)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    static nn::time::TimeZoneRule rule;
    const nn::time::LocationName locationName = {"UTC"};
    NNT_ASSERT_RESULT_SUCCESS( nn::time::LoadTimeZoneRule(&rule, locationName) );

    for (int year = 1900; year <= 2100; ++year)
    {
        for (int month = 1; month <= 12; ++month)
        {
            const int DayMax = nn::time::GetDaysInMonth(year, month);
            //const int testDays[] = {1, 8, 15, 22, DayMax};
            //for (auto day : testDays)
            for(int day = 1 ; day <= DayMax ; ++day)
            {
                testing::Message msg;
                msg << year << "/" << month << "/" << day;
                SCOPED_TRACE(msg);

                const int testHours[] = {0, 23};
                for (auto hour : testHours)
                {
                    const int testMinutes[] = {0, 59};
                    for (auto minute : testMinutes)
                    {
                        const int testSeconds[] = {0, 59};
                        for(auto second : testSeconds)
                        {
                            nn::time::CalendarTime c = {static_cast<int16_t>(year), static_cast<int8_t>(month), static_cast<int8_t>(day), static_cast<int8_t>(hour), static_cast<int8_t>(minute), static_cast<int8_t>(second)};

                            nn::time::PosixTime outPosixTime;
                            int outCount;
                            NNT_ASSERT_RESULT_SUCCESS(nn::time::ToPosixTime(&outCount, &outPosixTime, 1, c, rule));
                            ASSERT_EQ(1, outCount);

                            ASSERT_EQ(outPosixTime, nn::time::ToPosixTimeFromUtc(c));

                            nn::time::CalendarTime c2 = nn::time::ToCalendarTimeInUtc(outPosixTime);
                            ASSERT_EQ(c, c2);
                        }
                    }
                }
            }
        }
    }

    NNT_EXPECT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, UtcApi2)
{
    // nn::time::Initialize() なしに利用できるかどうか

    const nn::time::CalendarTime calendar = { 1900, 1, 1, 0, 0, 0 };
    const nn::time::PosixTime posix = { -2208988800LL };

    EXPECT_EQ(posix, nn::time::ToPosixTimeFromUtc(calendar));
    EXPECT_EQ(calendar, nn::time::ToCalendarTimeInUtc(posix));
}

void TestSpec(const nn::time::TimeZoneRule& rule) NN_NOEXCEPT
{
    const auto start = nn::os::GetSystemTick();

    const int TryCount = 10000;
    nn::time::PosixTime posix = {1234567890};
    nn::time::CalendarTime c;
    for(int i = 0 ; i < TryCount ; ++i)
    {
        posix.value++;
        auto result = nn::time::ToCalendarTime(&c, nullptr, posix, rule);
        if(result.IsFailure())
        {
            NN_SDK_LOG("time::ToCalendarTime() failed. (%08x, %03d-%04d)\n",
                result.GetInnerValueForDebug(),
                result.GetModule(), result.GetDescription());
        }
    }

    const auto end = nn::os::GetSystemTick();
    NN_LOG("Rule ptr: %p\n", &rule);
    NN_LOG("ToCalendarTime total: %lld [micro sec]\n", (end - start).ToTimeSpan().GetMicroSeconds());
    NN_LOG("ToCalendarTime one call: %lld [micro sec]\n", (end - start).ToTimeSpan().GetMicroSeconds() / TryCount);
}

void TestSpec2(const nn::time::TimeZoneRule& rule) NN_NOEXCEPT
{
    const auto start = nn::os::GetSystemTick();

    const int TryCount = 10000;
    nn::time::PosixTime posix[2];
    nn::time::CalendarTime c;
    int outCount;
    c.year = 2016;
    c.month = 7;
    c.day = 15;
    c.hour = 14;
    c.minute = 28;
    c.second = 5;
    for(int i = 0 ; i < TryCount ; ++i)
    {
        auto result = nn::time::ToPosixTime(&outCount, posix, 2, c, rule);
        if(result.IsFailure())
        {
            NN_SDK_LOG("time::ToPosixTime() failed. (%08x, %03d-%04d)\n",
                result.GetInnerValueForDebug(),
                result.GetModule(), result.GetDescription());
        }
    }

    const auto end = nn::os::GetSystemTick();
    NN_LOG("Rule ptr: %p\n", &rule);
    NN_LOG("ToPosixTime total: %lld [micro sec]\n", (end - start).ToTimeSpan().GetMicroSeconds());
    NN_LOG("ToPosixTime one call: %lld [micro sec]\n", (end - start).ToTimeSpan().GetMicroSeconds() / TryCount);
}

TEST(TimeBasic, SpecCheck1)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );

    static nn::time::TimeZoneRule rule;
    nn::time::LocationName location = {"America/Los_Angeles"};
    NNT_ASSERT_RESULT_SUCCESS(nn::time::LoadTimeZoneRule(&rule, location));

    TestSpec(rule);
    TestSpec2(rule);

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, SpecCheck2)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForMenu() );

    static char Buffer[sizeof(nn::time::TimeZoneRule) * 2] = {};
    NN_LOG("Buffer: %p\n", Buffer);

    intptr_t p = reinterpret_cast<intptr_t>(Buffer);
    nn::time::TimeZoneRule* pRule =
        reinterpret_cast<nn::time::TimeZoneRule*>( nn::util::align_up(p, 0x1000) );

    nn::time::LocationName location = {"America/Los_Angeles"};
    NNT_ASSERT_RESULT_SUCCESS(nn::time::LoadTimeZoneRule(pRule, location));

    TestSpec(*pRule);
    TestSpec2(*pRule);

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// (SIGLO-38255) tz database tag 2016g 取り込み確認
TEST(TimeBasic, TzTag2016g)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    struct TestCase
    {
        nn::time::LocationName locatoinName;
        int32_t expectOffsetSeconds;
    };

    const TestCase testCase[] =
    {
        {{"America/Caracas"}, -14400},  // UTC -4:00  -> 2016/5月更新
        {{"Asia/Magadan"}, 39600},      // UTC +11:00 -> 2016/4月更新
    };

    const nn::time::CalendarTime checkCalendarTime = {2016, 10, 1, 0, 0, 0};

    static nn::time::TimeZoneRule rule;
    for(auto test : testCase)
    {
        testing::Message msg;
        msg << "LocationName:" << test.locatoinName._value;
        SCOPED_TRACE(msg);

        NNT_ASSERT_RESULT_SUCCESS(nn::time::LoadTimeZoneRule(&rule, test.locatoinName));

        nn::time::PosixTime checkPosixTime;
        int out;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::ToPosixTime(&out, &checkPosixTime, 1, checkCalendarTime, rule));
        EXPECT_EQ(1, out);

        nn::time::CalendarTime c;
        nn::time::CalendarAdditionalInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::ToCalendarTime(&c, &info, checkPosixTime, rule));
        EXPECT_EQ(test.expectOffsetSeconds, info.timeZone.utcOffsetSeconds);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, InvalidLocationName)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    static nn::time::TimeZoneRule rule;
    nn::time::LocationName name = {"invalid"}; // 存在しない

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Windows は fs 使ってないので nn::time 独自エラー
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultNotFound,  nn::time::LoadTimeZoneRule(&rule, name));
#else
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound,  nn::time::LoadTimeZoneRule(&rule, name));
#endif

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}


template <
        int typecnt,
        int timecnt,
        int charcnt,
        unsigned char fillTypesValue,
        int fillAbbreviationIndexValue>
void OutOfRangeTimeZoneRuleTest()
{
    static nn::time::TimeZoneRule rule;
    std::memset(&rule, 0xFF, sizeof(rule));

    nn::timesrv::TimeZoneRuleInner* pTimeZoneRuleInner = reinterpret_cast<nn::timesrv::TimeZoneRuleInner*>(&rule);
    pTimeZoneRuleInner->typecnt = typecnt;
    pTimeZoneRuleInner->timecnt = timecnt;
    pTimeZoneRuleInner->charcnt = charcnt;
    for (int i = 0 ; i < nn::timesrv::TimesCountMax ; ++i)
    {
        pTimeZoneRuleInner->types[i] = fillTypesValue;
    }
    for (int i = 0 ; i < nn::timesrv::TypesCountMax ; ++i)
    {
        pTimeZoneRuleInner->ttis[i].tt_abbrind = fillAbbreviationIndexValue;
    }

    {
        const nn::time::PosixTime posix = {123456789};
        nn::time::CalendarTime calendar;
        NNT_EXPECT_RESULT_FAILURE(nn::time::ResultArgumentOutOfRange, nn::time::ToCalendarTime(&calendar, nullptr, posix, rule));
    }

    {
        const nn::time::CalendarTime calendar = {2016, 10, 1, 2, 3, 4};
        nn::time::PosixTime posix;
        int outCount;
        NNT_EXPECT_RESULT_FAILURE(nn::time::ResultArgumentOutOfRange, nn::time::ToPosixTime(&outCount, &posix, 1, calendar, rule));
    }
};
TEST(TimeBasic, OutOfRangeTimeZoneRule)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax + 1, nn::timesrv::TimesCountMax, nn::timesrv::CharsCountMax, nn::timesrv::TypesCountMax - 1, nn::timesrv::CharsCountMax - 1>();
    OutOfRangeTimeZoneRuleTest<                             0, nn::timesrv::TimesCountMax, nn::timesrv::CharsCountMax, nn::timesrv::TypesCountMax - 1, nn::timesrv::CharsCountMax - 1>();

    OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax, nn::timesrv::TimesCountMax + 1, nn::timesrv::CharsCountMax, nn::timesrv::TypesCountMax - 1, nn::timesrv::CharsCountMax - 1>();
    OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax,                              0, nn::timesrv::CharsCountMax, nn::timesrv::TypesCountMax - 1, nn::timesrv::CharsCountMax - 1>();

    OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax, nn::timesrv::TimesCountMax, nn::timesrv::CharsCountMax + 1, nn::timesrv::TypesCountMax - 1, nn::timesrv::CharsCountMax - 1>();
    OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax, nn::timesrv::TimesCountMax,                              0, nn::timesrv::TypesCountMax - 1, nn::timesrv::CharsCountMax - 1>();

    OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax, nn::timesrv::TimesCountMax, nn::timesrv::CharsCountMax, nn::timesrv::TypesCountMax, nn::timesrv::CharsCountMax - 1>();
    // fillTypesValue は unsigned なので -1 はテストなし
    //OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax, nn::timesrv::TimesCountMax, nn::timesrv::CharsCountMax,                         -1, nn::timesrv::CharsCountMax - 1>();

    OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax, nn::timesrv::TimesCountMax, nn::timesrv::CharsCountMax, nn::timesrv::TypesCountMax - 1, nn::timesrv::CharsCountMax>();
    OutOfRangeTimeZoneRuleTest<nn::timesrv::TypesCountMax, nn::timesrv::TimesCountMax, nn::timesrv::CharsCountMax, nn::timesrv::TypesCountMax - 1,                         -1>();

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, InvalidTimeZoneRule)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    static nn::time::TimeZoneRule rule;

    {
        // ToCalendarTime, ToPosixTime でパラメータチェックしているぎりぎり範囲内
        std::memset(&rule, 0xFF, sizeof(rule));
        nn::timesrv::TimeZoneRuleInner* pTimeZoneRuleInner = reinterpret_cast<nn::timesrv::TimeZoneRuleInner*>(&rule);
        pTimeZoneRuleInner->typecnt = nn::timesrv::TypesCountMax;
        pTimeZoneRuleInner->timecnt = nn::timesrv::TimesCountMax;
        pTimeZoneRuleInner->charcnt = nn::timesrv::CharsCountMax;
        for (int i = 0 ; i < pTimeZoneRuleInner->timecnt ; ++i)
        {
            pTimeZoneRuleInner->types[i] = nn::timesrv::TypesCountMax - 1;
        }
        for (int i = 0 ; i <= pTimeZoneRuleInner->typecnt ; ++i)
        {
            pTimeZoneRuleInner->ttis[i].tt_abbrind = nn::timesrv::CharsCountMax - 1;
        }

        // 他のパラメータによってオーバーフローなど発生しうるので成否は問わない
        {
            const nn::time::PosixTime posix = {123456789};
            nn::time::CalendarTime calendar;
            nn::time::ToCalendarTime(&calendar, nullptr, posix, rule);
        }
        {
            const nn::time::CalendarTime calendar = {2016, 10, 1, 2, 3, 4};
            nn::time::PosixTime posix;
            int outCount;
            nn::time::ToPosixTime(&outCount, &posix, 1, calendar, rule);
        }
    }

    // ランダム生成の TimeZoneRule でシステムが止まらないことの確認
    for(int i = 0 ; i < 10000 ; ++i)
    {
        nn::os::GenerateRandomBytes(&rule, sizeof(rule));
        {
            const nn::time::PosixTime posix = {123456789};
            nn::time::CalendarTime calendar;
            nn::time::ToCalendarTime(&calendar, nullptr, posix, rule);
        }

        {
            const nn::time::CalendarTime calendar = {2016, 10, 1, 2, 3, 4};
            nn::time::PosixTime posix;
            int outCount;
            nn::time::ToPosixTime(&outCount, &posix, 1, calendar, rule);
        }
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, ConvertToNetworkPosixTime)
{
    // ドキュメント記載用の関数のテスト
    // TODO nn::time:: にAPIが用意できたら関数を差し替える(SIGLO-21922)

    /**
     * @brief   SteadyClockTimePoint をネットワーク時計の時刻へ変換します。
     * @param[out]  pOutPosixTime           ネットワーク時計の時刻の出力
     * @param[in]   steadyClockTimePoint    変換元となる単調増加クロックの値
     * @return  処理結果
     * @retval  true        成功
     * @retval  false       失敗
     * @details
     *  ネットワーク時計の状態によって、同一の SteadyClockTimePoint から異なる PosixTime が得られる可能性があります。
     */
    auto ConvertToNetworkPosixTime = [](nn::time::PosixTime* pOutPosixTime, const nn::time::SteadyClockTimePoint& steadyClockTimePoint) -> bool
    {
        nn::time::SteadyClockTimePoint currentTimePoint;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&currentTimePoint));

        int64_t elapsedSeconds;
        auto result = nn::time::GetSpanBetween(&elapsedSeconds, steadyClockTimePoint, currentTimePoint);
        if(result.IsFailure())
        {
            // 既にStandardSteadyClockリセットが発生している。入力の steadyClockTimePoint では永遠に成功することがない。
            return false;
        }

        nn::time::PosixTime currentPosix;
        result = nn::time::StandardNetworkSystemClock::GetCurrentTime(&currentPosix);
        if(result.IsFailure())
        {
            // ネットワーク時計が補正されていない。後にネットワーク時計が補正されることで、成功を返せる可能性がある。
            return false;
        }

        *pOutPosixTime = currentPosix - nn::TimeSpan::FromSeconds(elapsedSeconds);
        return true;
    };

    {
        UnadjustedNetworkClock();

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() ); };

        nn::time::SteadyClockTimePoint timePoint;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&timePoint));

        nn::time::PosixTime posixTime;
        EXPECT_FALSE(ConvertToNetworkPosixTime(&posixTime, timePoint));

        timePoint.sourceId = nn::util::InvalidUuid;
        EXPECT_FALSE(ConvertToNetworkPosixTime(&posixTime, timePoint));
    }

    {
        nn::time::PosixTime setPosixTime = {1234567};
        SetNetworkClockCurrentTime(setPosixTime);

        NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
        NN_UTIL_SCOPE_EXIT{ NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() ); };

        nn::time::SteadyClockTimePoint timePoint;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&timePoint));

        nn::time::PosixTime posixTime;
        EXPECT_TRUE(ConvertToNetworkPosixTime(&posixTime, timePoint));
        NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(posixTime.value, setPosixTime.value);

        timePoint += nn::TimeSpan::FromSeconds(100);
        EXPECT_TRUE(ConvertToNetworkPosixTime(&posixTime, timePoint));
        NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(posixTime.value, setPosixTime.value + 100);

        timePoint -= nn::TimeSpan::FromSeconds(200);
        EXPECT_TRUE(ConvertToNetworkPosixTime(&posixTime, timePoint));
        NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(posixTime.value, setPosixTime.value - 100);

        timePoint.sourceId.data[0]++;
        EXPECT_FALSE(ConvertToNetworkPosixTime(&posixTime, timePoint));
    }
}

TEST(TimeBasic, CompareSystemClockContext)
{
    // nn::time:: にAPIが用意できたら関数を差し替えるが、その予定は現状なし
    // テストだけは残しておく。

    /**
     * @brief   時刻補正時の補正幅を考慮して、2つの SystemClockContext が等しいとみなせるかを判定します。
     * @param[in]   lhs     比較するコンテキストの一方を指定します。
     * @param[in]   rhs     比較するコンテキストの他方を指定します。
     * @return  処理結果
     * @retval  true        一致。時計が操作されていない、または操作されていないとみなせる。
     * @retval  false       不一致。時計に対して補正幅の大きい操作が行われた。
     */
    auto CompareSystemClockContext = [](const nn::time::SystemClockContext& lhs, const nn::time::SystemClockContext&rhs) -> bool
    {
        if(lhs == rhs)
        {
            return true;
        }

        // 以降、lhs != rhs であるが、補正幅が十分に小さいかを検証し、十分に小さければ true を返します。
        // SystemClockContext::timeStamp は、時計が補正された瞬間の nn::time::SteadyClockTimePoint です。
        // lhs と rhs の補正間隔(経過時間)に対し、時刻の補正幅が十分に小さいかを計算します。

        // lhsの補正処理とrhsの補正処理の間の経過時間
        int64_t elapsedSeconds;
        auto result = nn::time::GetSpanBetween(&elapsedSeconds, lhs.timeStamp, rhs.timeStamp);
        if(result.IsFailure())
        {
            // 2つのコンテキストの取得が、 nn::time::StandardSteadyClock のリセットをまたいでおり不一致
            return false;
        }
        elapsedSeconds = std::abs(elapsedSeconds);

        // 補正処理の間隔が 0 であることは基本的にないがはじく
        if(elapsedSeconds == 0)
        {
            return false;
        }

        // 2つのコンテキスト間の時刻の補正幅
        auto correctionDiff = std::abs(rhs.offset - lhs.offset);

        // 経過時間における補正幅の割合
        auto rate = static_cast<double>(correctionDiff) / static_cast<double>(elapsedSeconds);

        // StandardSteadyClock のカウントアップは、実際の時間の進みと比べて月に数分程度の誤差が累積する可能性があります。
        // StandardUserSystemClock, StandardNetworkSystemClock は、StandardSteadyClock のカウントアップを利用して時間を進めているため、
        // 上述の累積誤差がそれらクロックの精度にも影響しますが、サーバーから本来の時刻で補正されるたびに累積誤差がクリアされます。

        // 許容する最大の補正幅割合。 ひと月あたり3分未満の補正幅の割合であれば十分に小さいとしています。
        const auto ToleranceRate = static_cast<double>(nn::TimeSpan::FromMinutes(3).GetMinutes()) / static_cast<double>(nn::TimeSpan::FromDays(30).GetMinutes());

        return rate < ToleranceRate;
    };

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

    nn::time::SteadyClockTimePoint timePoint;
    NNT_ASSERT_RESULT_SUCCESS( nn::time::StandardSteadyClock::GetCurrentTimePoint(&timePoint) );

    {
        // 同一コンテキスト
        nn::time::SystemClockContext context1 = { 100, timePoint };
        nn::time::SystemClockContext context2 = { 100, timePoint };
        EXPECT_TRUE(CompareSystemClockContext(context1, context2));
        EXPECT_TRUE(CompareSystemClockContext(context2, context1));
    }
    {
        // 補正時刻が同一、でもオフセットが異なる(基本的にあり得ない)
        nn::time::SystemClockContext context1 = { 100, timePoint };
        nn::time::SystemClockContext context2 = { 101, timePoint };
        EXPECT_FALSE(CompareSystemClockContext(context1, context2));
        EXPECT_FALSE(CompareSystemClockContext(context2, context1));
    }
    {
        // 30日で2分59秒ずらした -> ギリギリtrue
        nn::time::SystemClockContext context1 = { 100, timePoint };
        nn::time::SystemClockContext context2 = { (100 + nn::TimeSpan::FromSeconds(179).GetSeconds()), (timePoint + nn::TimeSpan::FromDays(30)) };
        EXPECT_TRUE(CompareSystemClockContext(context1, context2));
        EXPECT_TRUE(CompareSystemClockContext(context2, context1));

        // SourceId違い
        context1.timeStamp.sourceId.data[0]++;
        EXPECT_FALSE(CompareSystemClockContext(context1, context2));
        EXPECT_FALSE(CompareSystemClockContext(context2, context1));
    }
    {
        // 30日で3分ずらした -> ギリギリfalse
        nn::time::SystemClockContext context1 = { 100, timePoint };
        nn::time::SystemClockContext context2 = { (100 + nn::TimeSpan::FromMinutes(3).GetSeconds()), (timePoint + nn::TimeSpan::FromDays(30)) };
        EXPECT_FALSE(CompareSystemClockContext(context1, context2));
        EXPECT_FALSE(CompareSystemClockContext(context2, context1));
    }
    {
        // 30日で5分ずらした -> false
        nn::time::SystemClockContext context1 = { 100, timePoint };
        nn::time::SystemClockContext context2 = { (100 + nn::TimeSpan::FromMinutes(5).GetSeconds()), (timePoint + nn::TimeSpan::FromDays(30)) };
        EXPECT_FALSE(CompareSystemClockContext(context1, context2));
        EXPECT_FALSE(CompareSystemClockContext(context2, context1));
    }
    {
        // 60日で5分ずらした -> true
        nn::time::SystemClockContext context1 = { 100, timePoint };
        nn::time::SystemClockContext context2 = { (100 + nn::TimeSpan::FromMinutes(5).GetSeconds()), (timePoint + nn::TimeSpan::FromDays(60)) };
        EXPECT_TRUE(CompareSystemClockContext(context1, context2));
        EXPECT_TRUE(CompareSystemClockContext(context2, context1));

        // SourceId違い
        context1.timeStamp.sourceId.data[0]++;
        EXPECT_FALSE(CompareSystemClockContext(context1, context2));
        EXPECT_FALSE(CompareSystemClockContext(context2, context1));
    }
}

TEST(TimeBasic, TimeZoneRuleVersion)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::TimeZoneRuleVersion version;
    nn::time::GetTimeZoneRuleVersion(&version);
    NN_LOG("TimeZoneRuleVersion:%s\n", version.data);

    char* pEnd = nullptr;
    uint64_t value = std::strtoull(version.data, &pEnd, 10);
    ASSERT_EQ(*pEnd, '\0');

    // "171215" 以上を期待
    EXPECT_LE(171215llu, value);

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, StandardUserSystemClockInitialYear)
{
    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    int year = nn::time::GetStandardUserSystemClockInitialYear();
    EXPECT_LE(2018, year);

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

namespace nn { namespace time {
extern Result GetSpanFromCorrection(int64_t* pOutSeconds, const SystemClockContext& systemClockContext) NN_NOEXCEPT;
}}
TEST(TimeBasic, GetSpanFromCorrection)
{
    {
        SetUserClockAutomaticCorrectionEnabled(false);

        nn::time::PosixTime posix = {654321};
        SetUserClockCurrentTime(posix);
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );

    nn::time::SystemClockContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetSystemClockContext(&context));


    for(int64_t i = 1LL ; i < 5LL ; ++i)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        int64_t outSeconds;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::GetSpanFromCorrection(&outSeconds, context));
        EXPECT_TRUE( (outSeconds - i) <= 1LL ) << "outSeconds:" << outSeconds;
    }

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

TEST(TimeBasic, NetworkSystemClockAccuracy)
{
    {
        nn::time::PosixTime initialTime = {1234567890};
        SetNetworkClockCurrentTime(initialTime);
    }

#if defined (NN_BUILD_CONFIG_OS_WIN)
    const nn::TimeSpan Threshold = nn::TimeSpan::FromDays(30);
#else
    uint32_t minutes;
    GetTimeFwdbgSettings(&minutes, "standard_network_clock_sufficient_accuracy_minutes");
    const nn::TimeSpan Threshold = nn::TimeSpan::FromMinutes(static_cast<int64_t>(minutes));
#endif

    NNT_ASSERT_RESULT_SUCCESS( nn::time::InitializeForSystem() );

    EXPECT_TRUE(nn::time::StandardNetworkSystemClock::IsAccuracySufficient()); // 補正したばかり

    nn::time::SystemClockContext context;
    nn::time::StandardNetworkSystemClock::GetSystemClockContext(&context);

    // ぎりぎり精度があるはず
    context.timeStamp -= (Threshold - nn::TimeSpan::FromMinutes(1)); // 補正時刻を過去に
    NNT_ASSERT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockContext(context) );
    EXPECT_TRUE(nn::time::StandardNetworkSystemClock::IsAccuracySufficient());

    // ぎりぎり精度がない判定になるはず
    context.timeStamp -= nn::TimeSpan::FromMinutes(1); // 補正時刻を過去に
    NNT_ASSERT_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockContext(context) );
    EXPECT_FALSE(nn::time::StandardNetworkSystemClock::IsAccuracySufficient());

    {
        nn::time::PosixTime initialTime = {1234567890};
        SetNetworkClockCurrentTime(initialTime);
    }
    EXPECT_TRUE(nn::time::StandardNetworkSystemClock::IsAccuracySufficient()); // 補正したばかり

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Finalize() );
}

// SIGLO-31317 の修正確認テスト
// このテストは一番最後に配置してください
TEST(TimeBasic, NoFinalizeBugFix)
{

    NNT_ASSERT_RESULT_SUCCESS( nn::time::Initialize() );
    // Finailze せずアプリ終了
}
