﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>

#include <cstdlib>
#include <cstring>

#include <nn/tics/tics_Api.h>

#include "TCPSlot.h"
#include "CrossBar.h"
#include "MuxDemux.h"
#include "MuxDemuxSlot.h"

#include "../testHtc_CrossBarRunThread.h"
#include "testHtc_EchoClientPlug.h"

extern "C" void nninitStartup()
{
}

/**
    HTC ライブラリと HotBridgePCDeamon を経由して、
    エコーサーバにアクセスするテスト。
*/
TEST(muxEchoClient, multiChannel)
{
    // メモリヒープ確保
    size_t heapSize = 16 * 1024 * 1024; // 16MB
    nn::Result result = nn::os::SetMemoryHeapSize( heapSize );
    ASSERT_TRUE( result.IsSuccess() );

    uintptr_t heapAddress;
    result = nn::os::AllocateMemoryBlock( &heapAddress, heapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // HTC ライブラリ初期化
    nn::tics::Initialize(heapAddress, heapSize);

    // 文字列定数
    const ::tics::portability::stl::string tcpSessionId("tcpserver");
    const ::tics::portability::stl::string tcpSlotId("tcp");
    const ::tics::portability::stl::string tcpHost("localhost");
    const ::tics::portability::stl::string tcpPort("6003");

    const ::tics::portability::stl::string muxSessionId("mux");

    const size_t channnelCount = 2;

    const ::tics::portability::stl::string testData[channnelCount] =
    {
        ::tics::portability::stl::string("ABCDEFGHABCDEFGH"),
        ::tics::portability::stl::string("1234567812345678")
    };

    const ::tics::portability::stl::string testSessionIds[channnelCount] =
    {
        ::tics::portability::stl::string("mux:test1"),
        ::tics::portability::stl::string("mux:test2")
    };

    // 別スレッドとの待ち合わせ用イベント
    nn::os::EventType events[channnelCount];
    for ( int i = 0; i < channnelCount; i++ )
    {
        nn::os::InitializeEvent( &(events[i]), false, nn::os::EventClearMode_AutoClear );
    }

    // 1 TcpSlot, 1 MuxDemux, N MuxDemux Slot を持つ CrossBar を作成
    ::tics::CrossBar* pCrossBar = new ::tics::CrossBar();

    ::tics::TCPSlot* pTcpSlot = new ::tics::TCPSlot(tcpSlotId, "bridge_client", tcpHost, tcpPort); // intType を空欄にして、サーバモードに
    pCrossBar->AddSlot(tcpSessionId, pTcpSlot);

    ::tics::MuxDemux* pMuxDemux = new ::tics::MuxDemux(pCrossBar, muxSessionId);
    pCrossBar->AddMuxDemux(muxSessionId, pMuxDemux);

    for ( int i = 0; i < channnelCount; i++ )
    {
        int channelId = i + 1;
        int channelPriority = 1;

        ::tics::MuxDemuxSlot* pEchoSlot = new ::tics::MuxDemuxSlot(testSessionIds[i], testSessionIds[i], channelId, channelPriority);
        pCrossBar->AddSlot(testSessionIds[i], pEchoSlot);
    }

    // MuxDemux の Plug を TcpSlot に登録
    pCrossBar->RegisterForSessionStart(tcpSessionId, pMuxDemux->GetPlug());

    // EchoClientPlug を登録
    ::EchoClientPlug* pEchoClientPlugs[channnelCount];
    for ( int i = 0; i < channnelCount; i++ )
    {
        pEchoClientPlugs[i] = new ::EchoClientPlug( &events[i] );
        pEchoClientPlugs[i]->SetSendData(reinterpret_cast<const void*>(testData[i].c_str()), testData[i].size() + 1);
        pCrossBar->RegisterForSessionStart(testSessionIds[i], pEchoClientPlugs[i]);
    }

    // CrossBar::Run() を別スレッドで呼び出し
    CrossBarRunThread* pCrossBarRunThread = new CrossBarRunThread( pCrossBar );
    pCrossBarRunThread->StartThread();

    // エコーサーバとの送受信の終了を待つ
    for ( int i = 0; i < channnelCount; i++ )
    {
        nn::os::WaitEvent( &events[i] );
    }
    pCrossBar->BeginShutdown(::tics::CrossBar::ExitDone);
    pCrossBarRunThread->WaitThread();

    // 受信データの検査。相手がエコーサーバのため、送信/受信データは等しくなるはず。
    for ( int i = 0; i < channnelCount; i++ )
    {
        const char* pRecvData = reinterpret_cast<const char*>( pEchoClientPlugs[i]->GetRecvData() );
        ASSERT_STREQ(testData[i].c_str(), pRecvData);
    }

    nn::tics::Finalize();
}
