﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <Tasks/OsTimerTask.h>
#include <FileUtility.h>

namespace nnt { namespace abuse {
    OsTimerTask::OsTimerTask(const String& typeName, const String& instanceName) :
        BaseTask(typeName, instanceName),
        m_threadArgs(nullptr),
        m_threads(nullptr),
        m_stacks(nullptr),
        m_timers(nullptr),
        m_mutex(nullptr),
        m_maxThreads(64),
        m_maxGroups(64),
        m_numGroups(1),
        m_numThreads(1),
        m_numActualThreads(0),
        m_priority(nn::os::DefaultThreadPriority),
        m_threadIterations(10),
        m_runCounter(0),
        m_minWait(1),
        m_maxWait(2)
    {}

    OsTimerTask::~OsTimerTask()
    {}

    InitStatus OsTimerTask::Initialize(const String& params)
    {
        ArgVector args;
        FileUtility::ParamsToArgs(params, args);

        for(ScriptArg& arg : args)
        {
            if(arg.argName == "MaxThreads")
                FileUtility::TryParseInt(arg, 1, 64, &m_maxThreads);
            else if(arg.argName == "MaxGroups")
                FileUtility::TryParseInt(arg, 1, 64, &m_maxGroups);
            else if(arg.argName == "Priority")
                FileUtility::TryParseInt(arg, nn::os::HighestThreadPriority, nn::os::LowestThreadPriority, &m_priority);
            else if(arg.argName == "NumThreadIterations")
                FileUtility::TryParseInt(arg, 1, std::numeric_limits<int>::max(), &m_threadIterations);
            else if(arg.argName == "MinWaitMilli")
                FileUtility::TryParseInt(arg, 1, std::numeric_limits<int>::max(), &m_minWait);
            else if(arg.argName == "MaxWaitMilli")
                FileUtility::TryParseInt(arg, 1, std::numeric_limits<int>::max(), &m_maxWait);
        }

        nn::os::InitializeMutex(&m_waitMutex, false, 0);
        m_threads = (nn::os::ThreadType*) Platform::Allocate(sizeof(nn::os::ThreadType) * m_maxThreads);

        if(m_minWait > m_maxWait)
        {
            LogWarning("minWait(%d) greater than maxWait(%d). Swapping them.\n", m_minWait, m_maxWait);
            std::swap(m_minWait, m_maxWait);
        }

        if(!m_threads)
        {
            LogError("Init Error - could not allocate threads\n");
            return INIT_ERROR;
        }

        for(int i = 0; i < m_maxThreads; ++i)
            ::new (&m_threads[i]) nn::os::ThreadType;

        LogInfo("Created thread array at %p\n", m_threads);
        m_stacks = (void**)Platform::Allocate(sizeof(void*) * m_maxThreads);

        if(!m_stacks)
        {
            LogError("Init Error - could not allocate stack array\n");
            Platform::Free(m_threads);
            m_threads = nullptr;
            return INIT_ERROR;
        }
        int i = 0;
        for(; i < m_maxThreads; ++i)
        {
            m_stacks[i] = Platform::AllocateAligned(4096, nn::os::StackRegionAlignment);

            if(!m_stacks[i])
                break;
        }

        if(i != m_maxThreads)
        {
            LogError("Init Error - could not allocate thread stacks\n");
            for(int j = 0; j <i; ++j)
            {
                Platform::Free(m_stacks[j]);
            }
            Platform::Free(m_stacks);
            Platform::Free(m_threads);
            m_threads = nullptr;
            m_stacks = nullptr;
            return INIT_ERROR;
        }

        m_threadArgs = (TimerThreadArg*)Platform::Allocate( m_maxThreads * sizeof(TimerThreadArg));

        if(!m_threadArgs)
            return INIT_ERROR;

        m_mutex = (nn::os::MutexType*) Platform::Allocate(m_maxGroups * sizeof(nn::os::MutexType));

        if(!m_mutex)
            return INIT_ERROR;

        m_timers = (nn::os::TimerEventType*) Platform::Allocate(m_maxGroups * sizeof(nn::os::TimerEventType));

        if(!m_timers)
            return INIT_ERROR;

        return INIT_OK;
    }

    StartStatus OsTimerTask::Start()
    {
        m_numGroups *= 2;
        if(m_numGroups > m_maxGroups || m_numGroups > m_numThreads)
        {
            m_numGroups = 1;

            m_numThreads *= 2;
            if(m_numThreads > m_maxThreads)
                m_numThreads = 1;
        }

        LogInfo("Starting %d threads in %d groups\n", m_numThreads, m_numGroups);

        return START_OK;
    }

    RunStatus OsTimerTask::Run()
    {
        ++m_runCounter;
        LogVerbose("++Run%d - %s\n", m_runCounter, m_runCounter % 4 >= 2 ? "AutoClear" : "ManualClear");

        for(int i = 0; i < m_numGroups; ++i)
        {
            nn::os::InitializeMutex(&m_mutex[i], 0, false);

            if(m_runCounter % 2 == 1)
                nn::os::InitializeTimerEvent(&m_timers[i], nn::os::EventClearMode_AutoClear);
            else
                nn::os::InitializeTimerEvent(&m_timers[i], nn::os::EventClearMode_ManualClear);

            int time = Abuse::RandRange(m_minWait, m_maxWait);

            if(i % 2 == 0)
            {
                nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds( time );
                nn::os::StartPeriodicTimerEvent(&m_timers[i], interval, interval);
            }
            else
            {
                time*=m_threadIterations;
                nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds( time );
                nn::os::StartOneShotTimerEvent(&m_timers[i], interval);
            }

            m_threadArgs[i % m_numGroups].interval = time;
            LogInfo("timer %p interval=%d\n",&m_timers[i], time);
        }

        m_numActualThreads = 0;
        for(int i = 0; i < m_numThreads; ++i)
        {
            int group = i % m_numGroups;
            m_threadArgs[i].task = this;
            m_threadArgs[i].group = group;
            m_threadArgs[i].numItr = 1;//group % 2 == 1 ? 1 : m_threadIterations;
            m_threadArgs[i].autoClear = m_runCounter % 2 == 1;
            m_threadArgs[i].signal = m_runCounter % 4 >= 2;
            if(! CreateThread(&m_threads[i], workerFunc, &m_threadArgs[i], m_stacks[i], 4096, m_priority) )
                break;
            ++m_numActualThreads;
        }

        if(m_numActualThreads != m_numThreads)
        {
            LogWarning("Could not create %d threads. Continuing this run with %d threads instead.\n", m_numThreads, m_numActualThreads);
        }

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::StartThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::WaitThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
        {
            DestroyThread(&m_threads[i]);
        }

        for(int i = 0; i < m_numGroups; ++i)
        {
            nn::os::FinalizeMutex(&m_mutex[i]);
            nn::os::FinalizeTimerEvent(&m_timers[i]);
        }
        LogVerbose("--Run\n");
        return RUN_OK;
    }

    StopStatus OsTimerTask::Stop()
    {
        return STOP_OK;
    }

    ShutdownStatus OsTimerTask::Shutdown()
    {
        nn::os::FinalizeMutex(&m_waitMutex);

        if(m_threads)
            Platform::Free(m_threads);

        if(m_stacks)
        {
            for(int i = 0; i < m_maxThreads; ++i)
                Platform::Free(m_stacks[i]);
            Platform::Free(m_stacks);
        }

        if(m_threadArgs)
            Platform::Free(m_threadArgs);

        if(m_mutex)
            Platform::Free(m_mutex);

        if(m_timers)
        {
            Platform::Free(m_timers);
        }
        return SHUTDOWN_OK;
    }

    const char* OsTimerTask::GetParamOptions()
    {
        return "MaxThreads=[1 - 64, default = 64] - maximum number of threads to run\n\
\tMaxGroups=[1 - 64, default = 64] - maximum number of groups\n\
\tPriority=[0 - 31, default = 16] - priority of threads\n\
\tNumThreadIterations=[1+, default = 10] - number of iterations each thread runs before returning\n\
\tMinWaitMilli=[1+, default = 1] - minimum number of milliseconds to wait for TimedWaitEvent\n\
\tMaxWaitMilli=[1+, default = 2] - maximum number of milliseconds to wait.";
    }

    void OsTimerTask::workerFunc(void* vTimerThreadArg)
    {
        TimerThreadArg* arg = (TimerThreadArg*)vTimerThreadArg;
        nn::os::TimerEventType* timer = &arg->task->m_timers[arg->group];
        int numItr = arg->numItr;
        for(int i = 0; i < numItr; ++i)
        {
            nn::os::LockMutex(&arg->task->m_waitMutex);
            arg->task->LogVerbose("i: %d, signal: %d, autoClear: %d\n",
                i, arg->signal, arg->autoClear);

            if(arg->signal)
                nn::os::SignalTimerEvent(timer);

            if(i % 2 == 0)
                nn::os::WaitTimerEvent(timer);
            else
            {
                while(!nn::os::TryWaitTimerEvent(timer))
                    nn::os::YieldThread();
            }

            if(!arg->autoClear)
                lockedClear(arg);

            nn::os::UnlockMutex(&arg->task->m_waitMutex);
        }
    }

    void OsTimerTask::lockedClear(TimerThreadArg* arg)
    {
        nn::os::MutexType* mutex = &arg->task->m_mutex[arg->group];
        nn::os::LockMutex(mutex);

        nn::os::ClearTimerEvent(&arg->task->m_timers[arg->group]);

        nn::os::UnlockMutex(mutex);
    }

}    }
