﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <Tasks/OsBarrierTask.h>
#include <FileUtility.h>
namespace nnt { namespace abuse {
    OsBarrierTask::OsBarrierTask(const String& typeName, const String& instanceName) :
        BaseTask(typeName, instanceName),
        m_threadArgs(nullptr),
        m_threads(nullptr),
        m_stacks(nullptr),
        m_maxThreads(64),
        m_maxGroups(64),
        m_numGroups(1),
        m_numThreads(1),
        m_numActualThreads(0),
        m_priority(nn::os::DefaultThreadPriority)
    {}

    OsBarrierTask::~OsBarrierTask()
    {}

    InitStatus OsBarrierTask::Initialize(const String& params)
    {
        ArgVector args;
        FileUtility::ParamsToArgs(params, args);

        for(ScriptArg& arg : args)
        {
            if(arg.argName == "MaxThreads")
                FileUtility::TryParseInt(arg, 1, 64, &m_maxThreads);
            else if(arg.argName == "MaxGroups")
                FileUtility::TryParseInt(arg, 1, 64, &m_maxGroups);
            else if(arg.argName == "Priority")
                FileUtility::TryParseInt(arg, nn::os::HighestThreadPriority, nn::os::LowestThreadPriority, &m_priority);
        }

        m_threads = (nn::os::ThreadType*) Platform::Allocate(sizeof(nn::os::ThreadType) * m_maxThreads);

        if(!m_threads)
        {
            LogError("Init Error - could not allocate threads\n");
            return INIT_ERROR;
        }

        for(int i = 0; i < m_maxThreads; ++i)
            ::new (&m_threads[i]) nn::os::ThreadType;

        LogInfo("Created thread array at %p\n", m_threads);
        m_stacks = (void**)Platform::Allocate(sizeof(void*) * m_maxThreads);

        if(!m_stacks)
        {
            LogError("Init Error - could not allocate stack array\n");
            Platform::Free(m_threads);
            m_threads = nullptr;
            return INIT_ERROR;
        }
        int i = 0;
        for(; i < m_maxThreads; ++i)
        {
            m_stacks[i] = Platform::AllocateAligned(4096, nn::os::StackRegionAlignment);
            if(!m_stacks[i])
                break;
        }

        if(i != m_maxThreads)
        {
            LogError("Init Error - could not allocate thread stacks\n");
            for(int j = 0; j <i; ++j)
            {
                Platform::Free(m_stacks[j]);
            }
            Platform::Free(m_stacks);
            Platform::Free(m_threads);
            m_threads = nullptr;
            m_stacks = nullptr;
            return INIT_ERROR;
        }

        m_threadArgs = (BarrierThreadArg*)Platform::Allocate( m_maxThreads * sizeof(BarrierThreadArg));

        if(!m_threadArgs)
            return INIT_ERROR;

        m_barriers = (nn::os::BarrierType*) Platform::Allocate(m_maxGroups * sizeof(nn::os::BarrierType));

        if(!m_barriers)
            return INIT_ERROR;

        return INIT_OK;
    }

    StartStatus OsBarrierTask::Start()
    {
        m_numGroups *= 2;
        if(m_numGroups > m_maxGroups || m_numGroups > m_numThreads)
        {
            m_numGroups = 1;

            m_numThreads *= 2;
            if(m_numThreads > m_maxThreads)
                m_numThreads = 1;
        }

        LogInfo("Starting %d threads in %d groups\n", m_numThreads, m_numGroups);

        return START_OK;
    }

    RunStatus OsBarrierTask::Run()
    {
        LogVerbose("++Run\n");

        m_numActualThreads = 0;
        for(int i = 0; i < m_numThreads; ++i)
        {
            m_threadArgs[i].task = this;
            m_threadArgs[i].group = i % m_numGroups;
            if(! CreateThread(&m_threads[i], workerFunc, &m_threadArgs[i], m_stacks[i], 4096, m_priority) )
                break;
            ++m_numActualThreads;
        }

        if(m_numActualThreads != m_numThreads)
        {
            LogWarning("Could not create %d threads. Continuing this run with %d threads instead.\n", m_numThreads, m_numActualThreads);
        }

        for(int i = 0; i < m_numGroups; ++i)
        {
            int groupSize = m_numActualThreads / m_numGroups;

            if(i < m_numActualThreads % m_numGroups)
                ++groupSize;
            LogVerbose("group %d waiting for %d threads\n", i, groupSize );
            nn::os::InitializeBarrier(&m_barriers[i], groupSize);
        }

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::StartThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
            nn::os::WaitThread(&m_threads[i]);

        for(int i = 0; i < m_numActualThreads; ++i)
        {
            DestroyThread(&m_threads[i]);
        }

        for(int i = 0; i < m_numGroups; ++i)
            nn::os::FinalizeBarrier(&m_barriers[i]);
        LogVerbose("--Run\n");
        return RUN_OK;
    }

    StopStatus OsBarrierTask::Stop()
    {
        return STOP_OK;
    }

    ShutdownStatus OsBarrierTask::Shutdown()
    {
        if(m_threads)
            Platform::Free(m_threads);

        if(m_stacks)
        {
            for(int i = 0; i < m_maxThreads; ++i)
                Platform::Free(m_stacks[i]);
            Platform::Free(m_stacks);
        }

        if(m_threadArgs)
            Platform::Free(m_threadArgs);

        if(m_barriers)
            Platform::Free(m_barriers);
        return SHUTDOWN_OK;
    }

    const char* OsBarrierTask::GetParamOptions()
    {
        return "MaxThreads = [1 - 64, default = 64] - maximum number of threads to run\n\
\tMaxGroups = [1 - 64, default = 64] - maximum number of groups\n\
\tPriority=[0 - 31, default = 16] - priority of threads";
    }

    void OsBarrierTask::workerFunc(void* vBarrierThreadArg)
    {
        BarrierThreadArg* arg = (BarrierThreadArg*) vBarrierThreadArg;
        arg->core = nn::os::GetCurrentCoreNumber();
        waitRandom(arg);
        arg->task->LogVerbose("++%d waiting\n", arg->group);
        nn::os::AwaitBarrier(&arg->task->m_barriers[arg->group]);
    }

    void OsBarrierTask::waitRandom(BarrierThreadArg* arg)
    {
        int64_t tick     = nn::os::GetSystemTick().GetInt64Value();
        int milli = ((tick & 0xf) + 1);
        auto    waitTime = nn::TimeSpan::FromMilliSeconds( milli );
        arg->task->LogVerbose("Sleeping for %d ms\n", milli);
        nn::os::SleepThread( waitTime );
    }

}    }
