﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <Abuse.h>
#include <Tasks/BaseTask.h>
#include <cstring>
#include <nn/util/util_FormatString.h>
namespace nnt
{
    namespace abuse
    {
        int BaseTask::s_nextId = 0;

        BaseTask::BaseTask(const String& typeName, const String& instanceName)
            : m_threads(),
              m_name(instanceName),
              m_typename(typeName),
              m_params(""),
              m_log(nullptr),
              m_startTime(0),
              m_runTime(0),
              m_basePriority(0),
              m_currentPriority(0),
              m_id(s_nextId++),
              m_processor(0),
              m_isRunning(false),
              m_canRun(false),
              m_isSuspended(false),
              m_isShutdown(false),
              m_isKilled(false),
              m_isInitialized(false)
        {
            int logNameLength = m_name.length() + m_typename.length() + m_name.length();

            char* logName = (char*)Platform::Allocate(logNameLength);

            if(m_name == "")
                nn::util::SNPrintf(logName, logNameLength, "%s", m_typename.c_str());
            else
                nn::util::SNPrintf(logName, logNameLength, "%s_%s", m_typename.c_str(), m_name.c_str());

            m_log = (Log*)Platform::Allocate(sizeof(Log));

            ::new (m_log) Log(logName, m_id);

            m_startTime = nn::os::GetSystemTick();
            Platform::Free(logName);
        }

        BaseTask::~BaseTask()
        {
            m_log->~Log();
            Platform::Free(m_log);
        }

        const char* BaseTask::GetParamOptions()
        {
            return " ";
        }

        int BaseTask::GetId()
        {
            return m_id;
        }

        const String& BaseTask::GetTypeName()
        {
            return m_typename;
        }

        const String& BaseTask::GetName()
        {
            return m_name;
        }

        void BaseTask::LogVerbose(const char* format, ...)
        {
            va_list marker;
            va_start(marker, format);

            m_log->VWrite(VerbosityVerbose, format, marker);

            va_end(marker);
        }

        void BaseTask::LogInfo(const char* format, ...)
        {
            va_list marker;
            va_start(marker, format);

            m_log->VWrite(VerbosityInfo, format, marker);

            va_end(marker);
        }

        void BaseTask::LogWarning(const char* format, ...)
        {
            va_list marker;
            va_start(marker, format);

            m_log->VWrite(VerbosityWarning, format, marker);

            va_end(marker);
        }

        void BaseTask::LogError(const char* format, ...)
        {
            va_list marker;
            va_start(marker, format);

            m_log->VWrite(VerbosityError, format, marker);

            va_end(marker);
        }

        void BaseTask::FlushLog(bool crashing)
        {
            m_log->Flush(crashing);
        }

        Log* BaseTask::GetLog()
        {
            return m_log;
        }

        bool BaseTask::IsInitialized()
        {
            return m_isInitialized;
        }

        void BaseTask::SetInitialized()
        {
            m_isInitialized = true;
        }

        bool BaseTask::IsRunning()
        {
            return m_isRunning;
        }

        void BaseTask::SetRunning(bool running)
        {
            //LogWarning("%s running=%d\n", m_name.c_str(), running);
            m_isRunning = running;
        }

        bool BaseTask::CanRun()
        {
            return m_canRun && !m_isKilled;
        }

        void BaseTask::SetCanRun(bool canRun)
        {
            m_canRun = canRun;
        }

        bool BaseTask::IsSuspended()
        {
            return m_isSuspended && !m_isKilled;
        }

        void BaseTask::SetSuspended(bool suspended)
        {
            m_isSuspended = suspended;
        }

        bool BaseTask::IsShutdown()
        {
            return m_isShutdown && !m_isKilled;
        }

        void BaseTask::SetShutdown()
        {
            m_isShutdown = true;
        }

        bool BaseTask::IsKilled()
        {
            return m_isKilled;
        }

        void BaseTask::SetKilled()
        {
            m_isKilled = true;
        }

        void BaseTask::SetPriority(int priority)
        {
            m_currentPriority = priority;
            m_basePriority = priority;
        }

        int BaseTask::GetBasePriority()
        {
            return m_basePriority;
        }

        int BaseTask::GetCurrentPriority()
        {
            return m_currentPriority;
        }

        void BaseTask::RaisePriority()
        {
            if(m_currentPriority > 0)
                --m_currentPriority;
        }

        void BaseTask::ResetPriority()
        {
            m_currentPriority = m_basePriority;
        }

        void BaseTask::SetProcessor(int processor)
        {
            m_processor = processor;
        }

        int BaseTask::GetProcessor()
        {
            return m_processor;
        }

        const String& BaseTask::GetParams()
        {
            return m_params;
        }

        void BaseTask::SetParams(const String& params)
        {
            m_params = params;
        }

        int BaseTask::Serialize(char* buffer, int bufferStart, int bufferSize)
        {
            int requiredSize = 3 * sizeof(int) + sizeof(int64_t) + m_name.length() + m_typename.length();
            if(requiredSize > (bufferSize - bufferStart))
                return -1;

            char workBuffer[1024];
            //id
            uint16_t size = Platform::PutNetworkBuffer(workBuffer, 0, (unsigned)m_id);
            //start time
            size = Platform::PutNetworkBuffer(workBuffer, size, (uint64_t)m_startTime.GetInt64Value());
            //type name
            size = Platform::PutNetworkBuffer(workBuffer, size, m_typename.length());
            strncpy(workBuffer + size, m_typename.c_str(), m_typename.length());
            (workBuffer + size)[m_typename.length()]=  '\0';
            size += (uint16_t)m_typename.length();
            //instance name
            size = Platform::PutNetworkBuffer(workBuffer, size, m_name.length());
            strncpy(workBuffer + size, m_name.c_str(), m_name.length() + 1);
            (workBuffer + size)[m_name.length()]=  '\0';
            size += (uint16_t)m_name.length();

            memcpy(buffer + bufferStart, workBuffer, size);
            return (bufferStart + size);
        }

        bool BaseTask::CreateThread(nn::os::ThreadType* thread, void (*threadFunc)(void*), void* arg, void* stack, int stackSize, int priority)
        {
            return CreateThread(thread,threadFunc, arg, stack, stackSize, priority, Abuse::GetLeastUsedProcessor());
        }

        bool BaseTask::CreateThread(nn::os::ThreadType* thread, void (*threadFunc)(void*), void* arg, void* stack, int stackSize, int priority, int processor)
        {

            if(!stack)
                return false;

            bool result = nn::os::CreateThread(thread, threadFunc, arg, stack, stackSize, priority, processor).IsSuccess();

            if(result)
            {
                m_threads.push_back(ThreadInfo(thread, processor));
                Abuse::ThreadRunningOnProcessor(processor);
            }

            return result;
        }

        bool BaseTask::DestroyThread(nn::os::ThreadType* thread)
        {
            auto itr = m_threads.begin();
            while(itr != m_threads.end())
            {
                if(itr->thread == thread)
                {
                    Abuse::ThreadStoppingOnProcessor(itr->processor);
                    nn::os::DestroyThread(itr->thread);
                    m_threads.erase(itr);
                    return true;
                }

                ++itr;
            }

            return false;
        }

        bool BaseTask::sOpenFiles(StringVector& fileNames, std::vector<File, PlatformAllocator<File>>& files, OpenOptions options)
        {
            for (unsigned i = 0; i < fileNames.size(); ++i)
            {
                File file = Platform::FileOpen(fileNames[i].c_str(), options);
                if (file != FILE_INVALID_HANDLE)
                    files.push_back(file);
                else
                {
                    for (unsigned j = 0; j < i; ++j)
                        Platform::FileClose(files[j]);
                    files.clear();
                    return false;
                }
            }
            return true;
        }
    }
}
