﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/ssl.h>
#include <curl/curl.h>

#include "../Include/HelperFunctions/HttpsHelper.h"

const bool bImportServerPki = false;
const bool bImportClientPki = false;

//---------------------------------------------------------------------------------------------
//
// HttpsHelperForCtxImport class
//
//---------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------
// HttpsHelperForCtxImport class : Public methods
//---------------------------------------------------------------------------------------------
HttpsHelperForCtxImport::HttpsHelperForCtxImport() : m_pSslContext(nullptr)
{
}

HttpsHelperForCtxImport::~HttpsHelperForCtxImport()
{
    Finalize();
}

// Configures CURL handle with CURLOPT_SSL_CONTEXT.
//
// In this method, nn::ssl::Context is created outside Libcurl, and it will be imported into
// Libcurl by CURLOPT_SSL_CONTEXT.
//
// When this method is used, the user is required to prepare the SSL context first, then tell each
// CURL handle to use it via the custom Nintendo SDK libcurl option, CURLOPT_SSL_CONTEXT.
// The SSL context must be prepared before the first CURL handle uses it and it must remain valid
// until the final CURL handle that uses it has been cleaned up with curl_easy_cleanup().
int HttpsHelperForCtxImport::Initialize()
{
    nn::Result result       = nn::ResultSuccess();
    bool       isCtxCreated = false;
    do
    {
        // Allocate a memory for nn::ssl::Context
        m_pSslContext = new nn::ssl::Context();
        if ( m_pSslContext == nullptr )
        {
            NN_LOG("[ERROR] Failed to allocate a memory for the SSL context.\n");
            break;
        }

        // Create nn::ssl::Context
        result = m_pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
        if( result.IsFailure() )
        {
            NN_LOG("[ERROR] Failed to create the SSL context\n");
            break;
        }

        isCtxCreated = true;

        // Import PKI data if necessary
        result = ImportPkiData(m_pSslContext);
        if(result.IsFailure())
        {
            NN_LOG("[ERROR] Failed to import PKI data.\n");
            break;
        }

        NN_LOG("Created the SSL context.\n");
        return 0; // Success
    } while (NN_STATIC_CONDITION(false));

    if ( m_pSslContext != nullptr )
    {
        if (isCtxCreated)
        {
            m_pSslContext->Destroy();
        }
        delete m_pSslContext;
        m_pSslContext = nullptr;
    }
    return -1;
}

// Frees resources
void HttpsHelperForCtxImport::Finalize()
{
    if ( m_pSslContext != nullptr )
    {
        m_pSslContext->Destroy();
        delete m_pSslContext;
        m_pSslContext = nullptr;

       NN_LOG("Destroyed the SSL context.\n");
    }
}

// Imports the SSL context into the specific CURL handle
int HttpsHelperForCtxImport::ImportSslContext(CURL* pInCurlHandle)
{
    CURLcode res = CURLE_OK;

    if ( pInCurlHandle == nullptr )
    {
        NN_LOG("[ERROR] Valid CURL handle was not passed.\n");
        return -1;
    }

    if ( m_pSslContext == nullptr )
    {
        NN_LOG("[ERROR] The SSL context is not allocated.\n");
        return -1;
    }

    // Set SSL context to be shared by this given handle
    res = curl_easy_setopt(pInCurlHandle, CURLOPT_SSL_CONTEXT, m_pSslContext);
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to share external SSL context: curl error $d\n", res);
        return -1;
    }

    return 0; // Success
}

void HttpsHelperForCtxImport::PrintErrorMessage(CURL* pInCurlHandle, CURLcode curlError)
{
    HttpsHelperBase::PrintErrorMessage(pInCurlHandle, nullptr, curlError);
}

//---------------------------------------------------------------------------------------------
//
// HttpsHelperForCallback class
//
//---------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------
// HttpsHelperForCallback class : Public methods
//---------------------------------------------------------------------------------------------
HttpsHelperForCallback::HttpsHelperForCallback() :
    m_Result(nn::ResultSuccess()),
    m_pServerCertBuf(nullptr)
{
}

HttpsHelperForCallback::~HttpsHelperForCallback()
{
    Finalize();
}

// Configures CURL handle with CURLOPT_SSL_CTX_FUNCTION.
//
// In this method, the callback function is registered by CURLOPT_SSL_CTX_FUNCTION.
// After that, it sets pointer to the nn::Result by CURLOPT_SSL_CTX_DATA to store nn::Result in the
// callback function. The callback function will be called before performing the SSL handshake.
//
// When this method is used, nn::ssl::Context which is maintained inside each CURL handle will be
// used. This context will be cleaned automatically when CURL handle is closed hence the user is
// not requried to perform cleanup process for the context.
int HttpsHelperForCallback::Initialize(CURL *pInCurlHandle)
{
    CURLcode res = CURLE_OK;

    if ( pInCurlHandle == nullptr )
    {
        NN_LOG("[ERROR] Valid CURL handle was not passed.\n");
        return -1;
    }

    // Set CURLOPT_SSL_CTX_FUNCTION callback function
    res = curl_easy_setopt(
        pInCurlHandle,
        CURLOPT_SSL_CTX_FUNCTION,
        HttpsHelperForCallback::CurlSslContextCallback);
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to set CURLOPT_SSL_CTX_FUNCTION.\n");
        return -1;
    }

    // Set CURLOPT_SSL_CTX_DATA to store nn::Result in CURLOPT_SSL_CTX_FUNCTION
    res = curl_easy_setopt(pInCurlHandle, CURLOPT_SSL_CTX_DATA , &m_Result);
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to set CURLOPT_SSL_CTX_DATA.\n");
        return -1;
    }

    // Set CURLOPT_SSL_CONN_FUNCTION_BEFORE callback function
    res = curl_easy_setopt(
        pInCurlHandle,
        CURLOPT_SSL_CONN_FUNCTION_BEFORE,
        HttpsHelperForCallback::CurlSslConnectionBeforeFunction);
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to set CURLOPT_SSL_CONN_FUNCTION_BEFORE.\n");
        return -1;
    }

    // Set CURLOPT_SSL_CONN_FUNCTION_AFTER callback function
    res = curl_easy_setopt(
        pInCurlHandle,
        CURLOPT_SSL_CONN_FUNCTION_AFTER,
        HttpsHelperForCallback::CurlSslConnectionAfterFunction);
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to set CURLOPT_SSL_CONN_FUNCTION_AFTER.\n");
        return -1;
    }

    // Set CURLOPT_SSL_CONN_DATA callback data ptr
    res = curl_easy_setopt(
        pInCurlHandle,
        CURLOPT_SSL_CONN_DATA,
        this);
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to set CURLOPT_SSL_CONN_DATA.\n");
        return -1;
    }

    m_pServerCertBuf = new char[ServerCertBufLen];
    if(m_pServerCertBuf == nullptr)
    {
        NN_LOG("[ERROR] Failed to allocate server cert buffer.\n");
        return -1;
    }

    NN_LOG("Server cert buffer allocated.\n");

    return 0; // Success
}

void HttpsHelperForCallback::Finalize()
{
    if(m_pServerCertBuf)
    {
        delete [] m_pServerCertBuf;
        m_pServerCertBuf = nullptr;
        NN_LOG("Deleted server cert buffer.\n");
    }

    return;
}

void HttpsHelperForCallback::PrintErrorMessage(CURL* pInCurlHandle, CURLcode curlError)
{
    HttpsHelperBase::PrintErrorMessage(pInCurlHandle, &m_Result, curlError);
}

const char* HttpsHelperForCallback::GetServerCertBuffer() const
{
    return m_pServerCertBuf;
}

//---------------------------------------------------------------------------------------------
// HttpsHelperForCallback class : Private static methods
//---------------------------------------------------------------------------------------------
// Callback function to be called before initializing SSL connection
size_t HttpsHelperForCallback::CurlSslContextCallback(
    CURL* pCurl,
    void* pSslContext,
    void* pUserData)
{
    nn::Result  result     = nn::ResultSuccess();
    nn::Result* pOutResult = static_cast<nn::Result *>(pUserData);

    // Obtain pointer to the SSL context passed by CURLOPT_SSL_CTX_FUNCTION
    nn::ssl::Context* pContext = static_cast<nn::ssl::Context*>(pSslContext);

    do
    {
        // Note that the created context will be destroyed automatically when cleanup for
        // corresponding CURL handle is performed when CURLOPT_SSL_CTX_FUNCTION is used.
        // Hence the user of Libcurl is not required to deal with destroying the context.
        result = pContext->Create(nn::ssl::Context::SslVersion_Auto);
        if( result.IsFailure() )
        {
            NN_LOG("[ERROR] Failed to create SSL context.\n");
            break;
        }

        // Import PKI data if necessary
        result = ImportPkiData(pContext);
        if( result.IsFailure() )
        {
            NN_LOG("[ERROR] Failed to import PKI data.\n");
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    // Store the result in the user data buffer set by CURLOPT_SSL_CTX_DATA
    if (pOutResult != nullptr)
    {
        *pOutResult = result;
    }

    if (result.IsSuccess())
    {
        return 0; // Success
    }
    else
    {
        return static_cast<size_t>(-1);
    }
}

// Callback function to be called before DoHandshake() by LibCurl.
//  This is where you should configure the SSL connection object if more configuration is needed than what is
//  provided by curl_easy_setopt().
//  This is an opportunity to cause LibCurl to fail with CURLE_ABORTED_BY_CALLBACK by returning non-zero.
size_t HttpsHelperForCallback::CurlSslConnectionBeforeFunction(CURL* pCurl, void *pSslConnection, void *pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pCurl);

    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    HttpsHelperForCallback* pHelper = reinterpret_cast<HttpsHelperForCallback*>(pUserData);

    NN_LOG("\n ** CurlSslConnectionBeforeFunction\n\n");

    // Calling SetServerCertBuffer() is not required, but this shows where it should be done if needed.
    nn::Result result = pConnection->SetServerCertBuffer(pHelper->m_pServerCertBuf, ServerCertBufLen);
    if(result.IsFailure())
    {
        NN_LOG(" * Failed to SetServerCertBuffer. Desc: %d\n\n", result.GetDescription());
        return 1; // Fail LibCurl.
    }

    return 0;
}

// Callback function to be called after DoHandshake() by LibCurl.
//  Here is where you can obtain handshake info by accessing the curl_handshake_info struct that is passed.
//  This is also an opportunity to cause LibCurl to fail with CURLE_ABORTED_BY_CALLBACK by returning non-zero.
size_t HttpsHelperForCallback::CurlSslConnectionAfterFunction(CURL* pCurl, void *pSslConnectionPtr, void *pUserData, curl_handshake_info handshakeInfo) NN_NOEXCEPT
{
    NN_UNUSED(pCurl);
    NN_UNUSED(pUserData);
    size_t ret = 0;

    NN_LOG("\n ** CurlSslConnectionAfterFunction\n\n");

    if(handshakeInfo.isHandshakeSuccess)
    {
        NN_LOG("Handshake success!\n");
        NN_LOG("Server Cert Data Len: %u\n", handshakeInfo.certDataSize);
        NN_LOG("Server Cert Count: %u\n\n", handshakeInfo.certCount);
    }
    else
    {
        NN_LOG("Handshake failure!\n");

        nn::Result* pErrorArray = new nn::Result[VerifyResultArrayLen];
        if(pErrorArray != nullptr)
        {
            nn::ssl::Connection* pSslConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnectionPtr);

            uint32_t errorsWritten = 0;
            uint32_t totalErrors = 0;

            nn::Result result = pSslConnection->GetVerifyCertErrors(pErrorArray, &errorsWritten, &totalErrors, VerifyResultArrayLen);
            if(result.IsFailure())
            {
                if(nn::ssl::ResultBufferTooShort::Includes(result))
                {
                    NN_LOG("nn::ssl::GetVerifyErrors() returned ResultBufferTooShort! ErrorsWritten: %u TotalErrors: %u\n\n", errorsWritten, totalErrors);
                }
                else
                {
                    NN_LOG("nn::ssl::GetVerifyErrors() failed! Desc: %d\n\n", result.GetDescription());
                }

                ret = 1; // Fail LibCurl.
            }
            else
            {
                NN_LOG("ErrorsWritten: %u TotalErrors: %u\n", errorsWritten, totalErrors);
                for(uint32_t iError = 0; iError < errorsWritten; ++iError)
                {
                    NN_LOG("Verify error %u: desc: %d\n", iError, pErrorArray[iError].GetDescription());
                }
            }

            delete [] pErrorArray;
            pErrorArray = nullptr;
        }
        else
        {
            NN_LOG("CurlSslConnectionAfterFunction: Failed to allocate Verify result array\n\n");
            ret = 1; // Fail LibCurl.
        }
    }

    return ret;
}

//---------------------------------------------------------------------------------------------
//
// HttpsHelperBase class
//
//---------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------
// HttpsHelperBase class : Public methods
//---------------------------------------------------------------------------------------------

// Configures SSL verify options
int HttpsHelperBase::ConfigureVerifyOption(CURL* pInCurlHandle, VerifyOption* pInVerifyOption)
{
    CURLcode res = CURLE_OK;

    if ( pInCurlHandle == nullptr )
    {
        NN_LOG("[ERROR] Valid CURL handle was not passed.\n");
        return -1;
    }

    if ( pInVerifyOption == nullptr )
    {
        NN_LOG("[ERROR] NULL was passed in pInVerifyOption.\n");
        return -1;
    }

    // NOTE: CURLOPT_SSL_VERIFYPEER is enabled by default in Libcurl
    res = curl_easy_setopt(
        pInCurlHandle,
        CURLOPT_SSL_VERIFYPEER,
        (pInVerifyOption->isVerifyPeer == true)?(1L):(0L));
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to set CURLOPT_SSL_VERIFYPEER.\n");
        return -1;
    }

    // NOTE: CURLOPT_SSL_VERIFYHOST is enabled by default in Libcurl
    res = curl_easy_setopt(
        pInCurlHandle,
        CURLOPT_SSL_VERIFYHOST,
        (pInVerifyOption->isVerifyName == true)?(2L):(0L));
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to set CURLOPT_SSL_VERIFYHOST.\n");
        return -1;
    }

    // NOTE: CURLOPT_SSL_VERIFYDATE is DISABLED by default in Libcurl
    res = curl_easy_setopt(
        pInCurlHandle,
        CURLOPT_SSL_VERIFYDATE,
        (pInVerifyOption->isVerifyTime == true)?(1L):(0L));
    if( res != CURLE_OK )
    {
        NN_LOG("[ERROR] Failed to set SSL CURLOPT_SSL_VERIFYDATE.\n");
        return -1;
    }

    return 0; // Success
}

// Gets SSL result from CURLINFO value
bool HttpsHelperBase::GetSslResultByCurlInfo(
    nn::Result* pOutResult,
    CURL *pInCurlHandle,
    CURLINFO curlInfo)
{
    long       tmpInfoValue = 0;
    CURLcode   res          = CURLE_OK;

    do
    {
        if ( pOutResult == nullptr )
        {
            break;
        }

        if ( (curlInfo != CURLINFO_SSL_VERIFYRESULT) &&
             (curlInfo != CURLINFO_SSL_HANDSHAKE_RESULT) )
        {
            break;
        }

        // Get CURLINFO value
        res = curl_easy_getinfo(pInCurlHandle, curlInfo, &tmpInfoValue);
        if (res != CURLE_OK)
        {
            NN_LOG("curl_easy_getinfo failed: %d\n", res);
            break;
        }

        // Convert CURLINFO value into the form of nn::Result
        if (nn::ssl::GetSslResultFromValue(
                pOutResult,
                reinterpret_cast<char*>(&tmpInfoValue),
                sizeof(tmpInfoValue)).IsFailure())
        {
            NN_LOG("Failed to convert SSL result by GetSslResultFromValue.\n");
            break;
        }

        return true;
    } while (NN_STATIC_CONDITION(false));

    NN_LOG("Failed to convert CURLINFO value into nn::Result.\n");
    return false;
}

// Prints an error message based on CURLcode
void HttpsHelperBase::PrintErrorMessage(
    CURL* pInCurlHandle,
    nn::Result* pInResult,
    CURLcode curlError)
{
    NN_LOG(" HTTPS failed - ");

    switch ( curlError )
    {
    case CURLE_PEER_FAILED_VERIFICATION:
        {
            nn::Result sslVerifyResult = nn::ResultSuccess();
            bool isGotCurlInfo = GetSslResultByCurlInfo(
                &sslVerifyResult,
                pInCurlHandle,
                CURLINFO_SSL_VERIFYRESULT);
            if ( isGotCurlInfo == true )
            {
                NN_LOG("Failed to verify the peer (failure reason in SSL Result description: %d)\n",
                    sslVerifyResult.GetDescription());
            }
            else
            {
                NN_LOG("Failed to verify the peer (failed to get CURLINFO_SSL_VERIFYRESULT)\n");
            }
        }
        break;
    case CURLE_SSL_CONNECT_ERROR:
        {
            nn::Result sslHandshakeResult = nn::ResultSuccess();
            bool isGotCurlInfo = GetSslResultByCurlInfo(
                &sslHandshakeResult,
                pInCurlHandle,
                CURLINFO_SSL_HANDSHAKE_RESULT);
            if ( isGotCurlInfo == true )
            {
                if (sslHandshakeResult.IsFailure())
                {
                    NN_LOG("Failed to complete the SSL handshake (failure reason in SSL Result description: %d)\n",
                        sslHandshakeResult.GetDescription());
                }
                else
                {
                    NN_LOG("Failed to start the SSL handshake\n");
                }
            }
            else
            {
                NN_LOG("Failed to perform the SSL handshake (failed to get CURLINFO_SSL_HANDSHAKE_RESULT)\n");
            }
        }
        break;
    case CURLE_SSL_CACERT:
        {
            NN_LOG("The SSL handshake failed because the server is untrusted. "
                    "Please consider importing an appropriate CA certificate by nn::ssl::Context::ImportServerPki.\n");
        }
        break;
    case CURLE_SSL_CERTPROBLEM:
        {
            NN_LOG("The SSL handshake failed because the appropliate client certificate was not found. "
                    "Please consider importing an appropriate client PKI by nn::ssl::Context::ImportClientPki.\n");
        }
        break;
    case CURLE_SSL_CTX_FUNCTION_NOT_SET:
        {
            NN_LOG("Failed to perform the SSL handshake because CURLOPT_SSL_CTX_FUNCTION is not set, "
                    "or the SSL context is not imported by CURLOPT_SSL_CONTEXT.\n");
        }
        break;
    case CURLE_SSL_CTX_INVALID:
        {
            NN_LOG("Failed to perform the SSL handshake because nn::ssl::Context is not created yet.\n");
            break;
        }
    case CURLE_ABORTED_BY_CALLBACK:
        {
            if ( (pInResult != nullptr) && (pInResult->IsFailure()) )
            {
                NN_LOG("Failed to setup the SSL connection due to an error in CURLOPT_SSL_CTX_FUNCTION "
                        "(SSL Result description: %d)\n", pInResult->GetDescription());
            }
            else
            {
                NN_LOG("Failed to setup the SSL connection due to aborted callback function not for HTTPS.\n");
            }
        }
        break;
    case CURLE_SSL_CTX_FATAL:
    case CURLE_SSL_INVALID_REFERENCE:
        {
            NN_LOG("The SSL setup failed unexpectedly. Please finalize libraries and try again.\n");
            break;
        }
    default:
        NN_LOG("The error is not specific to https (CURLCode: %d).\n", curlError);
        break;
    }
}

//---------------------------------------------------------------------------------------------
// HttpsHelperBase class : Public static methods
//---------------------------------------------------------------------------------------------
// Returns true when a given string starts with "https"
bool HttpsHelperBase::IsUrlHttps(const char* pInUrl)
{
    static const char* s_pHttpsString = "https";
    if (pInUrl != nullptr && strncmp(pInUrl, s_pHttpsString, strlen(s_pHttpsString)) == 0)
    {
        return true;
    }

    return false;
}

//---------------------------------------------------------------------------------------------
// HttpsHelperBase class : Protected static methods
//---------------------------------------------------------------------------------------------

// Import PKI data
//
// To import a server certificate, nn::ssl::Context::ImportServerPki needs to be called.
// The format of a server certificate needs to be PEM or DER. Please note that ImportServerPki
// takes only one certificate even when multiple certificates are stored in the given buffer.
// Only the first one will be imported in such the case.
//
// To import a client PKI, nn::ssl::Context::ImportClientPki needs to be called.
// The format of a client PKI needs to be in PKCS12 format. The password for PKCS12 data needs to
// be passed to this function too. If there's no password, please just pass a NULL pointer for
// the pointer to the password string and 0 as password length. Please note that ImportClientPki
// allows importing only one client PKI per SSL context.
nn::Result HttpsHelperBase::ImportPkiData(nn::ssl::Context* pSslContext)
{
    nn::Result result = nn::ResultSuccess();

    do
    {
        if (bImportServerPki)
        {
            // It is possible to import CA certificates here by using obtained SSL context
            nn::ssl::CertStoreId serverCertStoreId = 0;
            char*    pCaCert    = nullptr; // Please provide CA certificate in PEM or DER format!
            uint32_t caCertSize = 0;    // Please set the size of CA certificate!
            result = pSslContext->ImportServerPki(
                &serverCertStoreId,
                pCaCert,
                caCertSize,
                nn::ssl::CertificateFormat_Pem); // Please pass nn::ssl::CertificateFormat_Der if certificate in pCaCert is in DER format
            if (result.IsFailure())
            {
                NN_LOG("[ERROR] Failed to import server PKI.\n");
                break;
            }
            else
            {
                NN_LOG("Imported server PKI (certStoreID:%d).\n", serverCertStoreId);
            }
        }

        if (bImportClientPki)
        {
            // It is possible to client PKI information here by using obtained SSL context
            char*       pPkcs12Data    = nullptr;    // Please set pointer to the data in PKCS12 format!
            uint32_t    pkcs12DataSize = 0;          // Please set the size of PKCS12 data!
            const char* pkcs12Password = "password"; // Please set the appropriate password for the PKCS12 data!
            uint32_t    passwordSize   = static_cast<uint32_t>(strlen(pkcs12Password));

            nn::ssl::CertStoreId clientCertStoreId = 0;

            result = pSslContext->ImportClientPki(
                &clientCertStoreId,
                pPkcs12Data,
                pkcs12Password,
                pkcs12DataSize,
                passwordSize);
            if (result.IsFailure())
            {
                NN_LOG("[ERROR] Failed to import client PKI.\n");
                break;
            }
            else
            {
                NN_LOG("Imported client PKI (certStoreID:%d).\n", clientCertStoreId);
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}
