﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstddef>
#include <cstdarg>
#include <cstdint>
#include "fwk_Print.h"

namespace nnt { namespace svc {

// From kern_DebugString.cpp

typedef char CharT;

class TVSNPrintfImpl
{
private:
    struct dst_string
    {
        size_t  len;
        CharT   *cur;
        CharT   *base;
    };

    static void string_put_char(dst_string * p, CharT c)
    {
        if (p->len > 0)
        {
            *p->cur = c, --p->len;
        }
        ++p->cur;
    }

    static void string_fill_char(dst_string * p, CharT c, int n)
    {
        if (n > 0)
        {
            size_t  i, k = p->len;
            if (k > (size_t) n)
            {
                k = (size_t) n;
            }
            for (i = 0; i < k; ++i)
            {
                p->cur[i] = c;
            }
            p->len -= k;
            p->cur += n;
        }
    }

    static void string_put_string(dst_string * p, const CharT *s, int n)
    {
        if (n > 0)
        {
            size_t  i, k = p->len;
            if (k > (size_t) n)
            {
                k = (size_t) n;
            }
            for (i = 0; i < k; ++i)
            {
                p->cur[i] = s[i];
            }
            p->len -= k;
            p->cur += n;
        }
    }

public:
    static void TVSNPrintf(CharT *dst, size_t len, const CharT *fmt, ::std::va_list vlist)
    {
        CharT   buf[24];
        int     n_buf;
        CharT   prefix[2];
        int     n_prefix;

        const CharT *s = fmt;

        dst_string str;
        str.len = len, str.cur = str.base = dst;

        while (*s)
        {
            if (*s != '%')
            {
                string_put_char(&str, *s++);
            }
            else
            {
                /* output with format */
                enum
                {
                    flag_blank = 000001,   /* ' ' */
                    flag_plus = 000002,    /* '+' */
                    flag_sharp = 000004,   /* '#' */
                    flag_minus = 000010,   /* '-' */
                    flag_zero = 000020,    /* '0' */
                    flag_l1 = 000040,      /* "l" */
                    flag_h1 = 000100,      /* "h" */
                    flag_l2 = 000200,      /* "ll" */
                    flag_h2 = 000400,      /* "hh" */
                    flag_p  = 001000,      /* %p */
                    flag_unsigned = 010000, /* 'o', 'u', ... */
                    flag_end
                };
                int     flag = 0, width = 0, precision = -1, radix = 10;
                CharT    hex_char = 'a' - 10;
                const CharT *p_start = s;
                /* flags */
                for (;;)
                {
                    switch (*++s)
                    {
                    case '+':
                        if (s[-1] != ' ')
                        {
                            break;
                        }
                        flag |= flag_plus;
                        continue;
                    case ' ':
                        flag |= flag_blank;
                        continue;
                    case '-':
                        flag |= flag_minus;
                        continue;
                    case '0':
                        flag |= flag_zero;
                        continue;
                    default:
                        break;
                    }
                    break;
                }
                /* width */
                if (*s == '*')
                {
                    ++s, width = va_arg(vlist, int);
                    if (width < 0)
                    {
                        width = -width, flag |= flag_minus;
                    }
                }
                else
                {
                    while ((*s >= '0') && (*s <= '9'))
                    {
                        width = (width * 10) + *s++ - '0';
                    }
                }
                /* precision */
                if (*s == '.')
                {
                    ++s, precision = 0;
                    if (*s == '*')
                    {
                        ++s, precision = va_arg(vlist, int);
                        if (precision < 0)
                        {
                            precision = -1;
                        }
                    }
                    else
                    {
                        while ((*s >= '0') && (*s <= '9'))
                        {
                            precision = (precision * 10) + *s++ - '0';
                        }
                    }
                }
                /* option */
                switch (*s)
                {
                case 'h':
                    if (*++s != 'h')
                    {
                        flag |= flag_h1;
                    }
                    else
                    {
                        ++s, flag |= flag_h2;
                    }
                    break;
                case 'l':
                    if (*++s != 'l')
                    {
                        flag |= flag_l1;
                    }
                    else
                    {
                        ++s, flag |= flag_l2;
                    }
                    break;
                default:
                    break;
                }

                /* type */
                switch (*s)
                {
                case 'd':                 /* signed decimal */
                case 'i':                 /* signed decimal */
                    goto put_integer;
                case 'o':                 /* unsigned octal */
                    radix = 8;
                    flag |= flag_unsigned;
                    goto put_integer;
                case 'u':                 /* unsigned decimal */
                    flag |= flag_unsigned;
                    goto put_integer;
                case 'X':                 /* unsigned hexadecimal */
                    hex_char = 'A' - 10;
                    goto put_hexadecimal;
                case 'x':                 /* unsigned hexadecimal */
                    goto put_hexadecimal;
                case 'p':                 /* pointer */
                    flag |= flag_p;
                    if (sizeof(long) == sizeof(void *))
                    {
                        flag |= flag_l1;
                    }
                    else if (sizeof(long long) == sizeof(void *))
                    {
                        flag |= flag_l2;
                    }
                    precision = sizeof(void *) * 2;
                    goto put_hexadecimal;

                case 'c':                 /* character */
                    if (precision >= 0)
                    {
                        goto put_invalid;
                    }
                    {
                        int     c = va_arg(vlist, int);
                        width -= 1;
                        if (flag & flag_minus)
                        {
                            string_put_char(&str, (CharT)c);
                            string_fill_char(&str, ' ', width);
                        }
                        else
                        {
                            CharT    pad = (CharT)((flag & flag_zero) ? '0' : ' ');
                            string_fill_char(&str, pad, width);
                            string_put_char(&str, (CharT)c);
                        }
                        ++s;
                    }
                    break;

                case 's':                 /* string */
                    {
                        int     n_buf2 = 0;
                        const CharT *p_buf = va_arg(vlist, const CharT *);
                        if (precision < 0)
                        {
                            while (p_buf[n_buf2])
                            {
                                ++n_buf2;
                            }
                        }
                        else
                        {
                            while ((n_buf2 < precision) && p_buf[n_buf2])
                            {
                                ++n_buf2;
                            }
                        }
                        width -= n_buf2;
                        if (flag & flag_minus)
                        {
                            string_put_string(&str, p_buf, n_buf2);
                            string_fill_char(&str, ' ', width);
                        }
                        else
                        {
                            CharT    pad = (CharT)((flag & flag_zero) ? '0' : ' ');
                            string_fill_char(&str, pad, width);
                            string_put_string(&str, p_buf, n_buf2);
                        }
                        ++s;
                    }
                    break;

                case '%':                 /* output '%' */
                    if (p_start + 1 != s)
                    {
                        goto put_invalid;
                    }
                    string_put_char(&str, *s++);
                    break;

                default:                  /* invalid type */
                    goto put_invalid;

                  put_invalid:
                    string_put_string(&str, p_start, s - p_start);
                    break;

                  put_hexadecimal:
                    radix = 16;
                    flag |= flag_unsigned;
                  put_integer:
                    {
                        unsigned long long val = 0;
                        n_prefix = 0;

                        if (flag & flag_minus)
                        {
                            flag &= ~flag_zero;
                        }
                        if (precision < 0)
                        {
                            precision = 1;
                        }
                        else
                        {
                            flag &= ~flag_zero;
                        }

                        if (flag & flag_unsigned)
                        {
                            if (flag & flag_h2)
                            {
                                val = static_cast<unsigned char>(va_arg(vlist, int));
                            }
                            else if (flag & flag_h1)
                            {
                                val = static_cast<unsigned short>(va_arg(vlist, int));
                            }
                            else if (flag & flag_l2)
                            {
                                val = va_arg(vlist, unsigned long long);
                            }
                            else if (flag & flag_l1)
                            {
                                val = va_arg(vlist, unsigned long);
                            }
                            else
                            {
                                val = va_arg(vlist, unsigned);
                            }
                            flag &= ~(flag_plus | flag_blank);
                            if (flag & (flag_sharp | flag_p))
                            {
                                if (radix == 16)
                                {
                                    if (val != 0 || (flag & flag_p))
                                    {
                                        prefix[0] = (CharT)(hex_char + (10 + 'x' - 'a'));
                                        prefix[1] = '0';
                                        n_prefix = 2;
                                    }
                                }
                                else if (radix == 8)
                                {
                                    prefix[0] = '0';
                                    n_prefix = 1;
                                }
                            }
                        }
                        else
                        {
                            if (flag & flag_h2)
                            {
                                val = static_cast<char>(va_arg(vlist, int));
                            }
                            else if (flag & flag_h1)
                            {
                                val = static_cast<short>(va_arg(vlist, int));
                            }
                            else if (flag & flag_l2)
                            {
                                val = va_arg(vlist, unsigned long long);
                            }
                            else if (flag & flag_l1)
                            {
                                val = va_arg(vlist, long);
                            }
                            else
                            {
                                val = va_arg(vlist, int);
                            }
                            if ((val >> 32) & 0x80000000)
                            {
                                val = ~val + 1;
                                prefix[0] = '-';
                                n_prefix = 1;
                            }
                            else
                            {
                                if (val || precision)
                                {
                                    if (flag & flag_plus)
                                    {
                                        prefix[0] = '+';
                                        n_prefix = 1;
                                    }
                                    else if (flag & flag_blank)
                                    {
                                        prefix[0] = ' ';
                                        n_prefix = 1;
                                    }
                                }
                            }
                        }
                        n_buf = 0;
                        switch (radix)
                        {
                        case 8:
                            while (val != 0)
                            {
                                int d = static_cast<int>(val & 0x07);
                                val >>= 3;
                                buf[n_buf++] = (CharT)(d + '0');
                            }
                            break;
                        case 10:
                            if ((val >> 32) == 0)
                            {
                                uint32_t v = static_cast<uint32_t>(val);
                                while (v != 0)
                                {
                                    // uint32_t と定数の除算であれば、コンパイラが自動的に
                                    // マジックナンバーを使用した積算に変換する
                                    uint32_t r = v / 10;
                                    int d = static_cast<int>(v - (r * 10));
                                    v = r;
                                    buf[n_buf++] = (CharT)(d + '0');
                                }
                            }
                            else
                            {
                                while (val != 0)
                                {
                                    uint64_t r = val / 10;
                                    int d = static_cast<int>(val - (r * 10));
                                    val = r;
                                    buf[n_buf++] = (CharT)(d + '0');
                                }
                            }
                            break;
                        case 16:
                            while (val != 0)
                            {
                                int d = static_cast<int>(val & 0x0f);
                                val >>= 4;
                                buf[n_buf++] = (CharT)((d < 10) ? (d + '0') : (d + hex_char));
                            }
                            break;
                        default:
                            break;
                        }
                        if ((n_prefix > 0) && (prefix[0] == '0'))
                        {
                            n_prefix = 0;
                            buf[n_buf++] = '0';
                        }
                    }
                    goto put_to_stream;

                  put_to_stream:
                    {
                        int n_pad = precision - n_buf;
                        if (flag & flag_zero)
                        {
                            if (n_pad < width - n_buf - n_prefix)
                            {
                                n_pad = width - n_buf - n_prefix;
                            }
                        }
                        if (n_pad > 0)
                        {
                            width -= n_pad;
                        }
                        width -= n_prefix + n_buf;
                        if (!(flag & flag_minus))
                        {
                            string_fill_char(&str, ' ', width);
                        }
                        while (n_prefix > 0)
                        {
                            string_put_char(&str, prefix[--n_prefix]);
                        }
                        string_fill_char(&str, '0', n_pad);
                        while (n_buf > 0)
                        {
                            string_put_char(&str, buf[--n_buf]);
                        }
                        if (flag & flag_minus)
                        {
                            string_fill_char(&str, ' ', width);
                        }
                        ++s;
                    }
                    break;
                }
            }
        }

        if (str.len > 0)
        {
            *str.cur = '\0';
        }
        else if (len > 0)
        {
            str.base[len - 1] = '\0';
        }
    }   // NOLINT(readability/fn_size)
};

int TVSNPrintf(char* buffer, size_t size, const char* format, ::std::va_list vlist)
{
    TVSNPrintfImpl::TVSNPrintf(buffer, size, format, vlist);

    int len = 0;
    while (buffer[len] != 0)
        len++;
    return len;
}

} }
