﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_SdCardDevice.h"
#include <nn/nn_Abort.h>
#include "sdmmc_Log.h"

namespace nn { namespace sdmmc1 {
namespace detail {

namespace
{
    // CMD8 SEND_IF_COND の check pattern (任意の値)
    const uint8_t CheckPatternOfCommandSendIfCond = 0xAA;

    // OCR Register ビット定義
    const uint32_t OcrCardPowerUpStatus = 0x1 << 31;
    const uint32_t OcrCardCapacityStatus = 0x1 << 30;
}

Result SdCardDevice::IssueCommandSendIfCond() const NN_NOEXCEPT
{
    // argument[31:12] reserved bits : 0
    // argument[11: 8] supply voltage(VHS) : 0x01(2.7-3.6V)
    // argument[ 7: 0] check pattern : 任意の値
    uint32_t argument = (0x01 << 8) | static_cast<uint32_t>(CheckPatternOfCommandSendIfCond);
    ResponseType responseType = ResponseType_R7;
    Command command(8, argument, responseType, false);

    IHostController* pHostController = BaseDevice::GetHostController();
    Result result = pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    if (response != argument)
    {
        NN_DETAIL_SDMMC_ERROR_LOG("CMD8 Unexpected Response: 0x%04X (Expected Response)\n", response, argument);
        return ResultSdCardValidationError();
    }

    return ResultSuccess();
}

Result SdCardDevice::IssueCommandSendOpCond(uint32_t* pOutOcr, uint8_t hcs) const NN_NOEXCEPT
{
    // argument[31] reserved bit
    // argument[30] HCS : 1(Host Capacity Support) or 0(Not Support)
    // argument[29] reserved for eSD
    // argument[28] XPC : 1(150mA, High Speed対応) SDXC専用,
    // argument[27;25] reserved bits
    // argument[24] S18R : 0(1.8Vにしない),
    // argument[23:0] Voltage : 0x100000(3.2-3.3V),
    uint32_t argument = (static_cast<uint32_t>(hcs) << 30) | (0x1 << 28) | 0x100000;
    ResponseType responseType = ResponseType_R3;
    Command command(41, argument, responseType, false);

    IHostController* pHostController = BaseDevice::GetHostController();
    Result result = pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    pHostController->GetLastResponse(pOutOcr, sizeof(uint32_t), responseType);

    return ResultSuccess();
}

Result SdCardDevice::IssueCommandSendRelativeAddr(uint16_t* pOutRca) const NN_NOEXCEPT
{
    // argument[31:0] stuff bit
    ResponseType responseType = ResponseType_R6;
    Command command(3, 0, responseType, false);

    IHostController* pHostController = BaseDevice::GetHostController();
    Result result = pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    // 上位 16bit RCA, 下位 16bit Device Status (TODO: Device Status のチェック)
    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    *pOutRca = static_cast<uint16_t>((response >> 16) & 0xFFFF);

    return ResultSuccess();
}

Result SdCardDevice::IssueCommandAppCmd(uint16_t rca, DeviceState expectedState) const NN_NOEXCEPT
{
    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    uint32_t argument = static_cast<uint32_t>(rca) << 16;
    return BaseDevice::IssueCommandAndCheckR1(55, argument, false, expectedState);
}

Result SdCardDevice::Activate() NN_NOEXCEPT
{
    // バス設定, クロック供給
    BaseDevice::SetBusIdentificationMode(nullptr);

    // idle state へ遷移
    Result result = BaseDevice::IssueCommandGoIdleState();
    if (result.IsFailure())
    {
        return result;
    }

    // Physical Spec Version 2.00 以上かチェック
    uint8_t hcs = 1;
    result = IssueCommandSendIfCond();
    if (result.IsFailure())
    {
        // 失敗した場合は Physical Spec Version 2.00 未満として HCS = 0 を指定する
        hcs = 0;
    }

    // 取得できるまで RCA は 0 とする
    uint16_t rca = 0;
    BaseDevice::SetRca(rca);

    // ready state へ遷移、High Capacity 有無取得
    bool isCompleted = false;
    while (true)
    {
        result = IssueCommandAppCmd(rca, DeviceState_Unknown);   // Device State のチェックは除外する
        if (result.IsFailure())
        {
            return result;
        }

        result = IssueCommandSendOpCond(&m_Ocr, hcs);
        if (result.IsFailure())
        {
            return result;
        }
        if ((m_Ocr & OcrCardPowerUpStatus) != 0)
        {
            isCompleted = true;
            if ((m_Ocr & OcrCardCapacityStatus) != 0)
            {
                BaseDevice::SetHighCapacity(true);
            }
            else
            {
                BaseDevice::SetHighCapacity(false);
            }
            break;
        }
        // TODO: タイムアウト実装
    }

    // ident state へ遷移、CID 取得
    result = BaseDevice::IssueCommandAllSendCid(m_Cid, sizeof(m_Cid));
    if (result.IsFailure())
    {
        return result;
    }

    // stby state へ遷移、RCA 取得
    while (true)
    {
        result = IssueCommandSendRelativeAddr(&rca);
        if (result.IsFailure())
        {
            return result;
        }
        if (rca != 0)
        {
            BaseDevice::SetRca(rca);
            break;
        }
        // TODO: タイムアウト実装
    }

    // CSD 取得
    result = IssueCommandSendCsd(m_Csd, sizeof(m_Csd), rca);
    if (result.IsFailure())
    {
        return result;
    }

    // Default Speed にクロックアップ
    BaseDevice::ChangeDeviceClockMode(IHostController::DeviceClockMode_SdCardDefaultSpeed);

    // tran state へ遷移
    result = BaseDevice::IssueCommandSelectCard(rca);
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
