﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include "test_TestCondition.h"
#include "test_TestSvc.h"
#include "util_TestAssert.h"
#include "util_TestProcess.h"
#include "util_TestIpc.h"
#include <cstring>

template <class T>
inline T RoundUp(T addr, T align)
{
    return (addr + align - 1) & ~(align - 1);
}

class AutoHandleClose
{
    public:
        explicit AutoHandleClose(nn::svc::Handle handle)
            : mHandle(handle)
        {}

        ~AutoHandleClose()
        {
            // 既に閉じられている場合、失敗するかもしれないのでResult のチェックはしない
            nn::svc::CloseHandle(mHandle);
        }

        nn::Result Close()
        {
            return nn::svc::CloseHandle(mHandle);
        }

    private:
        nn::svc::Handle mHandle;
};

class TestHeap
{
    public:
        explicit TestHeap(size_t size)
        {
            Resize(size);
        }

        ~TestHeap()
        {
            nn::Result result = nn::svc::SetHeapSize(&mAddr, 0);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        void Resize(size_t size)
        {
            mSize = (size + (HeapAlign - 1)) & ~(HeapAlign - 1);
            nn::Result result = nn::svc::SetHeapSize(&mAddr, mSize);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        uintptr_t GetAddress() const
        {
            return mAddr;
        }
        size_t GetSize() const
        {
            return mSize;
        }

    private:
        uintptr_t mAddr;
        size_t mSize;
};

class TestThread
{
    public:
        TestThread(uintptr_t pc, uintptr_t param, uintptr_t sp, int32_t priority, int32_t idealCore)
        {
            nn::Result result = nn::svc::CreateThread(&mHandle, pc, param, sp, priority, idealCore);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~TestThread()
        {
            Close();
        }

        void Start()
        {
            nn::Result result = nn::svc::StartThread(mHandle);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        void Wait()
        {
            int index;
            nn::Result result = nn::svc::WaitSynchronization(&index, &mHandle, 1, -1);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        void Close()
        {
            // 他で閉じられているかもしれないので、Result はチェックしない
            nn::svc::CloseHandle(mHandle);
        }

        nn::svc::Handle GetHandle()
        {
            return mHandle;
        }

    private:
        nn::svc::Handle mHandle;
};

class TestMemoryPermission
{
    public:
        TestMemoryPermission(uintptr_t addr, size_t size, nn::svc::MemoryPermission permission)
            : mAddr(addr), mSize(size), mPermission(permission)
        {
            nn::Result result = nn::svc::SetMemoryPermission(mAddr, mSize, mPermission);
            if (result.IsFailure())
            {
                NN_LOG("result module: %d, description: %d\n", result.GetModule(), result.GetDescription());
            }
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~TestMemoryPermission()
        {
            nn::Result result =
                nn::svc::SetMemoryPermission(mAddr, mSize, nn::svc::MemoryPermission_ReadWrite);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        void ChangePermission(nn::svc::MemoryPermission permission)
        {
            nn::Result result = nn::svc::SetMemoryPermission(mAddr, mSize, permission);
            NN_ASSERT_RESULT_SUCCESS(result);
            mPermission = permission;
        }

        uintptr_t GetAddress() const
        {
            return mAddr;
        }
        size_t GetSize() const
        {
            return mSize;
        }
        nn::svc::MemoryPermission GetPermission() const
        {
            return mPermission;
        }

    private:
        uintptr_t mAddr;
        size_t mSize;
        nn::svc::MemoryPermission mPermission;
};

class TestMapMemory
{
    public:
        TestMapMemory(
            uintptr_t toAddr, uintptr_t fromAddr, size_t size)
            : mToAddr(toAddr), mFromAddr(fromAddr), mSize(size)
        {
            nn::Result result = nn::svc::MapMemory(toAddr, fromAddr, size);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~TestMapMemory()
        {
            nn::Result result =
                nn::svc::UnmapMemory(mToAddr, mFromAddr, mSize);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        uintptr_t GetToAddr() const
        {
            return mToAddr;
        }
        uintptr_t GetFromAddr() const
        {
            return mFromAddr;
        }
        size_t GetSize() const
        {
            return mSize;
        }

    private:
        uintptr_t mToAddr;
        uintptr_t mFromAddr;
        size_t mSize;
};

class TestSharedMemory
{
    public:
        TestSharedMemory(
            nn::svc::Handle handle, uintptr_t addr, size_t size,
            nn::svc::MemoryPermission permission)
            : mSharedHandle(handle), mAddr(addr), mSize(size), mPermission(permission)
        {
            nn::Result result = nn::svc::MapSharedMemory(mSharedHandle, mAddr, mSize, mPermission);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~TestSharedMemory()
        {
            nn::Result result = nn::svc::UnmapSharedMemory(mSharedHandle, mAddr, mSize);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        nn::svc::Handle GetHandle() const
        {
            return mSharedHandle;
        }
        uintptr_t GetAddr() const
        {
            return mAddr;
        }
        size_t GetSize() const
        {
            return mSize;
        }
        nn::svc::MemoryPermission GetPermission() const
        {
            return mPermission;
        }

    private:
        nn::svc::Handle mSharedHandle;
        uintptr_t mAddr;
        size_t mSize;
        nn::svc::MemoryPermission mPermission;
};

class TestProcess
{
    public:
        TestProcess(int32_t pageNum, nn::Bit32 paramFlag = 0,
                    nn::Bit32* flagArray = NULL, int32_t flagNum = 0)
            : mPageNum(pageNum), mParamFlag(paramFlag), mFlagArray(flagArray)
            , mFlagNum(flagNum), mIsClosed(false)
            , mPriority(TestLowestThreadPriority), mIdealCore(g_ProcessIdealCore)
            , mStackSize(DefaultStackSize)
        {
            std::memset(&mParam, 0, sizeof(nn::svc::CreateProcessParameter));
            std::strncpy(mParam.name, "test", 11);
            mParam.version        = 0xbabeface;
            mParam.programId      = 0xdeadbeef0badcafeull;
            mParam.memoryNumPages = mPageNum;
            mParam.flags          = mParamFlag;
            mIs64Bit = (mParamFlag & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
            if ((mParam.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
            {
                mParam.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
            }
            else
            {
                mParam.memoryAddress = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            }

            nn::Bit32 flags[DefaultCapabilityFlagNum];
            if (!mFlagArray)
            {
                mFlagArray = flags;
                SetDefaultCapability(flags, DefaultCapabilityFlagNum);
                mFlagNum = DefaultCapabilityFlagNum;
            }
            nn::Result result = nn::svc::CreateProcess(&mHandle, mParam, mFlagArray, mFlagNum);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~TestProcess()
        {
            Close();
        }

        void Start() const
        {
            nn::Result result;
            result = nn::svc::StartProcess(
                        mHandle, mPriority, mIdealCore,
                        mStackSize);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        void Wait() const
        {
            nn::Result result;
            nn::svc::ProcessState state;
            int index;
            int64_t info;

            for(;;)
            {
                result = nn::svc::WaitSynchronization(&index, &mHandle, 1, -1);
                NN_ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::GetProcessInfo(&info, mHandle, nn::svc::ProcessInfoType_State);
                NN_ASSERT_RESULT_SUCCESS(result);
                state = static_cast<nn::svc::ProcessState>(info);
                result = nn::svc::ResetSignal(mHandle);
                if (result.IsFailure())
                {
                    while (state == nn::svc::ProcessState_Terminating)
                    {
                        nn::svc::SleepThread(100 * 1000 * 1000);
                        result = nn::svc::GetProcessInfo(&info, mHandle, nn::svc::ProcessInfoType_State);
                        NN_ASSERT_RESULT_SUCCESS(result);
                        state = static_cast<nn::svc::ProcessState>(info);
                    }
                    NN_ASSERT(state == nn::svc::ProcessState_Terminated);
                    break;
                }
            }
        }

        void Close()
        {
            if (mIsClosed)
            {
                return;
            }

            mIsClosed = true;
            // 閉じられているかもしれないので、Result のチェックはしない
            nn::svc::TerminateProcess(mHandle);
            nn::svc::CloseHandle(mHandle);
        }

        nn::svc::Handle GetHandle() const
        {
            return mHandle;
        }
        uintptr_t GetCodeAddress() const
        {
            return mParam.memoryAddress;
        }
        uintptr_t GetCodeAreaSize() const
        {
            return mPageNum * 0x1000;
        }

        bool IsClosed() const
        {
            return mIsClosed;
        }

        void SetPriority(int32_t priority)
        {
            mPriority = priority;
        }

        void SetIdealCore(int32_t idealCore)
        {
            mIdealCore = idealCore;
        }

        void SetStackSize(int32_t stackSize)
        {
            mStackSize = stackSize;
        }

        bool Is64Bit() const
        {
            return mIs64Bit;
        }

    private:
        nn::svc::Handle mHandle;
        nn::svc::CreateProcessParameter mParam;
        int32_t mPageNum;
        nn::Bit32 mParamFlag;
        nn::Bit32* mFlagArray;
        int32_t mFlagNum;
        bool mIsClosed;
        int32_t mPriority;
        int32_t mIdealCore;
        size_t mStackSize;
        bool mIs64Bit;
};

class TestMemoryBlock
{
    public:
        explicit TestMemoryBlock(uintptr_t addr)
        {
            nn::Result result = nn::svc::QueryMemory(&mMemInfo, &mPageInfo, addr);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        bool CheckPermission(nn::svc::MemoryPermission permission)
        {
            return mMemInfo.permission == permission;
        }

        bool CheckState(nn::svc::MemoryState state)
        {
            return mMemInfo.state == state;
        }

        nn::svc::MemoryInfo GetMemoryInfo() const
        {
            return mMemInfo;
        }

    private:
        nn::svc::MemoryInfo mMemInfo;
        nn::svc::PageInfo   mPageInfo;
};

class TestChangePriority
{
    public:
        explicit TestChangePriority(int32_t priority = nn::svc::HighestThreadPriority)
            : mCurPriority(priority)
        {
            nn::Result result;
            result = nn::svc::GetThreadPriority(
                        &mOrigPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
            NN_ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::SetThreadPriority(
                        nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, mCurPriority);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~TestChangePriority()
        {
            nn::Result result;
            result = nn::svc::SetThreadPriority(
                        nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, mOrigPriority);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

    private:
        int32_t mCurPriority;
        int32_t mOrigPriority;
};

class TestChangeCore
{
    public:
        explicit TestChangeCore(int32_t coreNo)
            : mCurCore(coreNo)
        {
            nn::Result result;
            nn::Bit64 affinity;
            result = nn::svc::GetThreadCoreMask(
                        &mOrigCore, &affinity, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
            NN_ASSERT_RESULT_SUCCESS(result);
            affinity = 1ul << mCurCore;
            result = nn::svc::SetThreadCoreMask(
                        nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, mCurCore, affinity);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~TestChangeCore()
        {
            nn::Result result;
            nn::Bit64 affinity = 1ul << mOrigCore;
            result = nn::svc::SetThreadCoreMask(
                        nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, mOrigCore, affinity);
            NN_ASSERT_RESULT_SUCCESS(result);
        }

    private:
        int32_t mCurCore;
        int32_t mOrigCore;
};

class TestResourceEnv
{
    public:
        TestResourceEnv()
            : mPriority(nn::svc::HighestThreadPriority + 16), mCore(0)
        {
            nn::Result result;
            nn::Bit64 affinity;
            int32_t core;
            int32_t priority;
            result = nn::svc::GetThreadCoreMask(&core, &affinity, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
            NN_ASSERT_RESULT_SUCCESS(result);
            NN_ASSERT(core == 0);
            result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
            NN_ASSERT_RESULT_SUCCESS(result);
            NN_ASSERT(priority == nn::svc::HighestThreadPriority + 16);
        }
    private:
        TestChangePriority mPriority;
        TestChangeCore mCore;
};

class AutoDetachDevice
{
    public:
        AutoDetachDevice(nn::svc::Handle handle, nn::svc::DeviceName name)
            : m_Handle(handle), m_Name(name)
        {
        }

        ~AutoDetachDevice()
        {
            nn::Result result;
            result = nn::svc::DetachDeviceAddressSpace(m_Name, m_Handle);
            NN_ASSERT_RESULT_SUCCESS(result);
        }
    private:
        nn::svc::Handle m_Handle;
        nn::svc::DeviceName m_Name;
};

inline void GetMemoryInfo(nn::svc::MemoryInfo* pOut, uintptr_t addr)
{
    nn::svc::PageInfo pageInfo;
    nn::Result result = nn::svc::QueryMemory(pOut, &pageInfo, addr);
    NN_ASSERT_RESULT_SUCCESS(result);
}

inline void ShowMemoryInfo(nn::svc::MemoryInfo info)
{
    NN_LOG("addr: 0x%llx\n", info.baseAddress);
    NN_LOG("size: 0x%llx\n", info.size);
    NN_LOG("state: %d\n", info.state);
    NN_LOG("permission: %d\n", info.permission);
    NN_LOG("attribute: %d\n", info.attribute);
}


inline bool IsInAslrRegion(uintptr_t addr)
{
    return !((g_ReservedAreaBegin <= addr && g_ReservedAreaEnd >= addr) ||
             (g_HeapAreaBegin <= addr && g_HeapAreaEnd >= addr) ||
             (addr > LargeRegionEnd));
}

inline bool IsInProcessAslrRegion(nn::svc::Handle processHandle, uint64_t addr)
{
    nn::Result result;
    nn::Bit64 reservedAddr;
    nn::Bit64 reservedSize;
    nn::Bit64 heapAddr;
    nn::Bit64 heapSize;

    result = nn::svc::GetInfo(
            &reservedAddr, nn::svc::InfoType_ReservedRegionAddress,
            processHandle, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetInfo(
            &reservedSize, nn::svc::InfoType_ReservedRegionSize,
            processHandle, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetInfo(
            &heapAddr, nn::svc::InfoType_HeapRegionAddress,
            processHandle, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetInfo(
            &heapSize, nn::svc::InfoType_HeapRegionSize,
            processHandle, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    return  !((reservedAddr <= addr && reservedAddr + reservedSize - 1 >= addr) ||
            (heapAddr <= addr & heapAddr + heapSize - 1 >= addr) ||
            (addr > LargeRegionEnd));
}

inline void SetExecuteCode(nn::svc::Handle processHandle, const uint32_t* codePtr, size_t codeSize,
                    uintptr_t mapAddr, size_t mapSize)
{
    nn::Result result = nn::svc::MapProcessMemory(g_FreeAreaBegin, processHandle, mapAddr, mapSize);
    NN_ASSERT_RESULT_SUCCESS(result);

    uint32_t* ptr = reinterpret_cast<uint32_t*>(g_FreeAreaBegin);
    ::std::memcpy(ptr, codePtr, codeSize);

    result = nn::svc::UnmapProcessMemory(g_FreeAreaBegin, processHandle, mapAddr, mapSize);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetProcessMemoryPermission(
            processHandle, mapAddr, mapSize,
            nn::svc::MemoryPermission_ReadExecute);
    NN_ASSERT_RESULT_SUCCESS(result);
}

inline void AssignExitCode(nn::svc::Handle processHandle, uintptr_t mapAddr, size_t mapSize)
{
    uint32_t code = 0xef000000 | NN_SVC_ID_EXIT_PROCESS; // svc NN_SVC_ID_EXIT_PROCESS
    SetExecuteCode(processHandle, &code, sizeof(uint32_t), mapAddr, mapSize);
}

const uint32_t SetHeapSizeCodes32[] = {
    0xe24dd004, //sub sp, sp, #4
    0xe1a0000d, //mov r0, sp
    0xe3a01602, //mov r1, #2097152    ; 0x200000
    0xe52d0004, //push    {r0}
    0xef000001, //svc 0x00000001 (SetHeapSize)
    0xe28dd008, //add sp, sp, #8
    0xeafffffe, // 自分自身へのブランチ命令
    0xef000000 | NN_SVC_ID_EXIT_PROCESS // 念のため
};

inline void AssignSetHeapSizeCode32(nn::svc::Handle handle, uintptr_t mapAddr, size_t mapSize)
{
    SetExecuteCode(handle, SetHeapSizeCodes32, sizeof(SetHeapSizeCodes32), mapAddr, mapSize);
}

class ConsumeResource
{
public:
    ConsumeResource()
        : m_Count(0), m_IsConsumed(false)
    {
    }

    virtual ~ConsumeResource() {}
    virtual void Consume() = 0;
    virtual void Release() = 0;
    int GetCount() const { return m_Count; }
protected:
    int32_t m_Count;
    bool m_IsConsumed;
};

class ConsumeHeapMemory : public ConsumeResource
{
public:
    ConsumeHeapMemory();
    virtual ~ConsumeHeapMemory();
    virtual void Consume();
    virtual void Release();
    uintptr_t GetAddress() const { return m_HeapAddr; }
    size_t GetSize() const { return m_HeapSize; }
private:
    uintptr_t   m_HeapAddr;
    size_t      m_HeapSize;
};

class ConsumeHandleBase : public ConsumeResource
{
public:
    ConsumeHandleBase()
        : m_pHandles(nullptr)
    {
    }

    virtual ~ConsumeHandleBase() {}

    virtual void Release();
    nn::svc::Handle* GetHandles() { return m_pHandles; }

protected:
    nn::svc::Handle* m_pHandles;
};

class ConsumeHandle : public ConsumeHandleBase
{
public:
    ConsumeHandle();
    virtual ~ConsumeHandle();
    virtual void Consume();

private:
    const int32_t NumHandle = NumMaxHandle + 1;

    TestHeap m_Heap;
};

class ConsumeThread : public ConsumeHandleBase
{
public:
    ConsumeThread();
    virtual ~ConsumeThread();
    virtual void Consume();
private:
    const int32_t NumHandle = NN_KERN_SLAB_OBJ_NUM_THREAD + 1;
    TestHeap m_Heap;
};

class ConsumeMemoryControlBlock : public ConsumeResource
{
public:
    ConsumeMemoryControlBlock();
    virtual ~ConsumeMemoryControlBlock();
    virtual void Consume();
    virtual void Release();
private:
    nn::svc::Handle m_Process;
};

class ConsumeEvent : public ConsumeHandleBase
{
public:
    ConsumeEvent();
    virtual ~ConsumeEvent();
    virtual void Consume();
private:
    const int32_t NumHandle = NN_KERN_SLAB_OBJ_NUM_EVENT + 1;
    const size_t MsgSize = 0x1000;

    nn::svc::Handle m_ServerSession;
    nn::svc::Handle m_ClientSession;
    uintptr_t m_MsgBuffer;
    TestHeap m_Heap;
};

class ConsumeSharedMemory : public ConsumeHandleBase
{
public:
    ConsumeSharedMemory();
    virtual ~ConsumeSharedMemory();
    virtual void Consume();
private:
    const int32_t NumHandle = NN_KERN_SLAB_OBJ_NUM_SHARED_MEMORY + 1;

    TestHeap m_Heap;
};

class ConsumeTransferMemory : public ConsumeHandleBase
{
public:
    ConsumeTransferMemory();
    virtual ~ConsumeTransferMemory();
    virtual void Consume();
private:
    const int32_t NumHandle = NN_KERN_SLAB_OBJ_NUM_TRANSFER_MEMORY + 1;
    const size_t AreaSize = 0x1000;
    const nn::svc::MemoryPermission Perm = nn::svc::MemoryPermission_ReadWrite;
    uintptr_t m_Addr;
    TestHeap m_Heap;

};

class ConsumeDeviceAddressSpace : public ConsumeHandleBase
{
public:
    ConsumeDeviceAddressSpace();
    virtual ~ConsumeDeviceAddressSpace();
    virtual void Consume();
private:
    const int32_t NumHandle = NN_KERN_SLAB_OBJ_NUM_DEVICE_ADDRESS_SPACE + 1;
    TestHeap m_Heap;
};

class ConsumeSession : public ConsumeHandleBase
{
public:
    ConsumeSession();
    virtual ~ConsumeSession();
    virtual void Consume();
    virtual void Release();
private:
    const int32_t NumHandle = NN_KERN_SLAB_OBJ_NUM_SESSION + 1;
    const char* PortName = "ConsSess";
    TestHeap m_Heap;
    NamedPortManager m_NamedPort;
};

class ConsumeProcess : public ConsumeHandleBase
{
public:
    ConsumeProcess();
    virtual ~ConsumeProcess();
    virtual void Consume();
private:
    const int32_t NumHandle = NN_KERN_SLAB_OBJ_NUM_PROCESS + 1;
    TestHeap m_Heap;

};

class ConsumePort : public ConsumeHandleBase
{
public:
    ConsumePort();
    virtual ~ConsumePort();
    virtual void Consume();
    virtual void Release();
private:
    const int32_t NumHandle = (NN_KERN_SLAB_OBJ_NUM_PORT + 1) * 2;
    TestHeap m_Heap;
};

class ConsumeNamedPort : public ConsumeHandleBase
{
public:
    ConsumeNamedPort();
    virtual ~ConsumeNamedPort();
    virtual void Consume();
    virtual void Release();
private:
    const int32_t NumHandle = NN_KERN_SLAB_OBJ_NUM_OBJECT_NAME + 1;
    TestHeap m_Heap;
};

class ConsumeLightSession : public ConsumeHandleBase
{
public:
    ConsumeLightSession();
    virtual ~ConsumeLightSession();
    virtual void Consume();
    virtual void Release();
private:
    const int32_t NumHandle = (NN_KERN_SLAB_OBJ_NUM_LIGHT_SESSION + 1) * 2;
    TestHeap m_Heap;
};

class ConsumeDebug : public ConsumeHandleBase
{
public:
    ConsumeDebug();
    virtual ~ConsumeDebug();
    virtual void Consume();
    virtual void Release();
private:
    const int32_t NumHandle = (NN_KERN_SLAB_OBJ_NUM_DEBUG + 1) * 2;
    TestHeap m_Heap;
};

class ConsumeInterruptEvent : public ConsumeHandleBase
{
public:
    ConsumeInterruptEvent() {}
    virtual ~ConsumeInterruptEvent() {}
    virtual void Consume() {}
    virtual void Release() {}
};

class ConsumeDma : public ConsumeHandleBase
{
public:
    ConsumeDma() {}
    virtual ~ConsumeDma() {}
    virtual void Consume() {}
    virtual void Release() {}
};

class TestThreadLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestThreadLeak()
        {
            ConsumeThread thread;
            m_Before = thread.GetCount();
        }

        ~TestThreadLeak()
        {
            nn::svc::SleepThread(100 * 1000 * 1000);
            {
                ConsumeThread thread;
                m_After = thread.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestMemoryControlBlockLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestMemoryControlBlockLeak()
        {
            ConsumeMemoryControlBlock block;
            m_Before = block.GetCount();
        }

        ~TestMemoryControlBlockLeak()
        {
            {
                ConsumeMemoryControlBlock block;
                m_After = block.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestEventLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestEventLeak()
        {
            ConsumeEvent event;
            m_Before = event.GetCount();
        }

        ~TestEventLeak()
        {
            {
                ConsumeEvent event;
                m_After = event.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestSharedMemoryLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestSharedMemoryLeak()
        {
            ConsumeSharedMemory shared;
            m_Before = shared.GetCount();
        }

        ~TestSharedMemoryLeak()
        {
            {
                ConsumeSharedMemory shared;
                m_After = shared.GetCount();
            }

#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestTransferMemoryLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestTransferMemoryLeak()
        {
            ConsumeTransferMemory transfer;
            m_Before = transfer.GetCount();
        }

        ~TestTransferMemoryLeak()
        {
            {
                ConsumeTransferMemory transfer;
                m_After = transfer.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestDeviceAddressSpaceLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestDeviceAddressSpaceLeak()
        {
            ConsumeDeviceAddressSpace das;
            m_Before = das.GetCount();
        }

        ~TestDeviceAddressSpaceLeak()
        {
            {
                ConsumeDeviceAddressSpace das;
                m_After = das.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestSessionLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestSessionLeak()
        {
            ConsumeSession session;
            m_Before = session.GetCount();
        }

        ~TestSessionLeak()
        {
            {
                ConsumeSession session;
                m_After = session.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestProcessLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestProcessLeak()
        {
            ConsumeProcess process;
            m_Before = process.GetCount();
        }

        ~TestProcessLeak()
        {
            nn::svc::SleepThread(100 * 1000 * 1000);
            {
                ConsumeProcess process;
                m_After = process.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestPortLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestPortLeak()
        {
            ConsumePort port;
            m_Before = port.GetCount();
        }

        ~TestPortLeak()
        {
            {
                ConsumePort port;
                m_After = port.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestNamedPortLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestNamedPortLeak()
        {
            ConsumeNamedPort namedPort;
            m_Before = namedPort.GetCount();
        }

        ~TestNamedPortLeak()
        {
            {
                ConsumeNamedPort namedPort;
                m_After = namedPort.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestLightSessionLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestLightSessionLeak()
        {
            ConsumeLightSession lightSession;
            m_Before = lightSession.GetCount();
        }

        ~TestLightSessionLeak()
        {
            {
                ConsumeLightSession lightSession;
                m_After = lightSession.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestInterruptEventLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestInterruptEventLeak()
        {
            ConsumeInterruptEvent interrupt;
            m_Before = interrupt.GetCount();
        }

        ~TestInterruptEventLeak()
        {
            {
                ConsumeInterruptEvent interrupt;
                m_After = interrupt.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestDebugLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestDebugLeak()
        {
            ConsumeDebug debug;
            m_Before = debug.GetCount();
        }

        ~TestDebugLeak()
        {
            {
                ConsumeDebug debug;
                m_After = debug.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

class TestDmaLeak
{
    private:
        int32_t m_Before;
        int32_t m_After;

    public:
        TestDmaLeak()
        {
            ConsumeDma dma;
            m_Before = dma.GetCount();
        }

        ~TestDmaLeak()
        {
            {
                ConsumeDma dma;
                m_After = dma.GetCount();
            }
#ifdef ENABLE_LEAK_TEST
            NN_ASSERT(m_Before == m_After);
#endif // ENABLE_LEAK_TEST
        }
};

