﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "util_TestProcess.h"
#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Version.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_BitPack.h>
#include <cstring>

nn::Bit32 CapabilitySignature(int32_t entryNo)
{
    return static_cast<nn::Bit32>((1 << entryNo) - 1);
}

void SetDefaultParam(nn::svc::CreateProcessParameter* param)
{
    std::memset(param, 0, sizeof(nn::svc::CreateProcessParameter));
    std::strncpy(param->name, "test", 11);
    param->version        = 0xbabeface;
    param->programId      = 0xdeadbeef0badcafeull;
    param->memoryAddress  = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
    param->memoryNumPages = 0x00000100;
    param->flags          = 0x00000000;
}

void SetDefaultCapability(nn::Bit32* pOut, int32_t maxFlagNum)
{
    NN_ASSERT(maxFlagNum >= DefaultCapabilityFlagNum);
    MakeNo3Flag(&pOut[0], nn::svc::LowestThreadPriority, nn::svc::HighestThreadPriority, 0, NumCore - 1);
    MakeNo4Flag(&pOut[1], NN_SVC_ID_EXIT_PROCESS);
    MakeNo14Flag(&pOut[2], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
    MakeNo15Flag(&pOut[3], 128);
}

void MakeNo3Flag(nn::Bit32* pOut, uint8_t lowestPriority, uint8_t highestPriority,
                 uint16_t minCore, uint16_t maxCore)
{
    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);
    flag->Set<No3Signature>(CapabilitySignature(3));
    flag->Set<No3LowestPriority>(lowestPriority);
    flag->Set<No3HighestPriority>(highestPriority);
    flag->Set<No3MinCoreNumber>(minCore);
    flag->Set<No3MaxCoreNumber>(maxCore);
}

void MakeNo4Flag(nn::Bit32* pOut, int32_t syscallNo)
{
    int32_t index = syscallNo / 24;
    int32_t shift = syscallNo % 24;

    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);
    flag->Set<No4Signature>(CapabilitySignature(4));
    flag->Set<No4SvcFlags>((1 << shift));
    flag->Set<No4Index>(index);
}

void MakeNo4Flag(nn::Bit32* pOut, int32_t* pFlagNum,
                 int32_t maxFlagNum, int32_t* syscallNo, int32_t syscallNum)
{
    *pFlagNum = (1 << No4Index::Width) - 1;
    NN_ASSERT(maxFlagNum > *pFlagNum);

    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);
    for (int32_t i = 0; i < *pFlagNum; i++)
    {
        flag[i].Set<No4Signature>(CapabilitySignature(4));
        flag[i].Set<No4SvcFlags>(0);
        flag[i].Set<No4Index>(i);
    }

    for (int32_t i = 0; i < syscallNum; i++)
    {
        int32_t index = syscallNo[i] / 24;
        int32_t shift = syscallNo[i] % 24;

        nn::Bit32 field = flag[index].Get<No4SvcFlags>();
        flag[index].Set<No4SvcFlags>(field | (1 << shift));
    }
}

void MakeNo6Flag(nn::Bit32* pOut, int32_t maxFlagNum, uint64_t phyAddr, size_t mapSize,
                 bool isWrite, bool isIo)
{
    NN_ASSERT(maxFlagNum > 0);

    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);

    flag->Set<No6Signature>(CapabilitySignature(6));
    flag->Set<No6Address>(phyAddr >> 12);
    flag->Set<No6Flag>(!isWrite);

    if (maxFlagNum > 1)
    {
        flag++;

        flag->Set<No6Signature>(CapabilitySignature(6));
        flag->Set<No6Address>(mapSize >> 12);
        flag->Set<No6Flag>(!isIo);
    }
}

void MakeNo7Flag(nn::Bit32* pOut, uint64_t phyAddr)
{
    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);
    flag->Set<No7Signature>(CapabilitySignature(7));
    flag->Set<No7Address>(phyAddr >> 12);
}

void MakeNo11Flag(nn::Bit32* pOut, int32_t* pFlagNum,
                  int32_t maxFlagNum, const uint16_t* names, int32_t nameNum)
{
    const int16_t Padding = 1023;

    NN_ASSERT(maxFlagNum > 0);
    NN_ASSERT(nameNum > 0);

    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);
    *pFlagNum = 0;
    int32_t i = 0;

    if (nameNum % 2 > 0)
    {
        flag->Set<No11Signature>(CapabilitySignature(11));
        flag->Set<No11Interrupt1>(names[0]);
        flag->Set<No11Interrupt2>(Padding);
        i++;
        (*pFlagNum)++;
        flag++;
    }

    for (;i < nameNum; i += 2, flag++, (*pFlagNum)++)
    {
        flag->Set<No11Signature>(CapabilitySignature(11));
        flag->Set<No11Interrupt1>(names[i]);
        flag->Set<No11Interrupt2>(names[i + 1]);
    }
}

void MakeNo13Flag(nn::Bit32* pOut, uint32_t programType, uint32_t reserved)
{
    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);

    flag->Set<No13Signature>(CapabilitySignature(13));
    flag->Set<No13ProgramType>(programType);
    flag->Set<No13Reserved>(reserved);
}

void MakeNo14Flag(nn::Bit32* pOut, uint16_t major, uint8_t minor)
{
    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);

    flag->Set<No14Signature>(CapabilitySignature(14));
    flag->Set<No14MajorVersion>(major);
    flag->Set<No14MinorVersion>(minor);
}

void MakeNo15Flag(nn::Bit32* pOut, uint16_t data)
{
    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);

    flag->Set<No15Signature>(CapabilitySignature(15));
    flag->Set<No15DataField>(data);
}

void MakeNo16Flag(nn::Bit32* pOut, uint16_t data)
{
    NN_ASSERT(data >= 0);
    nn::util::BitPack32* flag = reinterpret_cast<nn::util::BitPack32*>(pOut);
    flag->Set<No16Signature>(CapabilitySignature(16));
    flag->Set<No16DataField>(data);
}

void WaitProcess(nn::svc::Handle handle)
{
    nn::Result result;
    nn::svc::ProcessState state;
    int index;
    int64_t info;

    for(;;)
    {
        result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::ResetSignal(handle);
        if (result.IsFailure())
        {
            result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
            NN_ASSERT_RESULT_SUCCESS(result);
            state = static_cast<nn::svc::ProcessState>(info);

            while (state == nn::svc::ProcessState_Terminating)
            {
                nn::svc::SleepThread(100 * 1000 * 1000);
                result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
                NN_ASSERT_RESULT_SUCCESS(result);
                state = static_cast<nn::svc::ProcessState>(info);
            }
            NN_ASSERT(state == nn::svc::ProcessState_Terminated);
            break;
        }
    }
}

uint64_t GetIoAddress(uint64_t size)
{
    uint64_t phyAddr = PhysMemoryEnd;
    if (PhysMemoryEnd != 0 && phyAddr + size > phyAddr)
    {
        return phyAddr;
    }

    phyAddr = PhysMemoryBegin - size;
    if (phyAddr < PhysMemoryBegin)
    {
        return phyAddr;
    }

    return 0;
}

