﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "test_AliasCode.h"
#include "util_TestMemory.h"
#include "util_TestLoader.h"
#include "util_TestIpc.h"
#include "util_TestProcess.h"
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <new>

extern char BinAliasCode_begin[];
extern char BinAliasCode_end[];
extern "C" void nnMain();

namespace {
char g_Buffer[TestMemorySize] __attribute__((aligned(0x1000)));
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));
char g_ServerBuffer[TestMemorySize] __attribute__((aligned(0x1000)));
char g_ClientBuffer[TestMemorySize] __attribute__((aligned(0x1000)));
char g_ClientMapArea[TestMemorySize] __attribute__((aligned(0x1000)));

struct TestIpcData
{
    nn::svc::Handle handle;
    int ipcAttr;
    const void* mapData;
    size_t mapSize;
};

void SendIpcThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    TestIpcData* testData = reinterpret_cast<TestIpcData*>(arg);
    nn::svc::Handle handle = testData->handle;
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_ClientBuffer);
    size_t msgSize = sizeof(g_ClientBuffer);

    // Recv 転送
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));
    int offset = ipcMsg.Set(
                    nn::svc::ipc::MessageBuffer::MessageHeader(
                        0x0001, 0, 0, 0, 1, 0, 0, 0));
    nn::svc::ipc::MessageBuffer::MapData mapData(testData->mapData, testData->mapSize);
    offset = ipcMsg.Set(offset, mapData);
    nn::Bit32* pBuffer = reinterpret_cast<nn::Bit32*>(pMsgBuffer);
    pBuffer[offset - 1] |= testData->ipcAttr;

    result = nn::svc::SendSyncRequestWithUserBuffer(pMsgBuffer, msgSize, handle);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void SendIpcThreadForSend(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    TestIpcData* testData = reinterpret_cast<TestIpcData*>(arg);
    nn::svc::Handle handle = testData->handle;
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_ClientBuffer);
    size_t msgSize = sizeof(g_ClientBuffer);

    // Send 転送
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));
    int offset = ipcMsg.Set(
                    nn::svc::ipc::MessageBuffer::MessageHeader(
                        0x0001, 0, 0, 1, 0, 0, 0, 0));
    nn::svc::ipc::MessageBuffer::MapData mapData(testData->mapData, testData->mapSize);
    offset = ipcMsg.Set(offset, mapData);
    nn::Bit32* pBuffer = reinterpret_cast<nn::Bit32*>(pMsgBuffer);
    pBuffer[offset - 1] |= testData->ipcAttr;

    result = nn::svc::SendSyncRequestWithUserBuffer(pMsgBuffer, msgSize, handle);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    NN_ASSERT_RESULT_SUCCESS(result);
}

// Large 領域の Free 領域がそもそも存在しない場合は False
// Size 分のFree 領域が見つからなかった場合は ASSERT
bool GetLargeFreeAreaImpl(uint64_t* pOutAddr, uint64_t size, nn::svc::Handle process, uint64_t largeBegin, uint64_t largeEnd)
{
    uint64_t largeSize = largeEnd - largeBegin + 1;

    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    nn::Result result;

    nn::Bit64 tmpAddr;
    nn::Bit64 tmpSize;
    uint64_t reservedBegin;
    uint64_t reservedEnd;
    result = nn::svc::GetInfo(
            &tmpAddr, nn::svc::InfoType_ReservedRegionAddress,
            process, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    reservedBegin = tmpAddr;

    result = nn::svc::GetInfo(
            &tmpSize, nn::svc::InfoType_ReservedRegionSize,
            process, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    // AddressSpace32BitNoReserved の時にサイズが 0 になる
    // その場合、 reservedEnd が reservedBegin より小さくなるので、レンジの比較にいつも失敗するようになる
    reservedEnd = tmpAddr + tmpSize - 1;
    NN_ASSERT(largeSize >= tmpSize);
    largeSize -= tmpSize;

    uint64_t heapBegin;
    uint64_t heapEnd;
    result = nn::svc::GetInfo(
            &tmpAddr, nn::svc::InfoType_HeapRegionAddress,
            process, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    heapBegin = tmpAddr;

    result = nn::svc::GetInfo(
            &tmpSize, nn::svc::InfoType_HeapRegionSize,
            process, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(tmpSize != 0);
    heapEnd = tmpAddr + tmpSize - 1;
    NN_ASSERT(largeSize >= tmpSize);
    largeSize -= tmpSize;

    if (largeSize == 0)
    {
        NN_LOG("*** WARNING: Cannot find Large Free Area ***\n");
        return false;
    }

    NN_ASSERT(largeSize >= size);

    for(uint64_t addr = largeBegin; addr >= largeBegin && addr <= largeEnd;)
    {
        if (addr >= reservedBegin && addr <= reservedEnd)
        {
            addr = reservedEnd + 1;
            continue;
        }
        if (addr >= heapBegin && addr <= heapEnd)
        {
            addr = heapEnd + 1;
            continue;
        }
        result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, process, addr);
        NN_ASSERT_RESULT_SUCCESS(result);
        if (blockInfo.state == nn::svc::MemoryState_Free)
        {
            uint64_t blockEnd = blockInfo.baseAddress + blockInfo.size - 1;
            if (blockEnd > largeEnd)
            {
                blockEnd = largeEnd;
            }

            if ((blockEnd + 1) - addr >= size)
            {
                *pOutAddr = addr;
                return true;
            }
        }
        addr = blockInfo.baseAddress + blockInfo.size;
    }
    // 見つからなかったら ASSERT
    NN_ASSERT(false);
    // コンパイルを通すため
    return false;
}
}

#ifdef ASSERT_TRUE
void CheckMemory(
        uintptr_t addr,
        nn::svc::MemoryState state,
        nn::svc::MemoryPermission permission,
        uint32_t attribute)
{
    TestMemoryAreaObject assertObj(addr);
    nn::svc::MemoryInfo blockInfo;
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == state);
    ASSERT_TRUE(blockInfo.permission == permission);
    ASSERT_TRUE(blockInfo.attribute == attribute);
    assertObj.Cancel();
}

void CheckProcessMemory(
        nn::svc::Handle processHandle,
        uintptr_t addr,
        nn::svc::MemoryState state,
        nn::svc::MemoryPermission permission,
        uint32_t attribute)
{
    TestMemoryAreaObject assertObj(addr, processHandle);
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(blockInfo.state == state);
    ASSERT_TRUE(blockInfo.permission == permission);
    ASSERT_TRUE(blockInfo.attribute == attribute);
    assertObj.Cancel();
}
#endif

void InitTestMemory()
{
    nn::Result result;

    // Reserved 領域の取得
    nn::Bit64 tmpAddr;
    nn::Bit64 tmpSize;
    result = nn::svc::GetInfo(
            &tmpAddr, nn::svc::InfoType_ReservedRegionAddress,
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    g_ReservedAreaBegin = static_cast<uintptr_t>(tmpAddr);

    result = nn::svc::GetInfo(
            &tmpSize, nn::svc::InfoType_ReservedRegionSize,
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(tmpSize != 0);
    g_ReservedAreaEnd = g_ReservedAreaBegin + tmpSize - 1;

    // ヒープ領域の取得
    result = nn::svc::GetInfo(
            &tmpAddr, nn::svc::InfoType_HeapRegionAddress,
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    g_HeapAreaBegin = static_cast<uintptr_t>(tmpAddr);

    result = nn::svc::GetInfo(
            &tmpSize, nn::svc::InfoType_HeapRegionSize,
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(tmpSize != 0);
    g_HeapAreaEnd = g_HeapAreaBegin + tmpSize - 1;


    // Free 領域の取得
    size_t freeSize;
    NN_ASSERT(GetFreeArea(&g_FreeAreaBegin, &freeSize, 0x1000000));
    g_FreeAreaEnd = g_FreeAreaBegin + freeSize;
}

bool GetFreeArea(uintptr_t* pOutAddr, size_t* pOutSize, size_t size)
{
    nn::svc::MemoryInfo blockInfo;
    for (uintptr_t addr = SmallRegionBegin + NN_SVC_ADDR_NULL_GUARD_SIZE; addr <= SmallRegionEnd;)
    {
        GetMemoryInfo(&blockInfo, addr);
        if (blockInfo.state == nn::svc::MemoryState_Free && blockInfo.baseAddress == addr)
        {
            // size == 0 なら最初に見つかったブロックを渡す
            if (size == 0 || size <= blockInfo.size)
            {
                *pOutAddr = blockInfo.baseAddress;
                *pOutSize = size == 0 ? blockInfo.size : size;
                return true;
            }
        }
        addr = blockInfo.baseAddress + blockInfo.size;
    }
    return false;
}

bool GetProcessLargeFreeArea(uint64_t* pOutAddr, uint64_t size, nn::svc::Handle process, bool is64Bit)
{
    uint64_t largeBegin = (is64Bit ? NN_SVC_ADDR_MEMORY_REGION_LARGE64_BEGIN : NN_SVC_ADDR_MEMORY_REGION_LARGE32_BEGIN);
    uint64_t largeEnd = (is64Bit ? NN_SVC_ADDR_MEMORY_REGION_LARGE64_END : NN_SVC_ADDR_MEMORY_REGION_LARGE32_END) - 1;
    return GetLargeFreeAreaImpl(pOutAddr, size, process, largeBegin, largeEnd);
}

bool GetLargeFreeArea(uint64_t* pOutAddr, uint64_t size)
{
    return GetLargeFreeAreaImpl(pOutAddr, size, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, LargeRegionBegin, LargeRegionEnd);
}

bool GetReservedFreeMemoryArea(uintptr_t* pOutAddr, size_t* pOutSize)
{
    nn::svc::MemoryInfo blockInfo;
    uintptr_t addr = g_ReservedAreaBegin;
    while(addr <= g_ReservedAreaEnd)
    {
        GetMemoryInfo(&blockInfo, addr);
        if (blockInfo.state == nn::svc::MemoryState_Free)
        {
            *pOutAddr = static_cast<uintptr_t>(addr);
            *pOutSize = static_cast<size_t>(blockInfo.size);
            if (blockInfo.baseAddress < g_ReservedAreaBegin)
            {
                *pOutSize -= g_ReservedAreaBegin - static_cast<uintptr_t>(blockInfo.baseAddress);
            }
            if (blockInfo.baseAddress + blockInfo.size - 1 > g_ReservedAreaEnd)
            {
                *pOutSize -= static_cast<size_t>((blockInfo.baseAddress + blockInfo.size) - (g_ReservedAreaEnd + 1));
            }
            return true;
        }
        addr = blockInfo.baseAddress + blockInfo.size;
    }
    return false;
}

template <class T>
size_t AddMemoryStateImpl(TestMemoryInfo* header, uintptr_t bufferAddr, size_t bufferSize)
{
    size_t typeSize = sizeof(T);
    NN_ASSERT(bufferSize > typeSize);
    T* state =
        new(reinterpret_cast<void*>(bufferAddr)) T();
    NN_ASSERT(state != nullptr);
    header->AddTest(state);
    return sizeof(T);
}

void GenerateMemoryStateList(TestMemoryInfo** pOut, uintptr_t bufferAddr, size_t bufferSize)
{
    size_t consumeSize = 0;
    size_t typeSize = 0;

    typeSize = sizeof(TestMemoryInfo);
    NN_ASSERT(bufferSize - consumeSize > typeSize);
    TestMemoryInfo* header;
    header = new(reinterpret_cast<void*>(bufferAddr + consumeSize)) TestMemoryInfo();
    NN_ASSERT(header != nullptr);
    consumeSize += typeSize;
    *pOut = header;

    consumeSize += AddMemoryStateImpl<TestFreeMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestIoMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestStaticMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestCodeMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestCodeDataMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestNormalMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestSharedMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestAliasMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestAliasCodeMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

#ifdef SUPPORT_ALIAS_CODE_DATA
    consumeSize += AddMemoryStateImpl<TestAliasCodeDataMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);
#endif

    consumeSize += AddMemoryStateImpl<TestIpcMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestNonSecureIpcMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestNonDeviceIpcMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestStackMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestThreadLocalMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestTransferedMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestSharedTransferedMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestSharedCodeMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestInaccessibleMemoryState>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    NN_ASSERT(header->GetListNum() == NumTestMemoryState);
}

void GenerateMemoryAttributeList(TestMemoryInfo** pOut, uintptr_t bufferAddr, size_t bufferSize)
{
    size_t consumeSize = 0;
    size_t typeSize = 0;

    typeSize = sizeof(TestMemoryInfo);
    NN_ASSERT(bufferSize - consumeSize > typeSize);
    TestMemoryInfo* header;
    header = new(reinterpret_cast<void*>(bufferAddr + consumeSize)) TestMemoryInfo();
    NN_ASSERT(header != nullptr);
    consumeSize += typeSize;
    *pOut = header;

    consumeSize += AddMemoryStateImpl<TestNoneAttribute>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestLockedAttribute>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestDeviceSharedAttribute>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestUncachedAttribute>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestIpcLockedAttribute>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestDeviceSharedAndUncachedAttribute>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);

    consumeSize += AddMemoryStateImpl<TestDeviceSharedAndIpcLockedAttribute>(
            header, bufferAddr + consumeSize, bufferSize - consumeSize);
    NN_ASSERT(header->GetListNum() == NumTestMemoryAttribute);
}

void TestIpcMapArea::Initialize(const void* mapData, size_t mapSize, bool canWrite, int ipcType)
{
    m_pMapData = mapData;
    m_Size = mapSize;
    m_CanWrite = canWrite;
    m_IpcType = ipcType;
    m_IsMapped = false;
    m_MapAddr = 0;
}

void TestIpcMapArea::Map()
{
    // セッションの準備
    nn::Result result = nn::svc::CreateSession(&m_ServerSession, &m_ClientSession, false, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    TestIpcData data;
    data.mapData = m_pMapData;
    data.mapSize = m_Size;
    data.ipcAttr = m_IpcType;
    data.handle = m_ClientSession;

    // スレッドの準備
    uintptr_t pc = m_CanWrite ? reinterpret_cast<uintptr_t>(SendIpcThread) : reinterpret_cast<uintptr_t>(SendIpcThreadForSend);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param = reinterpret_cast<uintptr_t>(&data);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    result = nn::svc::CreateThread(&m_Thread, pc, param, sp, priority, idealCore);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(m_Thread);
    NN_ASSERT_RESULT_SUCCESS(result);

    // IPCの設定
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    size_t msgSize = sizeof(g_ServerBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));
    {
        ipcMsg.SetNull();

        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, pMsgBuffer, msgSize,
                &m_ServerSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
    NN_ASSERT(ipcHeader.GetTag() == 0x0001);
    if (m_CanWrite)
    {
        NN_ASSERT_EQUAL(ipcHeader.GetReceiveNum(), 1);
    }
    else
    {
        NN_ASSERT_EQUAL(ipcHeader.GetSendNum(), 1);
    }

    int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
    nn::svc::ipc::MessageBuffer::MapData mapData(offset, ipcMsg);
    NN_ASSERT_EQUAL(mapData.GetAttribute(), m_IpcType);
    m_MapAddr = mapData.GetDataAddress();
    m_IsMapped = true;
}

void TestIpcMapArea::Unmap()
{
    if (!m_IsMapped)
    {
        return;
    }

    nn::Result result;
    uintptr_t pMsgBuffer = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    size_t msgSize = sizeof(g_ServerBuffer);
    int32_t index;
    nn::svc::ipc::MessageBuffer ipcMsg(reinterpret_cast<nn::Bit32*>(pMsgBuffer));
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 0);
    ipcMsg.Set(ipcHeader);

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, pMsgBuffer, msgSize,
            &m_ServerSession, 0, m_ServerSession, 0);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::WaitSynchronization(&index, &m_Thread, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(m_Thread);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(m_ServerSession);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_IsMapped = false;
}

TestIpcMapArea::~TestIpcMapArea()
{
    if (m_IsMapped)
    {
        Unmap();
    }
}

void TestMemoryInfo::CheckDefaultState()
{
    nn::svc::MemoryInfo blockInfo;
    GetMemoryInfo(&blockInfo, m_Addr);
    NN_ASSERT(blockInfo.state == m_State, "block state=%d, m_State=%d", blockInfo.state, m_State);
    NN_ASSERT(blockInfo.permission == m_Permission,
            "block permission=%d, m_Permission=%d", blockInfo.permission, m_Permission);
    NN_ASSERT(blockInfo.attribute == static_cast<nn::svc::MemoryAttribute>(m_Attribute),
            "block attribute=%d, m_Attribute=%d", blockInfo.attribute, m_Attribute);
}

void TestMemoryInfo::AddTest(TestMemoryInfo* test)
{
    m_TestList.insert(m_TestList.end(), *test);
}

void TestMemoryInfo::SetSize(size_t size)
{
    m_Size = size;
}

void TestMemoryInfo::SetHeap(uintptr_t addr, size_t size)
{
    m_HeapAddr = addr;
    m_HeapSize = size;
}

void TestMemoryInfo::SetFree(uintptr_t addr, size_t size)
{
    m_FreeAddr = addr;
    m_FreeSize = size;
}

void TestMemoryInfo::GetTestListWithStates(
        TestMemoryInfo** pOut, const nn::svc::MemoryState* states, int numState)
{
    TestList::iterator it = m_TestList.begin();
    int index = 0;
    for (; it != m_TestList.end(); it++)
    {
        for (int i = 0; i < numState; i++)
        {
            if (it->GetState() == states[i])
            {
                NN_ASSERT(index < numState);
                pOut[index++] = static_cast<TestMemoryInfo*>(&*it);
            }
        }
    }
    NN_ASSERT(numState == index);
}

void TestMemoryInfo::GetTestListExceptStates(
        TestMemoryInfo** pOut, int numOut, const nn::svc::MemoryState* states, int numState)
{
    TestList::iterator it = m_TestList.begin();
    int index = 0;
    for (; it != m_TestList.end(); it++)
    {
        bool isInvalid = true;
        for (int i = 0; i < numState; i++)
        {
            if (it->GetState() == states[i])
            {
                isInvalid = false;
                break;
            }
        }
        if (isInvalid)
        {
            pOut[index++] = static_cast<TestMemoryInfo*>(&*it);
        }
    }
    NN_ASSERT(numOut == index);
}

void TestMemoryInfo::GetTestListWithAttributes(
        TestMemoryInfo** pOut, const uint32_t* attrs, int numAttr)
{
    TestList::iterator it = m_TestList.begin();
    int index = 0;
    for (; it != m_TestList.end(); it++)
    {
        for (int i = 0; i < numAttr; i++)
        {
            if (it->GetAttribute() == attrs[i])
            {
                NN_ASSERT(index < numAttr);
                pOut[index++] = static_cast<TestMemoryInfo*>(&*it);
            }
        }
    }
    NN_ASSERT(numAttr == index);
}

void TestMemoryInfo::GetTestListExceptAttributes(
        TestMemoryInfo** pOut, int numOut, const uint32_t* attrs, int numAttr)
{
    TestList::iterator it = m_TestList.begin();
    int index = 0;
    for (; it != m_TestList.end(); it++)
    {
        bool isInvalid = true;
        for (int i = 0; i < numAttr; i++)
        {
            if (it->GetAttribute() == attrs[i])
            {
                isInvalid = false;
                break;
            }
        }
        if (isInvalid)
        {
            pOut[index++] = static_cast<TestMemoryInfo*>(&*it);
        }
    }
    NN_ASSERT(numOut == index);
}


TestFreeMemoryState::TestFreeMemoryState()
{
    m_State = nn::svc::MemoryState_Free;
    m_Permission = nn::svc::MemoryPermission_None;
    m_Attribute = 0;
    m_Size = g_FreeAreaEnd - g_FreeAreaBegin;
    m_CanWriteData = false;
}

void TestFreeMemoryState::Initialize(const void* pData, size_t size)
{
    NN_UNUSED(pData);
    NN_UNUSED(size);
    DEBUG_PRINT("TestFreeMemoryState\n");
    m_End = false;
    m_Addr = g_FreeAreaBegin;
    CheckDefaultState();
}

TestFreeMemoryState::~TestFreeMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestFreeMemoryState::Close()
{
    m_End = true;
}

TestIoMemoryState::TestIoMemoryState()
{
    m_State = nn::svc::MemoryState_Io;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_CanWriteData = false;
    m_Attribute = 0;
#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || \
    defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || \
    defined (NN_BUILD_CONFIG_HARDWARE_NX)
    uint64_t physAddr = 0x60005000;
    m_Size = 0x1000;

    nn::Result result;
    result = nn::svc::QueryIoMapping(&m_Addr, physAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);
#else
    m_Addr = 0;
    m_Size = 0;
#endif
}

void TestIoMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestIoMemoryState\n");
    NN_UNUSED(pData);
    NN_UNUSED(size);

    m_End = false;
    if (m_Size > 0)
    {
        CheckDefaultState();
    }
}

TestIoMemoryState::~TestIoMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestIoMemoryState::Close()
{
    m_End = true;
}

TestStaticMemoryState::TestStaticMemoryState()
{
    m_State = nn::svc::MemoryState_Static;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = 0;
    m_Addr = 0;
    m_End = true;
    m_CanWriteData = false;

    uintptr_t v;
    nn::svc::MemoryInfo blockInfo;
    v = CheckVirtualBegin;
    uintptr_t end = CheckVirtualEnd - 1;
    do
    {
        GetMemoryInfo(&blockInfo, v);

        if (blockInfo.state == nn::svc::MemoryState_Static)
        {
            m_Addr = blockInfo.baseAddress;
            m_Size = blockInfo.size;
            m_Permission = blockInfo.permission;
            m_Attribute = blockInfo.attribute;
            break;
        }

        v = blockInfo.baseAddress + blockInfo.size;
    } while (v <= end && v != 0);
}

void TestStaticMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestStaticMemoryState\n");
    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }
    m_End = false;
}

TestStaticMemoryState::~TestStaticMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestStaticMemoryState::Close()
{
    m_End = true;
}

TestCodeMemoryState::TestCodeMemoryState()
{
    m_State = nn::svc::MemoryState_Code;
    m_Permission = nn::svc::MemoryPermission_ReadExecute;
    m_Attribute = 0;
    m_CanWriteData = false;

    uintptr_t addr = reinterpret_cast<uintptr_t>(nnMain);
    nn::svc::MemoryInfo blockInfo;
    GetMemoryInfo(&blockInfo, addr);
    m_Addr = blockInfo.baseAddress;
    m_Size = blockInfo.size;
}

void TestCodeMemoryState::Initialize(const void* pData, size_t size)
{
    NN_UNUSED(pData);
    NN_UNUSED(size);
    DEBUG_PRINT("TestCodeMemoryState\n");
    m_End = false;
    CheckDefaultState();
}

TestCodeMemoryState::~TestCodeMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestCodeMemoryState::Close()
{
    m_End = true;
}

TestCodeDataMemoryState::TestCodeDataMemoryState()
{
    m_State = nn::svc::MemoryState_CodeData;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_CanWriteData = false;
    m_Size = sizeof(g_Buffer);
}

void TestCodeDataMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestCodeDataMemoryState\n");
    m_End = false;
    m_Addr = reinterpret_cast<uintptr_t>(g_Buffer);
    if (size <= m_Size)
    {
        std::memcpy(g_Buffer, pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestCodeDataMemoryState::~TestCodeDataMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestCodeDataMemoryState::Close()
{
    m_End = true;
}

TestNormalMemoryState::TestNormalMemoryState()
{
    m_State = nn::svc::MemoryState_Normal;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
    m_HeapAddr = 0;
}

void TestNormalMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestNormalMemoryState\n");
    m_End = false;
    nn::Result result;
    if (m_HeapAddr == 0)
    {
        result = nn::svc::SetHeapSize(&m_Addr, HeapAlign);
        NN_ASSERT_RESULT_SUCCESS(result);
        m_Size = HeapAlign;
    }
    else
    {
        NN_ASSERT(m_Size <= m_HeapSize);
        m_Addr = m_HeapAddr;
        m_Size = m_HeapSize;
    }

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestNormalMemoryState::~TestNormalMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestNormalMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;
    if (m_HeapAddr == 0)
    {
        result = nn::svc::SetHeapSize(&m_Addr, 0);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
}

TestSharedMemoryState::TestSharedMemoryState()
{
    m_State = nn::svc::MemoryState_Shared;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
}

void TestSharedMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestSharedMemoryState\n");
    m_End = false;
    nn::Result result;
    nn::svc::MemoryPermission sharedPerm = nn::svc::MemoryPermission_ReadWrite;

    result = nn::svc::CreateSharedMemory(&handle, m_Size, sharedPerm, sharedPerm);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Addr = m_FreeAddr;
    result = nn::svc::MapSharedMemory(handle, m_Addr, m_Size, sharedPerm);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestSharedMemoryState::~TestSharedMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestSharedMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;

    result = nn::svc::UnmapSharedMemory(handle, m_Addr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    NN_ASSERT_RESULT_SUCCESS(result);
}

TestAliasMemoryState::TestAliasMemoryState()
{
    m_State = nn::svc::MemoryState_Alias;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
}

void TestAliasMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestAliasMemoryState\n");
    m_End = false;
    nn::Result result;
    m_FromAddr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t reservedSize;
    NN_ASSERT(GetReservedFreeMemoryArea(&m_ToAddr, &reservedSize));
    NN_ASSERT(reservedSize >= m_Size);

    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, m_FromAddr);
    NN_ASSERT(info.state == nn::svc::MemoryState_CodeData);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT(info.attribute == 0);
    GetMemoryInfo(&info, m_ToAddr);
    NN_ASSERT(info.state == nn::svc::MemoryState_Free);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_None);
    NN_ASSERT(info.attribute == 0);

    result = nn::svc::MapMemory(m_ToAddr, m_FromAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Addr = m_ToAddr;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }


    CheckDefaultState();
}

TestAliasMemoryState::~TestAliasMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestAliasMemoryState::Close()
{
    if (m_End)
    {
        return;
    }
    m_End = true;
    nn::Result result;

    result = nn::svc::UnmapMemory(m_ToAddr, m_FromAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);
}

TestAliasCodeMemoryState::TestAliasCodeMemoryState()
{
    m_State = nn::svc::MemoryState_AliasCode;
    m_Permission = nn::svc::MemoryPermission_Read;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_HeapAddr = 0;
    m_CanWriteData = false;
}

void TestAliasCodeMemoryState::SetPermission(nn::svc::MemoryPermission perm)
{
    m_Permission = perm;
    if (m_Permission == nn::svc::MemoryPermission_ReadWrite)
    {
        m_State = nn::svc::MemoryState_AliasCodeData;
    }
}

void TestAliasCodeMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestAliasCodeMemoryState\n");
    m_End = false;
    nn::Result result;

    nn::svc::Handle namedPort;
    NamedPortManager namedPortManager(AliasCodePortName, 1);
    namedPort = namedPortManager.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinAliasCode_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinAliasCode_end);
    TestLoader loader(BinAliasCode_begin, end - begin);
    loader.SpawnProcess(&m_Handle);
    loader.StartProcess(m_Handle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &namedPort, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, namedPort);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sessionCloser(serverSession);

    ReceiveMoveHandle(&m_ReadableEvent, serverSession);
    ReceiveMoveHandle(&m_WritableEvent, serverSession);

    SendServerProcessHandle(serverSession);

    uintptr_t heapAddr;
    if (m_HeapAddr == 0)
    {
        result = nn::svc::SetHeapSize(&heapAddr, HeapAlign);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
    else
    {
        NN_ASSERT(m_Size <= m_HeapSize);
        heapAddr = m_HeapAddr;
    }

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(heapAddr), pData, size);
        m_CanWriteData = true;
    }

    uint64_t sendData = heapAddr;
    ServerSendData(serverSession, &sendData, sizeof(sendData));
    sendData = m_FreeAddr;
    ServerSendData(serverSession, &sendData, sizeof(sendData));
    sendData = m_Size;
    ServerSendData(serverSession, &sendData, sizeof(sendData));
    ServerSendData(serverSession, &m_Permission, sizeof(m_Permission));

    result = nn::svc::WaitSynchronization(&index, &m_ReadableEvent, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Addr = m_FreeAddr;
    CheckDefaultState();
}

TestAliasCodeMemoryState::~TestAliasCodeMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestAliasCodeMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;

    result = nn::svc::SignalEvent(m_WritableEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    WaitProcess(m_Handle);

    result = nn::svc::CloseHandle(m_Handle);
    NN_ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(m_ReadableEvent);
    NN_ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(m_WritableEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (m_HeapAddr == 0)
    {
        uintptr_t heapAddr;
        result = nn::svc::SetHeapSize(&heapAddr, 0);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
}

TestAliasCodeDataMemoryState::TestAliasCodeDataMemoryState()
{
    m_State = nn::svc::MemoryState_AliasCodeData;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_HeapAddr = 0;
}

TestIpcMemoryState::TestIpcMemoryState()
{
    m_State = nn::svc::MemoryState_Ipc;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = sizeof(g_ClientMapArea);
    m_CanWriteData = false;
}

void TestIpcMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestIpcMemoryState\n");
    m_End = false;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(g_ClientMapArea), pData, size);
        m_CanWriteData = true;
    }

    m_IpcMap.Initialize(g_ClientMapArea, m_Size, true, 0);
    m_IpcMap.Map();
    m_Addr = m_IpcMap.GetMapAddress();

    CheckDefaultState();
}

TestIpcMemoryState::~TestIpcMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestIpcMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    m_IpcMap.Unmap();
}

TestNonSecureIpcMemoryState::TestNonSecureIpcMemoryState()
{
    m_State = nn::svc::MemoryState_NonSecureIpc;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
}

void TestNonSecureIpcMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestNonSecureIpcMemoryState\n");
    m_End = false;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(g_ClientMapArea), pData, size);
        m_CanWriteData = true;
    }

    m_IpcMap.Initialize(g_ClientMapArea, sizeof(g_ClientMapArea), true, 1);
    m_IpcMap.Map();
    m_Addr = m_IpcMap.GetMapAddress();

    CheckDefaultState();
}

TestNonSecureIpcMemoryState::~TestNonSecureIpcMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestNonSecureIpcMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    m_IpcMap.Unmap();
}

TestNonDeviceIpcMemoryState::TestNonDeviceIpcMemoryState()
{
    m_State = nn::svc::MemoryState_NonDeviceIpc;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
}

void TestNonDeviceIpcMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestNonDeviceIpcMemoryState\n");
    m_End = false;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(g_ClientMapArea), pData, size);
        m_CanWriteData = true;
    }

    m_IpcMap.Initialize(g_ClientMapArea, sizeof(g_ClientMapArea), true, 3);
    m_IpcMap.Map();
    m_Addr = m_IpcMap.GetMapAddress();

    CheckDefaultState();
}

TestNonDeviceIpcMemoryState::~TestNonDeviceIpcMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestNonDeviceIpcMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    m_IpcMap.Unmap();
}

TestStackMemoryState::TestStackMemoryState()
{
    m_State = nn::svc::MemoryState_Stack;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
}

void TestStackMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestStackMemoryState\n");
    m_End = false;

    nn::Result result;
    m_FromAddr = reinterpret_cast<uintptr_t>(g_Buffer);
    m_ToAddr = m_FreeAddr;
    NN_ASSERT(m_FreeSize >= m_Size);

    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, m_FromAddr);
    NN_ASSERT(info.state == nn::svc::MemoryState_CodeData);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT(info.attribute == 0);
    GetMemoryInfo(&info, m_ToAddr);
    NN_ASSERT(info.state == nn::svc::MemoryState_Free);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_None);
    NN_ASSERT(info.attribute == 0);

    result = nn::svc::MapMemory(m_ToAddr, m_FromAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Addr = m_ToAddr;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestStackMemoryState::~TestStackMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestStackMemoryState::Close()
{
    if (m_End)
    {
        return;
    }
    m_End = true;
    nn::Result result;

    result = nn::svc::UnmapMemory(m_ToAddr, m_FromAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);
}

TestThreadLocalMemoryState::TestThreadLocalMemoryState()
{
    m_State = nn::svc::MemoryState_ThreadLocal;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    m_CanWriteData = false;
}

void TestThreadLocalMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestThreadLocalMemoryState\n");
    m_End = false;
    m_Addr = reinterpret_cast<uintptr_t>(nn::svc::ipc::GetMessageBuffer());

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestThreadLocalMemoryState::~TestThreadLocalMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestThreadLocalMemoryState::Close()
{
    m_End = true;
}

TestTransferedMemoryState::TestTransferedMemoryState()
{
    m_State = nn::svc::MemoryState_Transfered;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_FromAddr = 0;
    m_ToAddr = m_FreeAddr;
    m_Size = TestMemorySize;
    m_HeapAddr = 0;
    m_CanWriteData = false;
}

void TestTransferedMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestTransferedMemoryState\n");
    m_End = false;
    nn::Result result;
    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_None;

    if (m_HeapAddr == 0)
    {
        result = nn::svc::SetHeapSize(&m_FromAddr, HeapAlign);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
    else
    {
        NN_ASSERT(m_Size <= m_HeapSize);
        m_FromAddr = m_HeapAddr;
    }

    result = nn::svc::CreateTransferMemory(&m_Handle, m_FromAddr, m_Size, perm);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, m_FromAddr);
    NN_ASSERT(info.state == nn::svc::MemoryState_Normal);
    NN_ASSERT(info.permission == perm);
    NN_ASSERT(info.attribute == nn::svc::MemoryAttribute_Locked);

    GetMemoryInfo(&info, m_ToAddr);
    NN_ASSERT(info.state == nn::svc::MemoryState_Free);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_None);
    NN_ASSERT(info.attribute == 0);

    result = nn::svc::MapTransferMemory(m_Handle, m_ToAddr, m_Size, perm);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Addr = m_ToAddr;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestTransferedMemoryState::~TestTransferedMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestTransferedMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;
    result = nn::svc::UnmapTransferMemory(m_Handle, m_ToAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(m_Handle);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (m_HeapAddr == 0)
    {
        result = nn::svc::SetHeapSize(&m_FromAddr, 0);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
}

TestSharedTransferedMemoryState::TestSharedTransferedMemoryState()
{
    m_State = nn::svc::MemoryState_SharedTransfered;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_FromAddr = 0;
    m_ToAddr = m_FreeAddr;
    m_Size = TestMemorySize;
    m_HeapAddr = 0;
    m_CanWriteData = false;
}

void TestSharedTransferedMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestSharedTransferedMemoryState\n");
    m_End = false;
    nn::Result result;
    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_ReadWrite;

    if (m_HeapAddr == 0)
    {
        result = nn::svc::SetHeapSize(&m_FromAddr, HeapAlign);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
    else
    {
        NN_ASSERT(m_Size <= m_HeapSize);
        m_FromAddr = m_HeapAddr;
    }

    result = nn::svc::CreateTransferMemory(&m_Handle, m_FromAddr, m_Size, perm);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, m_FromAddr);
    NN_ASSERT(info.state == nn::svc::MemoryState_Normal);
    NN_ASSERT(info.permission == m_Permission);
    NN_ASSERT(info.attribute == nn::svc::MemoryAttribute_Locked);

    GetMemoryInfo(&info, m_ToAddr);
    NN_ASSERT(info.state == nn::svc::MemoryState_Free);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_None);
    NN_ASSERT(info.attribute == 0);

    result = nn::svc::MapTransferMemory(m_Handle, m_ToAddr, m_Size, perm);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Addr = m_ToAddr;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestSharedTransferedMemoryState::~TestSharedTransferedMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestSharedTransferedMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;
    result = nn::svc::UnmapTransferMemory(m_Handle, m_ToAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(m_Handle);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (m_HeapAddr == 0)
    {
        result = nn::svc::SetHeapSize(&m_FromAddr, 0);
        NN_ASSERT_RESULT_SUCCESS(result);
    }
}

TestSharedCodeMemoryState::TestSharedCodeMemoryState()
    : m_Process(TestMemorySize / 0x1000)
{
    m_State = nn::svc::MemoryState_SharedCode;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
}

void TestSharedCodeMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestSharedCodeMemoryState\n");
    m_End = false;
    nn::Result result;
    uintptr_t toAddr =m_FreeAddr;
    uintptr_t fromAddr = m_Process.GetCodeAddress();

    result = nn::svc::MapProcessMemory(toAddr, m_Process.GetHandle(), fromAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Addr = m_FreeAddr;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestSharedCodeMemoryState::~TestSharedCodeMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestSharedCodeMemoryState::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;
    uintptr_t toAddr = m_FreeAddr;
    uintptr_t fromAddr = m_Process.GetCodeAddress();

    result = nn::svc::UnmapProcessMemory(toAddr, m_Process.GetHandle(), fromAddr, m_Size);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Process.Close();
}

TestInaccessibleMemoryState::TestInaccessibleMemoryState()
{
    m_State = nn::svc::MemoryState_Inaccessible;
    m_Permission = nn::svc::MemoryPermission_None;
    m_Attribute = 0;

    if (ProcessEnd + 1 > ProcessBegin)
    {
        m_Addr = ProcessEnd + 1;
        m_Size = CheckVirtualEnd - m_Addr;
    }
    else
    {
        m_Addr = 0;
        m_Size = 0;
    }

    m_CanWriteData = false;
}

void TestInaccessibleMemoryState::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestInaccessibleMemoryState\n");
    m_End = false;
}

TestInaccessibleMemoryState::~TestInaccessibleMemoryState()
{
    if (!m_End)
    {
        Close();
    }
}

void TestInaccessibleMemoryState::Close()
{
    m_End = true;
}

TestNoneAttribute::TestNoneAttribute()
{
    m_State = nn::svc::MemoryState_CodeData;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = 0;
    m_Size = TestMemorySize;
    m_Addr = reinterpret_cast<uintptr_t>(g_Buffer);
    m_Size = sizeof(g_Buffer);
    m_CanWriteData = false;
}

void TestNoneAttribute::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestNoneAttributeMemory\n");
    m_End = false;
    nn::Result result;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestNoneAttribute::~TestNoneAttribute()
{
    if (!m_End)
    {
        Close();
    }
}

void TestNoneAttribute::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
}

void TestNoneAttribute::SetNoneArea(uintptr_t addr, size_t size)
{
    nn::Result result;
    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, addr);
    NN_ASSERT(info.baseAddress <= addr);
    NN_ASSERT(info.size >= size);
    NN_ASSERT(info.baseAddress + info.size - 1 >= addr + size - 1);
    NN_ASSERT(info.attribute == 0);
    m_Addr = addr;
    m_Size = size;
    m_State = info.state;
    m_Permission = info.permission;
}

TestLockedAttribute::TestLockedAttribute()
{
    m_State = nn::svc::MemoryState_CodeData;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = nn::svc::MemoryAttribute_Locked;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
}

void TestLockedAttribute::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestLockedAttributeMemory\n");
    m_End = false;
    nn::Result result;
    m_Addr = reinterpret_cast<uintptr_t>(g_Buffer);
    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, m_Addr);
    NN_ASSERT(info.state == nn::svc::MemoryState_CodeData);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT(info.attribute == 0);

    result = nn::svc::CreateTransferMemory(&m_TransferHandle, m_Addr, m_Size, m_Permission);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestLockedAttribute::~TestLockedAttribute()
{
    if (!m_End)
    {
        Close();
    }
}

void TestLockedAttribute::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;

    result = nn::svc::CloseHandle(m_TransferHandle);
    NN_ASSERT_RESULT_SUCCESS(result);
}

TestDeviceSharedAttribute::TestDeviceSharedAttribute()
{
    m_State = nn::svc::MemoryState_CodeData;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = nn::svc::MemoryAttribute_DeviceShared;
    m_Size = TestMemorySize;
    m_DeviceAddress = 0;
    m_Process = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    m_AddressSpaceSize = 0x80000000;
    m_CanWriteData = false;
}

void TestDeviceSharedAttribute::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestDeviceSharedAttribute\n");
    m_End = false;
    nn::Result result;

    uint64_t align = 0x400000;
    m_Addr = reinterpret_cast<uintptr_t>(g_Buffer);
    m_DeviceAddress = m_Addr & (align - 1);
    result = nn::svc::CreateDeviceAddressSpace(&m_Device, 0, m_AddressSpaceSize);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            m_Device, m_Process,
            m_Addr, m_Size, m_DeviceAddress,
            m_Permission);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestDeviceSharedAttribute::~TestDeviceSharedAttribute()
{
    if (!m_End)
    {
        Close();
    }
}

void TestDeviceSharedAttribute::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;

    result = nn::svc::UnmapDeviceAddressSpace(
            m_Device, m_Process,
            m_Addr, m_Size, m_DeviceAddress);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(m_Device);
    NN_ASSERT_RESULT_SUCCESS(result);
}

TestUncachedAttribute::TestUncachedAttribute()
{
    m_State = nn::svc::MemoryState_CodeData;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = nn::svc::MemoryAttribute_Uncached;
    m_Size = TestMemorySize;
    m_CanWriteData = false;
}

void TestUncachedAttribute::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestUncachedAttribute\n");
    m_End = false;
    nn::Result result;

    m_Addr = reinterpret_cast<uintptr_t>(g_Buffer);
    result = nn::svc::SetMemoryAttribute(m_Addr, m_Size, m_Attribute, m_Attribute);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestUncachedAttribute::~TestUncachedAttribute()
{
    if (!m_End)
    {
        Close();
    }
}

void TestUncachedAttribute::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;

    result = nn::svc::SetMemoryAttribute(m_Addr, m_Size, m_Attribute, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
}

TestIpcLockedAttribute::TestIpcLockedAttribute()
{
    m_State = nn::svc::MemoryState_CodeData;
    m_Permission = nn::svc::MemoryPermission_Read;
    m_Attribute = nn::svc::MemoryAttribute_IpcLocked;
#ifdef SUPPORT_IPC_LOCKED
    m_Size = sizeof(g_ClientMapArea);
#else
    m_Size = 0;
#endif
    m_CanWriteData = false;
}

void TestIpcLockedAttribute::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestIpcLockedAttribute\n");
    m_End = false;

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(g_ClientMapArea), pData, size);
        m_CanWriteData = true;
    }

#ifdef SUPPORT_IPC_LOCKED
    m_IpcMap.Initialize(g_ClientMapArea, m_Size, false, 0);
    m_IpcMap.Map();
    m_Addr = reinterpret_cast<uintptr_t>(g_ClientMapArea);

    CheckDefaultState();
#else
    m_Addr = 0;
    m_Size = 0;
#endif
}

TestIpcLockedAttribute::~TestIpcLockedAttribute()
{
    if (!m_End)
    {
        Close();
    }
}

void TestIpcLockedAttribute::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;

#ifdef SUPPORT_IPC_LOCKED
    m_IpcMap.Unmap();
#endif
}

TestDeviceSharedAndUncachedAttribute::TestDeviceSharedAndUncachedAttribute()
{
    m_State = nn::svc::MemoryState_CodeData;
    m_Permission = nn::svc::MemoryPermission_ReadWrite;
    m_Attribute = nn::svc::MemoryAttribute_DeviceShared | nn::svc::MemoryAttribute_Uncached;
    m_Size = TestMemorySize;
    m_DeviceAddress = 0;
    m_Process = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    m_AddressSpaceSize = 0x80000000;
    m_CanWriteData = false;
}

void TestDeviceSharedAndUncachedAttribute::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestDeviceSharedAndUncachedAttribute\n");
    m_End = false;
    nn::Result result;

    uint64_t align = 0x400000;
    m_Addr = reinterpret_cast<uintptr_t>(g_Buffer);
    m_DeviceAddress = m_Addr & (align - 1);
    result = nn::svc::CreateDeviceAddressSpace(&m_Device, 0, m_AddressSpaceSize);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapDeviceAddressSpaceAligned(
            m_Device, m_Process,
            m_Addr, m_Size, m_DeviceAddress,
            m_Permission);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;
    nn::Bit32 attribute = nn::svc::MemoryAttribute_Uncached;
    result = nn::svc::SetMemoryAttribute(m_Addr, m_Size, mask, attribute);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    CheckDefaultState();
}

TestDeviceSharedAndUncachedAttribute::~TestDeviceSharedAndUncachedAttribute()
{
    if (!m_End)
    {
        Close();
    }
}

void TestDeviceSharedAndUncachedAttribute::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;

    nn::Bit32 mask = nn::svc::MemoryAttribute_Uncached;
    result = nn::svc::SetMemoryAttribute(m_Addr, m_Size, mask, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapDeviceAddressSpace(
            m_Device, m_Process,
            m_Addr, m_Size, m_DeviceAddress);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(m_Device);
    NN_ASSERT_RESULT_SUCCESS(result);
}

TestDeviceSharedAndIpcLockedAttribute::TestDeviceSharedAndIpcLockedAttribute()
{
    m_State = nn::svc::MemoryState_CodeData;
    m_Permission = nn::svc::MemoryPermission_Read;
    m_Attribute = nn::svc::MemoryAttribute_DeviceShared | nn::svc::MemoryAttribute_IpcLocked;
    m_Size = TestMemorySize;
    m_DeviceAddress = 0;
    m_Process = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;
    m_AddressSpaceSize = 0x80000000;
    m_CanWriteData = false;
}

void TestDeviceSharedAndIpcLockedAttribute::Initialize(const void* pData, size_t size)
{
    DEBUG_PRINT("TestDeviceSharedAndIpcLockedAttribute\n");
    m_End = false;
    nn::Result result;

    uint64_t align = 0x400000;
    m_Addr = reinterpret_cast<uintptr_t>(g_Buffer);
    m_DeviceAddress = m_Addr & (align - 1);
    result = nn::svc::CreateDeviceAddressSpace(&m_Device, 0, m_AddressSpaceSize);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (size <= m_Size)
    {
        std::memcpy(reinterpret_cast<void*>(m_Addr), pData, size);
        m_CanWriteData = true;
    }

    result = nn::svc::MapDeviceAddressSpaceAligned(
            m_Device, m_Process,
            m_Addr, m_Size, m_DeviceAddress,
            m_Permission);
    NN_ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_IPC_LOCKED
    m_IpcMap.Initialize(reinterpret_cast<void*>(m_Addr), m_Size, false, 1);
    m_IpcMap.Map();

    CheckDefaultState();
#else
    m_Addr = 0;
    m_Size = 0;
#endif
}

TestDeviceSharedAndIpcLockedAttribute::~TestDeviceSharedAndIpcLockedAttribute()
{
    if (!m_End)
    {
        Close();
    }
}

void TestDeviceSharedAndIpcLockedAttribute::Close()
{
    if (m_End)
    {
        return;
    }

    m_End = true;
    nn::Result result;

#ifdef SUPPORT_IPC_LOCKED
    m_IpcMap.Unmap();
#endif

    result = nn::svc::UnmapDeviceAddressSpace(
            m_Device, m_Process,
            m_Addr, m_Size, m_DeviceAddress);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(m_Device);
    NN_ASSERT_RESULT_SUCCESS(result);
}


