﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_DebugProcess.h"
#include <nn/svc/svc_Thread.h>

namespace {
const uint64_t SleepTime = 100 * 1000 * 1000;
}

bool CheckProcessState(nn::svc::Handle processHandle, nn::svc::ProcessState state)
{
    nn::Result result;
    int64_t info;
    result = nn::svc::GetProcessInfo(&info, processHandle, nn::svc::ProcessInfoType_State);
    NN_ASSERT_RESULT_SUCCESS(result);
    return (static_cast<nn::svc::ProcessState>(info) == state);
}

nn::Result ContinueDebugProcess(nn::svc::Handle debugHandle)
{
    nn::Result result;
    nn::Bit32 flags =
        nn::svc::ContinueFlag_ExceptionHandled |
        nn::svc::ContinueFlag_EnableExceptionEvent |
        nn::svc::ContinueFlag_ContinueAll;

    return nn::svc::ContinueDebugEvent(debugHandle, flags, 0);
}

nn::Result WaitAndGetDebugEvent(nn::svc::DebugEventInfo* pOut, nn::svc::Handle debugHandle, int64_t ns)
{
    nn::Result result;
    int32_t index;

    result = nn::svc::WaitSynchronization(&index, &debugHandle, 1, ns);
    if (!result.IsSuccess())
    {
        return result;
    }

    return nn::svc::GetDebugEvent(pOut, debugHandle);
}

void WaitDebugEvent(nn::svc::DebugEventInfo* pOut, nn::svc::Handle debugHandle)
{
    nn::Result result;
    for(;;)
    {
        result = WaitAndGetDebugEvent(pOut, debugHandle, SleepTime);
        if (result <= nn::svc::ResultTimeout())
        {
            result = ContinueDebugProcess(debugHandle);
            ASSERT_RESULT_SUCCESS(result);
        }
        else if (result.IsSuccess())
        {
            break;
        }
        else
        {
            ASSERT_TRUE(false);
        }
    }
}

void InitDebugProcess(nn::svc::Handle* pHandle, nn::Bit64* pTid, TestProcess* process)
{
    nn::Result result;

    nn::Bit64 processId;
    nn::svc::Handle processHandle = process->GetHandle();

    result = nn::svc::GetProcessId(&processId, processHandle);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_Initializing));

    result = nn::svc::DebugActiveProcess(pHandle, processId);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_PreAttached));

    process->Start();

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_Breaked));

    bool attached = false;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoCreateThread createThreadInfo;
    nn::svc::DebugInfoException except;

    for(int i = 0; i < 1 || !attached; )
    {
        result = WaitAndGetDebugEvent(&debugInfo, *pHandle, SleepTime);
        if (result <= nn::svc::ResultTimeout())
        {
            ASSERT_TRUE(attached);
            result = ContinueDebugProcess(*pHandle);
            ASSERT_RESULT_SUCCESS(result);
            continue;
        }
        ASSERT_RESULT_SUCCESS(result);

        if(debugInfo.event == nn::svc::DebugEvent_CreateThread)
        {
            createThreadInfo = debugInfo.info.createThread;
            if (pTid != nullptr)
            {
                *pTid = createThreadInfo.id;
            }
            i++;
        }
        if(debugInfo.event == nn::svc::DebugEvent_Exception)
        {
            except = debugInfo.info.exception;
            ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AttachBreak);
            attached = true;
        }
    }

    result = ContinueDebugProcess(*pHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void ExitDebugProcess(nn::svc::Handle debugProcess, int threadNum)
{
    nn::Result result;
    nn::svc::DebugEventInfo debugInfo;

    bool isExitProcess = false;
    for (int i = 0; i < threadNum; i++)
    {
        result = WaitAndGetDebugEvent(&debugInfo, debugProcess, SleepTime);
        if (result <= nn::svc::ResultTimeout())
        {
            result = ContinueDebugProcess(debugProcess);
            ASSERT_RESULT_SUCCESS(result);
            i--;
            continue;
        }
        else
        {
            ASSERT_RESULT_SUCCESS(result);
        }

        if (debugInfo.event == nn::svc::DebugEvent_Exception)
        {
            NN_LOG("exception: %d\n", debugInfo.info.exception.exceptionCode);
        }
        else if (debugInfo.event == nn::svc::DebugEvent_ExitProcess)
        {
            isExitProcess = true;
            break;
        }

        ASSERT_EQ(debugInfo.event, nn::svc::DebugEvent_ExitThread);
    }

    if (!isExitProcess)
    {
        result = WaitAndGetDebugEvent(&debugInfo, debugProcess);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_ExitProcess);
    }

    result = nn::svc::CloseHandle(debugProcess);
    ASSERT_RESULT_SUCCESS(result);
}

