﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>

namespace {
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

int g_Sequence;
struct TestData
{
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
};

void TestCancelSynchronization(TestData* pData)
{
    AutoThreadExit autoExit;

    nn::Result result;
    int32_t index;

    // TEST 20-1 (同じコア), 20-2 (違うコア)
    // イベントハンドルに通知が来るとスレッドを起こすことが出来る
    result = nn::svc::WaitSynchronization(&index, &pData->readableEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    result = nn::svc::ResetSignal(pData->readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(g_Sequence == 0);

    g_Sequence++;
    result = nn::svc::SignalEvent(pData->writableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 20-3 (同じコア), 20-4 (違うコア)
    // イベント待ちをキャンセルさせることが出来る
    result = nn::svc::WaitSynchronization(&index, &pData->readableEvent, 1,  -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
    ASSERT_TRUE(g_Sequence == 2);
    result = nn::svc::ResetSignal(pData->readableEvent);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

    g_Sequence++;
    result = nn::svc::SignalEvent(pData->writableEvent);
    ASSERT_RESULT_SUCCESS(result);

    volatile int64_t before = nn::svc::GetSystemTick();
    nn::svc::SleepThread(1000 * 1000 * 1000);
    volatile int64_t after = nn::svc::GetSystemTick();
    ASSERT_TRUE(((after - before) / (NN_HW_TICKS_PER_SECOND / (1000 * 1000)) * 1000) >= 1000 * 1000 * 1000);

    ASSERT_TRUE(g_Sequence == 4);

    // TEST 20-5 (同じコア), 20-6 (違うコア)
    // キャンセルとオブジェクト待ちの両方が成立していた場合、オブジェクト→キャンセルの順で取得される
    result = nn::svc::WaitSynchronization(&index, &pData->readableEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    result = nn::svc::ResetSignal(pData->readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &pData->readableEvent, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
    result = nn::svc::ResetSignal(pData->readableEvent);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());
}
}

TEST(CancelSynchronization, Test0)
{
    nn::Result result;
    nn::svc::Handle writableEvent0;
    nn::svc::Handle readableEvent0;
    result = nn::svc::CreateEvent(&writableEvent0, &readableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    nn::svc::Handle writableEvent1;
    nn::svc::Handle readableEvent1;
    result = nn::svc::CreateEvent(&writableEvent1, &readableEvent1);
    ASSERT_RESULT_SUCCESS(result);
    TestData data;
    data.readableEvent = readableEvent0;
    data.writableEvent = writableEvent1;
    int32_t index;

    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(TestCancelSynchronization);
        nn::svc::Handle handle;

        result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&data), sp, 32, idealCore);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(handle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 20-1 (同じコア), 20-2 (違うコア)
        // イベントハンドルに通知が来るとスレッドを起こすことが出来る
        g_Sequence = 0;
        result = nn::svc::SignalEvent(writableEvent0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::WaitSynchronization(&index, &readableEvent1, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);
        result = nn::svc::ResetSignal(readableEvent1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Sequence == 1);
        g_Sequence++;

        // TEST 20-3 (同じコア), 20-4 (違うコア)
        // イベント待ちをキャンセルさせることが出来る
        result = nn::svc::CancelSynchronization(handle);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::WaitSynchronization(&index, &readableEvent1, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);
        result = nn::svc::ResetSignal(readableEvent1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Sequence == 3);

        // TEST 20-5 (同じコア), 20-6 (違うコア)
        // 同期オブジェクトのシグナル待機とキャンセルの条件が整っている場合
        // キャンセルが優先される
        result = nn::svc::CancelSynchronization(handle);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::SignalEvent(writableEvent0);
        ASSERT_RESULT_SUCCESS(result);
        g_Sequence++;

        // TEST 20-7 (同じコア), 20-7 (違うコア)
        // スレッドの終了を受け、スレッドを起こすことが出来る
        result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
    result = nn::svc::CloseHandle(writableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(writableEvent1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent1);
    ASSERT_RESULT_SUCCESS(result);
}


