﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestMemory.h"

namespace {

const size_t MapSize = 0x1000;
const int DefaultAlignSize = 0x1000;

/* テストに使うマッピング領域が空きかどうかチェック*/
void CheckFreeArea(uintptr_t areaBegin, uintptr_t areaEnd)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, areaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(blockInfo.attribute == 0);
    ASSERT_TRUE(areaEnd - 1 <= blockInfo.baseAddress + blockInfo.size - 1);
}

void CheckNormalArea(uintptr_t addr, size_t size, nn::svc::MemoryPermission permission)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.permission == permission);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.size == size);
    ASSERT_TRUE(blockInfo.attribute == 0);
}

void CheckMapArea(uintptr_t toAddr, uintptr_t fromAddr, size_t size)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, fromAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == fromAddr);
    ASSERT_TRUE(blockInfo.size == size);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, toAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress <= toAddr);
    ASSERT_TRUE(blockInfo.size >= size);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= toAddr + size - 1);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Stack);
}

} //namespace {

extern "C" void nnMain();

TEST(UnmapMemoryTest, FromMemoryStateTest)
{
    nn::Result result;

    {
        TestNormalMemoryState normal;
        normal.Initialize();

        result = nn::svc::MapMemory(g_FreeAreaBegin, normal.GetAddress(), MapSize * 2);
        ASSERT_RESULT_SUCCESS(result);

        // メモリ状態が一様でないと失敗する
        // 前方
        uintptr_t fromAddr = normal.GetAddress() - MapSize;
        result = nn::svc::UnmapMemory(g_FreeAreaBegin, fromAddr, MapSize * 2);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
                result <= nn::svc::ResultInvalidRegion());

        // 後方
        fromAddr = normal.GetAddress() + normal.GetSize() - MapSize;
        result = nn::svc::UnmapMemory(g_FreeAreaBegin, fromAddr, MapSize * 2);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
                result <= nn::svc::ResultInvalidRegion());

        result = nn::svc::UnmapMemory(g_FreeAreaBegin, normal.GetAddress(), MapSize * 2);
        ASSERT_RESULT_SUCCESS(result);

        normal.Close();
    }
}

TEST(UnmapMemoryTest, ToMemoryStateTest)
{
    nn::Result result;

    {
        TestNormalMemoryState normal;
        normal.Initialize();

        result = nn::svc::MapMemory(g_FreeAreaBegin, normal.GetAddress(), MapSize * 2);
        ASSERT_RESULT_SUCCESS(result);

        // メモリ状態が一様でないと失敗する
        // 前方
        uintptr_t toAddr = g_FreeAreaBegin - MapSize;
        result = nn::svc::UnmapMemory(toAddr, normal.GetAddress(), MapSize * 2);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
                result <= nn::svc::ResultInvalidRegion());

        // 後方
        toAddr = g_FreeAreaBegin + MapSize;
        result = nn::svc::UnmapMemory(toAddr, normal.GetAddress(), MapSize * 2);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
                result <= nn::svc::ResultInvalidRegion());

        result = nn::svc::UnmapMemory(g_FreeAreaBegin, normal.GetAddress(), MapSize * 2);
        ASSERT_RESULT_SUCCESS(result);

        normal.Close();
    }
}

TEST(UnmapMemoryTest, InvalidAlignmentTest)
{
    uintptr_t addr;
    size_t size;
    nn::Result result;
    int i;
    const int MapSize = DefaultAlignSize * 2;

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // Heap Area Check
    ASSERT_TRUE(g_HeapAreaBegin <= addr);
    ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);
    CheckNormalArea(addr, size, nn::svc::MemoryPermission_ReadWrite);

    CheckFreeArea(g_FreeAreaBegin, g_FreeAreaEnd);

    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, MapSize);
    ASSERT_RESULT_SUCCESS(result);
    CheckMapArea(g_FreeAreaBegin, addr, MapSize);

    for (i = 1; i < DefaultAlignSize; i++)
    {
        // TEST 4-8
        // fromAddr は0x1000 にアライメントされていないといけない
        result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr + i, MapSize);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidAddress() ||
                result <= nn::svc::ResultInvalidRegion());

        // TEST 4-9
        // toAddr は0x1000 にアライメントされていないといけない
        result = nn::svc::UnmapMemory(g_FreeAreaBegin + i, addr, MapSize);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidAddress() ||
                result <= nn::svc::ResultInvalidRegion());

        // TEST 4-10
        // size は0x1000 にアライメントされていないといけない
        result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, DefaultAlignSize + i);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
                result <= nn::svc::ResultInvalidRegion());
    }

    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, MapSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapMemoryTest, UnmapPartOfMappedMemoryTest)
{
    uintptr_t addr;
    size_t size;
    nn::Result result;
    const int MapSize = DefaultAlignSize * 2;

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // Heap Area Check
    ASSERT_TRUE(g_HeapAreaBegin <= addr);
    ASSERT_TRUE((addr + size - 1) <= g_HeapAreaEnd);
    CheckNormalArea(addr, size, nn::svc::MemoryPermission_ReadWrite);

    CheckFreeArea(g_FreeAreaBegin, g_FreeAreaEnd);

    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, MapSize);
    ASSERT_RESULT_SUCCESS(result);
    CheckMapArea(g_FreeAreaBegin, addr, MapSize);

    // TEST 4-11
    // マップされた領域の一部分だけを解除することが出来る
    result = nn::svc::UnmapMemory(
            g_FreeAreaBegin + DefaultAlignSize, addr + DefaultAlignSize, DefaultAlignSize);
    CheckFreeArea(g_FreeAreaBegin + DefaultAlignSize, g_FreeAreaEnd);
    CheckNormalArea(
            addr + DefaultAlignSize, size - DefaultAlignSize,
            nn::svc::MemoryPermission_ReadWrite);
    CheckMapArea(g_FreeAreaBegin, addr, DefaultAlignSize);

    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, DefaultAlignSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapMemoryTest, InvalidRegionTest)
{
    nn::Result result;
    uintptr_t addr;

    result = nn::svc::SetHeapSize(&addr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 4-19
    // fromAddr にマップされていない領域をしていすることが出来ない
    result = nn::svc::MapMemory(g_FreeAreaBegin, addr, DefaultAlignSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapMemory(
            g_FreeAreaBegin + DefaultAlignSize, addr + DefaultAlignSize, DefaultAlignSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr + DefaultAlignSize, DefaultAlignSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

    result = nn::svc::UnmapMemory(g_FreeAreaBegin + DefaultAlignSize, addr , DefaultAlignSize);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, DefaultAlignSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapMemory(
            g_FreeAreaBegin + DefaultAlignSize, addr + DefaultAlignSize, DefaultAlignSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapMemoryTest, OverFlowTest)
{
    nn::Result result;
    uintptr_t heapPtr;
    uintptr_t addr = static_cast<uintptr_t>(-1) & ~(0xFFF);

    result = nn::svc::SetHeapSize(&heapPtr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapMemory(g_FreeAreaBegin, heapPtr, DefaultAlignSize);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 4-20
    // オーバーフローする fromAddr,size の組み合わせは許容しない
    result = nn::svc::UnmapMemory(g_FreeAreaBegin, addr, DefaultAlignSize);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

    // TEST 4-21
    // オーバーフローする fromAddr,size の組み合わせは許容しない
    result = nn::svc::UnmapMemory(addr, heapPtr, DefaultAlignSize);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

    result = nn::svc::UnmapMemory(g_FreeAreaBegin, heapPtr, DefaultAlignSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}
