﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>
#include <cstring>


TEST(TransferMemoryTest, CreateTransferMemory)
{
    TestTransferMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    nn::svc::MemoryPermission myPermission;
    size_t size;
    uintptr_t addr;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    const nn::svc::MemoryPermission perm[] =
    {
        nn::svc::MemoryPermission_None,
        nn::svc::MemoryPermission_Read,
        nn::svc::MemoryPermission_Write,
        nn::svc::MemoryPermission_Execute,
        nn::svc::MemoryPermission_ReadWrite,
        nn::svc::MemoryPermission_ReadExecute,
        nn::svc::MemoryPermission_DontCare
    };

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 126-30 から 126-36
    // パーミッションのテスト
    for (int i = 0; i < static_cast<int>(sizeof(perm) / sizeof(*perm)); i++)
    {
        myPermission = perm[i];
        bool myPermissionIsValid = false;
        switch (myPermission)
        {
            case nn::svc::MemoryPermission_None:
            case nn::svc::MemoryPermission_Read:
            case nn::svc::MemoryPermission_ReadWrite:
                {
                    myPermissionIsValid = true;
                }
                break;
            default:
                break;
        }

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

        result = nn::svc::CreateTransferMemory(&handle, addr, size, myPermission);
        if (myPermissionIsValid)
        {
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(blockInfo.permission == myPermission);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());
        }

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    }


    // TEST 126-28
    // メモリ状態が一様でないと失敗する
    {
        result = nn::svc::CreateTransferMemory(&handle, addr, size + 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    }

    {
        uintptr_t freeReservedAreaAddr;
        size_t freeReservedAreaSize;
        ASSERT_TRUE(GetReservedFreeMemoryArea(&freeReservedAreaAddr, &freeReservedAreaSize));
        result = nn::svc::MapMemory(freeReservedAreaAddr, addr, 0x1000);
        ASSERT_RESULT_SUCCESS(result);
        // TEST 126-17
        // MemoryState_Alias の領域は受け付けない
        result = nn::svc::CreateTransferMemory(&handle, freeReservedAreaAddr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, freeReservedAreaAddr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Alias);
        // TEST 126-16
        // MemoryState_Aliased の領域は受け付けない
        // MemoryState_Aliased はなくなったので廃止
        // 代わりにLocked 属性の場合を調べる
        result = nn::svc::CreateTransferMemory(&handle, addr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);
        result = nn::svc::UnmapMemory(freeReservedAreaAddr, addr, 0x1000);
        ASSERT_RESULT_SUCCESS(result);
    }

    {
        // TEST 126-13
        // MemoryState_Free の領域は受け付けない
        result = nn::svc::SetMemoryPermission(addr, 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CreateTransferMemory(&handle, g_FreeAreaBegin, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        result = nn::svc::SetMemoryPermission(addr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
    }

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
} // NOLINT (readability/fn_size)

TEST(TransferMemoryTest, CreateTransferMemoryCodeData)
{
    TestTransferMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    nn::svc::MemoryPermission myPermission;
    size_t size;
    uintptr_t addr;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    static char g_Buffer[0x2000] __attribute__((aligned(0x1000)));

    size = sizeof(g_Buffer);
    addr = reinterpret_cast<uintptr_t>(g_Buffer);

    const nn::svc::MemoryPermission perm[] =
    {
        nn::svc::MemoryPermission_None,
        nn::svc::MemoryPermission_Read,
        nn::svc::MemoryPermission_Write,
        nn::svc::MemoryPermission_Execute,
        nn::svc::MemoryPermission_ReadWrite,
        nn::svc::MemoryPermission_ReadExecute,
        nn::svc::MemoryPermission_DontCare
    };

    for (int i = 0; i < static_cast<int>(sizeof(perm) / sizeof(*perm)); i++)
    {
        myPermission = perm[i];
        bool myPermissionIsValid = false;
        switch (myPermission)
        {
            case nn::svc::MemoryPermission_None:
            case nn::svc::MemoryPermission_Read:
            case nn::svc::MemoryPermission_ReadWrite:
                {
                    myPermissionIsValid = true;
                }
                break;
            default:
                break;
        }

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

        result = nn::svc::CreateTransferMemory(&handle, addr, size, myPermission);
        if (myPermissionIsValid)
        {
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(blockInfo.permission == myPermission);
            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());
        }

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    }

    {
        uintptr_t freeReservedAreaAddr;
        size_t freeReservedAreaSize;
        ASSERT_TRUE(GetReservedFreeMemoryArea(&freeReservedAreaAddr, &freeReservedAreaSize));
        result = nn::svc::MapMemory(freeReservedAreaAddr, addr, 0x1000);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CreateTransferMemory(&handle, freeReservedAreaAddr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, freeReservedAreaAddr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Alias);
        result = nn::svc::CreateTransferMemory(&handle, addr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);
        result = nn::svc::UnmapMemory(freeReservedAreaAddr, addr, 0x1000);
        ASSERT_RESULT_SUCCESS(result);
    }

    {
        result = nn::svc::SetMemoryPermission(addr, 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CreateTransferMemory(&handle, g_FreeAreaBegin, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        result = nn::svc::SetMemoryPermission(addr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
    }
} // NOLINT (readability/fn_size)

TEST(TransferMemoryTest, MapTransferMemory)
{
    TestTransferMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t addr;

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    for (int i = 0; i < 3; i++)
    {
        // TEST 127-28, 127-29
        // CreateTransferMemory で指定したパーミッションのみを受け付ける
        *reinterpret_cast<uint32_t*>(addr) = 0xcccccccc;
        result = nn::svc::CreateTransferMemory(&handle, addr, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Execute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadExecute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_DontCare);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        // TEST 127-9
        // size が CreateTransferMemory と一致していないと失敗する
        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin + size, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);

        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(g_FreeAreaBegin) == 0xcccccccc);
        *reinterpret_cast<uint32_t*>(g_FreeAreaBegin) = 0xdeadbeef;
        asm volatile ("dmb ish":::"memory");
        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(addr) == 0xdeadbeef);
        *reinterpret_cast<uint32_t*>(addr) = 0xbabeface;
        asm volatile ("dmb ish":::"memory");
        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(g_FreeAreaBegin) == 0xbabeface);
        *reinterpret_cast<uint32_t*>(g_FreeAreaBegin) = 0xdeadbeef;
        asm volatile ("dmb ish":::"memory");

        // TEST 128-28
        // 一部分だけ解除することが出来ない
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size / 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
        // TEST 128-26
        // メモリ状態が一様でないと失敗する
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin + size / 2, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        // TEST 128-9
        // CreateTransferMemory で指定したサイズと一致していると成功する
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_SUCCESS(result);
        // TEST 128-11
        // MemoryState_Free の領域は受け付けない
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(addr) == 0xdeadbeef);
    }

    for (int i = 0; i < 3; i++)
    {
        // TEST 127-28, 127-29
        // CreateTransferMemory で指定したパーミッションのみを受け付ける
        *reinterpret_cast<uint32_t*>(addr) = 0xcccccccc;
        result = nn::svc::CreateTransferMemory(&handle, addr, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Execute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadExecute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_DontCare);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        // TEST 127-9
        // size が CreateTransferMemory と一致していないと失敗する
        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin + size, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_SharedTransfered);

        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(g_FreeAreaBegin) == 0xcccccccc);
        *reinterpret_cast<uint32_t*>(g_FreeAreaBegin) = 0xdeadbeef;
        asm volatile ("dmb ish":::"memory");
        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(addr) == 0xdeadbeef);

        // TEST 128-28
        // 一部分だけ解除することが出来ない
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size / 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
        // TEST 128-26
        // メモリ状態が一様でないと失敗する
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin + size / 2, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        // TEST 128-9
        // CreateTransferMemory で指定したサイズと一致していると成功する
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_SUCCESS(result);
        // TEST 128-11
        // MemoryState_Free の領域は受け付けない
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(addr) == 0xdeadbeef);
    }


    for (int i = 0; i < 3; i++)
    {
        // TEST 127-28, 127-29
        // CreateTransferMemory で指定したパーミッションのみを受け付ける
        *reinterpret_cast<uint32_t*>(addr) = 0xcccccccc;
        result = nn::svc::CreateTransferMemory(&handle, addr, size, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Execute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadExecute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_DontCare);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        // TEST 127-9
        // size が CreateTransferMemory と一致していないと失敗する
        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, 0x1000, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapTransferMemory(handle, g_FreeAreaBegin + size, size, nn::svc::MemoryPermission_None);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Transfered);

        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(g_FreeAreaBegin) == 0xcccccccc);
        *reinterpret_cast<uint32_t*>(g_FreeAreaBegin) = 0xdeadbeef;
        asm volatile ("dmb ish":::"memory");

        // TEST 128-28
        // 一部分だけ解除することが出来ない
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size / 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
        // TEST 128-26
        // メモリ状態が一様でないと失敗する
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin + size / 2, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        // TEST 128-9
        // CreateTransferMemory で指定したサイズと一致していると成功する
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_SUCCESS(result);
        // TEST 128-11
        // MemoryState_Free の領域は受け付けない
        result = nn::svc::UnmapTransferMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(*reinterpret_cast<uint32_t*>(addr) == 0xdeadbeef);
    }


    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
} // NOLINT (readability/fn_size)

