﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestIpc.h"
#include "util_TestLoader.h"
#include "util_TestProcess.h"
#include "test_TerminateProcess.h"
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <nn/util/util_FormatString.h>
#include <cstring>

#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
#include "test_TestTmrDevice_Jetson.h"
#endif // defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1)

extern nn::Result ServerReceiveLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReplyLightRequest(nn::svc::Handle handle);

extern char BinTerminate_begin[];
extern char BinTerminate_end[];

namespace {

const int64_t SleepTime = 100 * 1000 * 1000; // 100 msec
char g_IpcBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_IpcRecvBuffer[2][0x1000] __attribute__((aligned(0x1000)));
char g_IpcMapBuffer[MapDefaultSize * 3] __attribute__((aligned(0x1000)));

TestLoader* g_TestLoader;

class TestCondition
{
    public:
        TestCondition()
            : mSendMethod(-1), mRecvMethod(-1), mSessionIndex(-1), mDataIndex(-1), mEnd(false)
        {
        }

        bool Next()
        {
            if (mEnd)
            {
                return false;
            }

            if (mSendMethod == -1)
            {
                mSendMethod = 0;
                mRecvMethod = 0;
                mSessionIndex = 0;
                mDataIndex = 0;
                return true;
            }

            mSendMethod++;

            if (mSendMethod >= MaxSendIndex)
            {
                mSendMethod = 0;
                mRecvMethod++;
            }

            if(mRecvMethod >= MaxReceiveIndex)
            {
                mRecvMethod = 0;
                mDataIndex++;
            }

            if (mDataIndex >= MaxDataTypeIndex)
            {
                mDataIndex = 0;
                mSessionIndex++;
            }

            if (mSessionIndex >= MaxSessionIndex)
            {
                mEnd = true;
                return false;
            }

            return true;
        }

        // Next 前に呼ばれるとデータアボートになるので注意
        SendMethod GetSendMethod() const
        {
            return SendMethodArray[mSendMethod];
        }
        ReceiveMethod GetReceiveMethod() const
        {
            return ReceiveMethodArray[mRecvMethod];
        }
        IpcDataType GetDataType() const
        {
            return IpcDataTypeArray[mDataIndex];
        }
        SessionType GetSessionType() const
        {
            return SessionTypeArray[mSessionIndex];
        }
        bool GetServerBuffer() const
        {
            return mRecvMethod > 0;
        }

        const char* GetScopedString()
        {
            nn::util::SNPrintf(mBuf, BufferSize, "Send: %d, Recv: %d, Type: %d, Session: %d\n",
                    mSendMethod, mRecvMethod, mDataIndex, mSessionIndex);
            return mBuf;
        }

    private:
        static const int BufferSize = 64;
        static const int MaxSendIndex = sizeof(SendMethodArray) / sizeof(SendMethod);
        static const int MaxReceiveIndex = sizeof(ReceiveMethodArray) / sizeof(ReceiveMethod);
#ifdef SVC_AGING_TEST
        static const int MaxDataTypeIndex = sizeof(IpcDataTypeArray) / sizeof(IpcDataType);
#else
        static const int MaxDataTypeIndex = 1; // All だけ実施
#endif
        static const int MaxSessionIndex = sizeof(SessionTypeArray) / sizeof(SessionType);

        int32_t mSendMethod;
        int32_t mRecvMethod;
        int32_t mSessionIndex;
        int32_t mDataIndex;
        bool mEnd;
        char mBuf[BufferSize];
};

void StartBinTerminateProcess()
{
    nn::svc::Handle handle;
    g_TestLoader->SpawnProcess(&handle);
    g_TestLoader->StartProcess(handle);

    // 実行後ハンドルを管理したくないので、先に閉じてしまう
    nn::Result result = nn::svc::CloseHandle(handle);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void CheckTag(uint16_t tag)
{
    ASSERT_TRUE(GetIpcTag(nn::svc::ipc::GetMessageBuffer()) == tag);
}

void WaitClient(nn::svc::Handle *pClientSession, nn::svc::Handle *pClientProcess,
                nn::svc::Handle namedPort)
{
    nn::Result result;

    // Accept Session
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &namedPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(pClientSession, namedPort);
    ASSERT_RESULT_SUCCESS(result);

    // Receive Request
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::ReplyAndReceive(
                &index, pClientSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    CheckTag(static_cast<uint16_t>(TestTag_Init));

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
    ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
    *pClientProcess = ipcMsg.GetHandle(offset);
}

void TestReadEvent(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test Read Event Start +++\n");
    nn::Result result;
    StartBinTerminateProcess();

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // 読み込みイベントをクライアントにムーブ
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_ReadEvent, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 0, 1);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = ipcMsg.SetHandle(offset, readableEvent);

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &clientSession, 0, clientSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // クライアントが読み込みイベントを待つのを待つ
    nn::svc::SleepThread(SleepTime);

    // クライアントプロセスを終了させる
    result = nn::svc::TerminateProcess(clientProcess);
    ASSERT_RESULT_SUCCESS(result);

    // readalbe event がクローズされていてもシグナル出来る
    result = nn::svc::SignalEvent(writableEvent);
    ASSERT_RESULT_SUCCESS(result);

    NN_LOG("--- Test Read Event END ---\n");
}

void TestWriteEvent(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test Write Event Start +++\n");
    StartBinTerminateProcess();
    nn::Result result;

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // 書き込みイベントをクライアントにムーブ
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_WriteEvent, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 0, 1);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = ipcMsg.SetHandle(offset, writableEvent);

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &clientSession, 1, clientSession, -1);
    ASSERT_RESULT_SUCCESS(result);

    // クライアントプロセスはシグナルを送った後、返信を待っている
    // クライアントプロセスを終了させる
    result = nn::svc::TerminateProcess(clientProcess);
    ASSERT_RESULT_SUCCESS(result);

    // writable event がクローズされていてもシグナル出来る
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    NN_LOG("--- Test Write Event END ---\n");
}

void SetHeaderWithTag(uint16_t tag, bool isBuffer)
{
    nn::Bit32* pMsgBuffer;
    if (isBuffer)
    {
        pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_IpcBuffer);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    }

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(tag, 0, 0, 0, 0, 0, 0, 0);
    ipcMsg.Set(ipcHeader);
}

nn::Result ReceiveRequest(nn::svc::Handle clientSession, ReceiveMethod method)
{
    nn::Result result;
    int32_t index;

    if (method == ReceiveMethod_UseBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, reinterpret_cast<uintptr_t>(g_IpcBuffer), sizeof(g_IpcBuffer),
                    &clientSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &clientSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    return result;
}

nn::Result ReplyRequest(nn::svc::Handle clientSession, ReceiveMethod method)
{
    nn::Result result;
    int32_t index;

    if (method == ReceiveMethod_UseBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, reinterpret_cast<uintptr_t>(g_IpcBuffer), sizeof(g_IpcBuffer),
                    &clientSession, 1, clientSession, 0);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(&index, &clientSession, 0, clientSession, 0);
    }
    return result;
}

nn::Result SendWithMethod(nn::svc::Handle clientSession, SendMethod method)
{
    nn::Result result;
    nn::svc::Handle eventHandle;
    nn::Result tmpResult;

    switch (method)
    {
    case SendMethod_UseTls:
        result = nn::svc::SendSyncRequest(clientSession);
        break;
    case SendMethod_UseBuffer:
        result = nn::svc::SendSyncRequestWithUserBuffer(
                    reinterpret_cast<uintptr_t>(g_IpcBuffer), sizeof(g_IpcBuffer), clientSession);
        break;
    case SendMethod_UseAsync:
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandle, reinterpret_cast<uintptr_t>(g_IpcBuffer), sizeof(g_IpcBuffer),
                    clientSession);
        if (result.IsFailure())
        {
            return result;
        }
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
        tmpResult = nn::svc::CloseHandle(eventHandle);
        NN_ASSERT_RESULT_SUCCESS(tmpResult);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    return result;
}

void GetReceiveMethodBuffer(nn::Bit32** pMsgBuffer, size_t* pMsgSize, ReceiveMethod method)
{
    if (method == ReceiveMethod_UseTls)
    {
        *pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        *pMsgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    else
    {
        *pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_IpcBuffer);
        *pMsgSize = sizeof(g_IpcBuffer);
    }
}

void GetSendBuffer(nn::Bit32** pMsgBuffer, size_t* pMsgSize, SendMethod method)
{
    if (method == SendMethod_UseTls)
    {
        *pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        *pMsgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    else
    {
        *pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_IpcBuffer);
        *pMsgSize = sizeof(g_IpcBuffer);
    }

}

void PrepareIpcTest(
        nn::svc::Handle clientSession, nn::svc::Handle targetHandle,
        TestTag testTag, TestInfo* info, bool isMove = true)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    pMsgBuffer = nn::svc::ipc::GetMessageBuffer();

    SetHeaderWithTag(static_cast<uint16_t>(testTag), false);
    result = ReplyRequest(clientSession, ReceiveMethod_UseTls);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    SetHeaderWithTag(static_cast<uint16_t>(testTag), false);
    result = ReceiveRequest(clientSession, ReceiveMethod_UseTls);
    ASSERT_RESULT_SUCCESS(result);
    CheckTag(static_cast<uint16_t>(testTag));

    int copy = isMove ? 0 : 1;
    int move = isMove ? 1 : 0;
    size_t bit32Size = sizeof(nn::Bit32);
    int rawNum = ((sizeof(TestInfo) + bit32Size - 1) & ~(bit32Size - 1)) / bit32Size;
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(testTag, 1, 0, 0, 0, 0, rawNum, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, copy, move);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = ipcMsg.SetHandle(offset, targetHandle);
    offset = ipcMsg.SetRawArray(offset, info, sizeof(TestInfo));

    result = ReplyRequest(clientSession, ReceiveMethod_UseTls);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void MakeNamedPortSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    const char* portName = "NamedPort";
    result = nn::svc::ManageNamedPort(&serverPort, portName, 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);

    result = nn::svc::ConnectToNamedPort(pOutClientSession, portName);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(pOutServerSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ManageNamedPort(&serverPort, portName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

void MakePortSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    result = nn::svc::ConnectToPort(pOutClientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(pOutServerSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

void MakeSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    result = nn::svc::CreateSession(pOutServerSession, pOutClientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
}

void MakeTestSession(
        nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession,
        SessionType sessionType)
{
    switch(sessionType)
    {
        case SessionType_NamedPort:
            MakeNamedPortSession(pOutServerSession, pOutClientSession);
            break;
        case SessionType_Port:
            MakePortSession(pOutServerSession, pOutClientSession);
            break;
        case SessionType_Session:
            MakeSession(pOutServerSession, pOutClientSession);
            break;
        default: FAIL();
    }
}

nn::Result ReceiveTestData(nn::svc::Handle session, ReceiveMethod method, IpcDataType dataType)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetReceiveMethodBuffer(&pMsgBuffer, &msgSize, method);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    int recvListNum;
    switch(dataType)
    {
    case IpcDataType_Pointer_IpcBuffer:
        recvListNum = 1;
        break;
    case IpcDataType_Pointer_OneBuffer:
        recvListNum = 2;
        break;
    case IpcDataType_Pointer_MultiBuffer:
    case IpcDataType_All:
        recvListNum = 4;
        break;
    default:
        recvListNum = 0;
    }
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, recvListNum);
    int offset = ipcMsg.Set(ipcHeader);
    if (recvListNum == 2)
    {
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvEntry(g_IpcRecvBuffer, sizeof(g_IpcRecvBuffer));
        offset = ipcMsg.Set(offset, recvEntry);
    }
    else if (recvListNum == 4)
    {
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvEntry1(&g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));
        offset = ipcMsg.Set(offset, recvEntry1);
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvEntry2(&g_IpcRecvBuffer[1], sizeof(g_IpcRecvBuffer[1]));
        offset = ipcMsg.Set(offset, recvEntry2);
    }

    result = ReceiveRequest(session, method);
    return result;
}

void CheckClientData(
        CheckInfo* checkInfo, ReceiveMethod method, IpcDataType dataType, bool isClientAlive)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetReceiveMethodBuffer(&pMsgBuffer, &msgSize, method);

    // For Send Pointer
    uintptr_t ptrAddr = reinterpret_cast<uintptr_t>(&g_IpcRecvBuffer[0]);
    size_t bufSize = sizeof(g_IpcRecvBuffer[0]);
    uintptr_t buffers[2] =
    {
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[0]),
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[1])
    };
    size_t bufferSizes[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };

    switch(dataType)
    {
    case IpcDataType_ProcessId:
        CheckOnlyProcessIdData(checkInfo->processHandle, pMsgBuffer);
        break;
    case IpcDataType_CopyHandle:
        CheckOnlyCopyHandleData(pMsgBuffer, isClientAlive);
        break;
    case IpcDataType_MoveHandle:
        break;
    case IpcDataType_Pointer_IpcBuffer:
        CheckOnlyPointerWithIpcBufferData(
                SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer, msgSize);
        break;
    case IpcDataType_Pointer_OneBuffer:
        CheckOnlyPointerWithOneUserBuffer(
                ptrAddr, bufSize, SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer);
        break;
    case IpcDataType_Pointer_MultiBuffer:
        CheckOnlyPointerWithMultiUserBuffer(
                buffers, bufferSizes,
                SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer);
        break;
    case IpcDataType_Map_Send:
        CheckOnlyMapSendData(&checkInfo->mapSendAddr, MapSendSize, SendMapSendData, pMsgBuffer);
        checkInfo->mapSendSize = MapSendSize;
        break;
    case IpcDataType_Map_Receive:
        CheckOnlyMapReceiveData(&checkInfo->mapReceiveAddr, MapRecvSize, SendMapRecvData, pMsgBuffer);
        checkInfo->mapReceiveSize = MapRecvSize;
        break;
    case IpcDataType_Map_Exchange:
        CheckOnlyMapExchangeData(&checkInfo->mapExchangeAddr, MapExchSize, SendMapExchData, pMsgBuffer);
        checkInfo->mapExchangeSize = MapExchSize;
        break;
    case IpcDataType_Raw:
        CheckOnlyRawData(SendRawData, SendRawDataSize, pMsgBuffer);
        break;
    case IpcDataType_All:
        CheckAllData(
            &checkInfo->mapSendAddr, &checkInfo->mapReceiveAddr, &checkInfo->mapExchangeAddr,
            checkInfo->processHandle, isClientAlive, false,
            SendPointData, SendPointDataSize, SendPointDataNum,
            buffers, bufferSizes,
            MapDefaultSize, SendMapSendData,
            MapDefaultSize, SendMapRecvData,
            MapDefaultSize, SendMapExchData,
            SendRawData, SendRawDataSize,
            pMsgBuffer
            );
        checkInfo->mapSendSize = MapDefaultSize;
        checkInfo->mapReceiveSize = MapDefaultSize;
        checkInfo->mapExchangeSize = MapDefaultSize;
        break;
    default: FAIL();
    }
}

void CheckServerData(
        CheckInfo* checkInfo, SendMethod method, IpcDataType dataType, bool isClientAlive)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetSendBuffer(&pMsgBuffer, &msgSize, method);

    // For Send Pointer
    uintptr_t ptrAddr = reinterpret_cast<uintptr_t>(&g_IpcRecvBuffer[0]);
    size_t bufSize = sizeof(g_IpcRecvBuffer[0]);
    uintptr_t buffers[2] =
    {
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[0]),
        reinterpret_cast<uintptr_t>(g_IpcRecvBuffer[1])
    };
    size_t bufferSizes[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };

    nn::Bit64 pid;
    nn::Result result = nn::svc::GetProcessId(&pid, checkInfo->processHandle);
    ASSERT_RESULT_SUCCESS(result);

    switch(dataType)
    {
    case IpcDataType_ProcessId:
        CheckOnlyProcessIdData(checkInfo->processHandle, pMsgBuffer);
        break;
    case IpcDataType_CopyHandle:
        CheckOnlyCopyHandleData(pMsgBuffer, isClientAlive);
        break;
    case IpcDataType_MoveHandle:
        CheckOnlyMoveHandleData(pMsgBuffer, isClientAlive);
        break;
    case IpcDataType_Pointer_IpcBuffer:
        CheckOnlyPointerWithIpcBufferData(
                SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer, msgSize);
        break;
    case IpcDataType_Pointer_OneBuffer:
        CheckOnlyPointerWithOneUserBuffer(
                ptrAddr, bufSize, SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer);
        break;
    case IpcDataType_Pointer_MultiBuffer:
        CheckOnlyPointerWithMultiUserBuffer(
                buffers, bufferSizes,
                SendPointData, SendPointDataSize, SendPointDataNum, pMsgBuffer);
        break;
    case IpcDataType_Map_Send:
        checkInfo->mapSendAddr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
        checkInfo->mapSendSize = MapSendSize;
        break;
    case IpcDataType_Map_Receive:
        CmpMap(reinterpret_cast<uint8_t*>(g_IpcMapBuffer), MapRecvSize, SendMapRecvData);
        checkInfo->mapSendAddr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
        checkInfo->mapReceiveSize = MapRecvSize;
        break;
    case IpcDataType_Map_Exchange:
        CmpMap(reinterpret_cast<uint8_t*>(g_IpcMapBuffer), MapExchSize, SendMapExchData);
        checkInfo->mapSendAddr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
        checkInfo->mapReceiveSize = MapExchSize;
        break;
    case IpcDataType_Raw:
        CheckOnlyRawData(SendRawData, SendRawDataSize, pMsgBuffer);
        break;
    case IpcDataType_All:
        CheckAllDataFromClient(
            pid, true,
            SendPointData, SendPointDataSize, SendPointDataNum,
            buffers, bufferSizes,
            reinterpret_cast<uint8_t*>(&g_IpcMapBuffer[0]), MapDefaultSize, SendMapSendData,
            reinterpret_cast<uint8_t*>(&g_IpcMapBuffer[MapDefaultSize]), MapDefaultSize, SendMapRecvData,
            reinterpret_cast<uint8_t*>(&g_IpcMapBuffer[MapDefaultSize * 2]), MapDefaultSize, SendMapExchData,
            SendRawData, SendRawDataSize,
            pMsgBuffer
            );
        checkInfo->mapSendAddr = reinterpret_cast<uintptr_t>(&g_IpcMapBuffer[0]);
        checkInfo->mapReceiveAddr = reinterpret_cast<uintptr_t>(&g_IpcMapBuffer[MapDefaultSize]);
        checkInfo->mapExchangeAddr = reinterpret_cast<uintptr_t>(&g_IpcMapBuffer[MapDefaultSize * 2]);
        checkInfo->mapSendSize = MapDefaultSize;
        checkInfo->mapReceiveSize = MapDefaultSize;
        checkInfo->mapExchangeSize = MapDefaultSize;
        break;
    default: FAIL();
    }
}

void CheckServerMapArea(uintptr_t addr, size_t size, bool isUsed, bool isSendOnly)
{
    nn::svc::MemoryInfo blockInfo;

    GetMemoryInfo(&blockInfo, addr);

    ASSERT_TRUE(blockInfo.baseAddress <= addr);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size - 1 >= addr + size - 1);

    if (isUsed)
    {
        //ASSERT_RESULT_SUCCESS(blockInfo.state == nn::svc::MemoryState_Ipc);
        if (isSendOnly)
        {
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
        }
        else
        {
            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        }
    }
    else
    {
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    }
}

void CheckServerMemoryArea(
        CheckInfo* checkInfo, ReceiveMethod method, IpcDataType dataType, bool isUsed)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    GetReceiveMethodBuffer(&pMsgBuffer, &msgSize, method);
    uintptr_t addr = reinterpret_cast<uintptr_t>(pMsgBuffer);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.baseAddress <= addr);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= addr + msgSize);
    if (isUsed)
    {
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    }
    else
    {
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    }

    switch(dataType)
    {
    case IpcDataType_Map_Send:
        CheckServerMapArea(checkInfo->mapSendAddr, checkInfo->mapSendSize, isUsed, true);
        break;
    case IpcDataType_Map_Receive:
        CheckServerMapArea(checkInfo->mapReceiveAddr, checkInfo->mapReceiveSize, isUsed, false);
        break;
    case IpcDataType_Map_Exchange:
        CheckServerMapArea(checkInfo->mapExchangeAddr, checkInfo->mapExchangeSize, isUsed, false);
        break;
    case IpcDataType_All:
        CheckServerMapArea(checkInfo->mapSendAddr, checkInfo->mapSendSize, isUsed, true);
        CheckServerMapArea(checkInfo->mapReceiveAddr, checkInfo->mapReceiveSize, isUsed, false);
        CheckServerMapArea(checkInfo->mapExchangeAddr, checkInfo->mapExchangeSize, isUsed, false);
        break;
    default:
        break;
    }
}

void CheckClientMapArea(uintptr_t addr, size_t size, bool isUsed)
{
    nn::svc::MemoryInfo blockInfo;

    GetMemoryInfo(&blockInfo, addr);

    ASSERT_TRUE(blockInfo.baseAddress <= addr);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= addr + size);

    if (isUsed)
    {
        //ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_IpcLocked);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    }
    else
    {
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    }
}

void CheckClientMemoryArea(CheckInfo* checkInfo, SendMethod method, IpcDataType dataType, bool isUsed)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    GetSendBuffer(&pMsgBuffer, &msgSize, method);
    uintptr_t addr = reinterpret_cast<uintptr_t>(pMsgBuffer);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.baseAddress <= addr);
    ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= addr + msgSize);
    if (isUsed)
    {
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    }
    else
    {
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    }

    switch(dataType)
    {
    case IpcDataType_Map_Send:
        CheckClientMapArea(checkInfo->mapSendAddr, checkInfo->mapSendSize, isUsed);
        break;
    case IpcDataType_Map_Receive:
        CheckClientMapArea(checkInfo->mapReceiveAddr, checkInfo->mapReceiveSize, isUsed);
        break;
    case IpcDataType_Map_Exchange:
        CheckClientMapArea(checkInfo->mapExchangeAddr, checkInfo->mapExchangeSize, isUsed);
        break;
    case IpcDataType_All:
        CheckClientMapArea(checkInfo->mapSendAddr, checkInfo->mapSendSize, isUsed);
        CheckClientMapArea(checkInfo->mapReceiveAddr, checkInfo->mapReceiveSize, isUsed);
        CheckClientMapArea(checkInfo->mapExchangeAddr, checkInfo->mapExchangeSize, isUsed);
        break;
    default:
        break;
    }
}

void CheckAfterReply(CheckInfo* info, ReceiveMethod method, IpcDataType dataType, bool isSuccess)
{
    NN_UNUSED(isSuccess);
    CheckServerMemoryArea(info, method, dataType, false);
    nn::Result result;

    switch(dataType)
    {
    case IpcDataType_CopyHandle:
        result = nn::svc::CloseHandle(info->copyHandle);
        ASSERT_RESULT_SUCCESS(result);
        break;
    case IpcDataType_MoveHandle:
        result = nn::svc::CloseHandle(info->moveHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        break;
    case IpcDataType_All:
        result = nn::svc::CloseHandle(info->copyHandle);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(info->moveHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        break;
    default:
        break;
    }
}

void CheckAfterSend(CheckInfo* info, SendMethod method, IpcDataType dataType, bool isSuccess)
{
    NN_UNUSED(isSuccess);

    CheckClientMemoryArea(info, method, dataType, false);
    nn::Result result;

    switch(dataType)
    {
    case IpcDataType_CopyHandle:
        result = nn::svc::CloseHandle(info->copyHandle);
        ASSERT_RESULT_SUCCESS(result);
        break;
    case IpcDataType_MoveHandle:
        result = nn::svc::CloseHandle(info->moveHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        break;
    case IpcDataType_All:
        result = nn::svc::CloseHandle(info->copyHandle);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(info->moveHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        break;
    default:
        break;
    }
}

void MakeServerIpcData(CheckInfo* info, nn::Bit32* pMsgBuffer, size_t msgSize, IpcDataType dataType)
{
    NN_UNUSED(msgSize);
    // For Send Pointer
    size_t bufferSize[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };
    char* buffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };

    switch(dataType)
    {
    case IpcDataType_ProcessId:
        MakeOnlyProcessIdData(pMsgBuffer);
        break;
    case IpcDataType_CopyHandle:
        MakeOnlyCopyHandleData(&info->copyHandle, pMsgBuffer);
        break;
    case IpcDataType_MoveHandle:
        MakeOnlyMoveHandleData(&info->moveHandle, pMsgBuffer);
        break;
    case IpcDataType_Pointer_IpcBuffer:
        MakeOnlyPointerWithIpcBufferData(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum);
        break;
    case IpcDataType_Pointer_OneBuffer:
        MakeOnlyPointerWithOneUserBuffer(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
                g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));
        break;
    case IpcDataType_Pointer_MultiBuffer:
        MakeOnlyPointerWithMultiUserBuffer(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
                buffers, bufferSize);
        break;
    case IpcDataType_Map_Send:
        break;
    case IpcDataType_Map_Receive:
        MakeOnlyMapReceiveData(
                pMsgBuffer, reinterpret_cast<char*>(info->mapReceiveAddr), info->mapReceiveSize,
                SendMapRecvData);
        break;
    case IpcDataType_Map_Exchange:
        MakeOnlyMapExchangeData(
                pMsgBuffer, reinterpret_cast<char*>(info->mapExchangeAddr), info->mapExchangeSize,
                SendMapExchData);
        break;
    case IpcDataType_Raw:
        MakeOnlyRawData(pMsgBuffer, SendRawData, SendRawDataSize);
        break;
    case IpcDataType_All:
        MakeAllData(
                &info->copyHandle, &info->moveHandle,
                pMsgBuffer,
                SendPointData, SendPointDataSize, SendPointDataNum,
                reinterpret_cast<char*>(info->mapSendAddr), 0,
                reinterpret_cast<char*>(info->mapReceiveAddr), info->mapReceiveSize, SendMapRecvData,
                reinterpret_cast<char*>(info->mapExchangeAddr), info->mapExchangeSize, SendMapExchData,
                SendRawData, SendRawDataSize,
                buffers, bufferSize, 2
                );
        break;
    default: FAIL();
    }
}

void MakeClientIpcData(CheckInfo* info, nn::Bit32* pMsgBuffer, size_t msgSize, IpcDataType dataType)
{
    NN_UNUSED(msgSize);
    // For Send Pointer
    size_t bufferSize[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };
    char* buffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };

    info->mapSendAddr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
    info->mapReceiveAddr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
    info->mapExchangeAddr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer);
    info->mapSendSize = MapSendSize;
    info->mapReceiveSize = MapRecvSize;
    info->mapExchangeSize = MapExchSize;

    switch(dataType)
    {
    case IpcDataType_ProcessId:
        MakeOnlyProcessIdData(pMsgBuffer);
        break;
    case IpcDataType_CopyHandle:
        MakeOnlyCopyHandleData(&info->copyHandle, pMsgBuffer);
        break;
    case IpcDataType_MoveHandle:
        MakeOnlyMoveHandleData(&info->moveHandle, pMsgBuffer);
        break;
    case IpcDataType_Pointer_IpcBuffer:
        MakeOnlyPointerWithIpcBufferData(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum);
        break;
    case IpcDataType_Pointer_OneBuffer:
        MakeOnlyPointerWithOneUserBuffer(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
                g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));
        break;
    case IpcDataType_Pointer_MultiBuffer:
        MakeOnlyPointerWithMultiUserBuffer(
                pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
                buffers, bufferSize);
        break;
    case IpcDataType_Map_Send:
        MakeOnlyMapSendData(pMsgBuffer, g_IpcMapBuffer, MapSendSize, SendMapSendData);
        break;
    case IpcDataType_Map_Receive:
        MakeOnlyMapReceiveData(pMsgBuffer, g_IpcMapBuffer, MapRecvSize, SendMapRecvData);
        break;
    case IpcDataType_Map_Exchange:
        MakeOnlyMapExchangeData(pMsgBuffer, g_IpcMapBuffer, MapExchSize, SendMapExchData);
        break;
    case IpcDataType_Raw:
        MakeOnlyRawData(pMsgBuffer, SendRawData, SendRawDataSize);
        break;
    case IpcDataType_All:
        info->mapReceiveAddr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer + MapDefaultSize);
        info->mapExchangeAddr = reinterpret_cast<uintptr_t>(g_IpcMapBuffer + MapDefaultSize * 2);
        info->mapSendSize = MapDefaultSize;
        info->mapReceiveSize = MapDefaultSize;
        info->mapExchangeSize = MapDefaultSize;

        MakeAllData(
                &info->copyHandle, &info->moveHandle,
                pMsgBuffer,
                SendPointData, SendPointDataSize, SendPointDataNum,
                g_IpcMapBuffer, MapDefaultSize, SendMapSendData,
                g_IpcMapBuffer + MapDefaultSize, MapDefaultSize, SendMapRecvData,
                g_IpcMapBuffer + MapDefaultSize * 2, MapDefaultSize, SendMapExchData,
                SendRawData, SendRawDataSize,
                buffers, bufferSize, 2
                );
        break;
    default: FAIL();
    }
}

void TestClientReceive(nn::svc::Handle namedPort, TestTag testTag)
{
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetClientSession, testTag, &info);

        nn::svc::Handle readEvent;
        ReceiveMoveHandle(&readEvent, clientSession);
        AutoHandleClose rEventCloser(readEvent);

        // 相手が送信済みであることを確認
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        // 受信確認
        CheckInfo checkInfo;
        checkInfo.processHandle = clientProcess;
        result = ReceiveTestData(targetServerSession, cond.GetReceiveMethod(), cond.GetDataType());
        ASSERT_RESULT_SUCCESS(result);
        CheckClientData(&checkInfo, cond.GetReceiveMethod(), cond.GetDataType(), true);

        nn::Bit32* pMsgBuffer;
        size_t msgSize;
        GetReceiveMethodBuffer(&pMsgBuffer, &msgSize, cond.GetReceiveMethod());

        // 送信データの作成
        MakeServerIpcData(&checkInfo, pMsgBuffer, msgSize, cond.GetDataType());

        // 返答を返す前にクライアントプロセスを終了させる
        result = nn::svc::TerminateProcess(clientProcess);
        ASSERT_RESULT_SUCCESS(result);

        // 返信
        result = ReplyRequest(targetServerSession, cond.GetReceiveMethod());
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

        CheckAfterReply(&checkInfo, cond.GetReceiveMethod(), cond.GetDataType(), false);
    }
}

void TestServerSend(nn::svc::Handle namedPort, TestTag testTag)
{
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetServerSession, testTag, &info);

        nn::svc::Handle writeEvent;
        ReceiveMoveHandle(&writeEvent, clientSession);
        AutoHandleClose wEventCloser(writeEvent);

        // 送信データ作成
        nn::Bit32* pMsgBuffer;
        size_t msgSize;
        GetSendBuffer(&pMsgBuffer, &msgSize, cond.GetSendMethod());
        CheckInfo checkInfo;
        MakeClientIpcData(&checkInfo, pMsgBuffer, msgSize, cond.GetDataType());

        result = nn::svc::SignalEvent(writeEvent);
        ASSERT_RESULT_SUCCESS(result);

        result = SendWithMethod(targetClientSession, cond.GetSendMethod());

        // 相手プロセスが終了している
        if (cond.GetSendMethod() == SendMethod_UseAsync)
        {
            if (result.IsFailure())
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
            }
            else
            {
                CheckAsyncFailure(pMsgBuffer);
            }
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
        }

        CheckAfterSend(&checkInfo, cond.GetSendMethod(), cond.GetDataType(), false);
    }
}

void TestServerReply(nn::svc::Handle namedPort, TestTag testTag)
{
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetServerSession, testTag, &info);

        // 送信データ作成
        nn::Bit32* pMsgBuffer;
        size_t msgSize;
        GetSendBuffer(&pMsgBuffer, &msgSize, cond.GetSendMethod());
        CheckInfo checkInfo;
        MakeClientIpcData(&checkInfo, pMsgBuffer, msgSize, cond.GetDataType());

        result = SendWithMethod(targetClientSession, cond.GetSendMethod());

        // 相手プロセスが終了しているが、受信できる
        ASSERT_RESULT_SUCCESS(result);

        checkInfo.processHandle = clientProcess;
        CheckServerData(&checkInfo, cond.GetSendMethod(), cond.GetDataType(), false);
        CheckAfterSend(&checkInfo, cond.GetSendMethod(), cond.GetDataType(), true);
    }
}

#ifdef SEND_ASYNC_REPLY_TEST
void TestClientReply(nn::svc::Handle namedPort, TestTag testTag)
{
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetClientSession, testTag, &info);

        result = ReceiveTestData(targetServerSession, cond.GetReceiveMethod(), cond.GetDataType());
        ASSERT_RESULT_SUCCESS(result);

        // 受信確認
        CheckInfo checkInfo;
        checkInfo.processHandle = clientProcess;
        CheckClientData(&checkInfo, cond.GetReceiveMethod(), cond.GetDataType(), true);

        // 送信データの作成
        nn::Bit32* pMsgBuffer;
        size_t msgSize;
        GetReceiveMethodBuffer(&pMsgBuffer, &msgSize, cond.GetReceiveMethod());
        MakeServerIpcData(&checkInfo, pMsgBuffer, msgSize, cond.GetDataType());

        result = ReplyRequest(targetServerSession, cond.GetDataType());
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        result = nn::svc::TerminateProcess(clientProcess);
        ASSERT_RESULT_SUCCESS(result);

        CheckAfterReply(&checkInfo, cond.GetReceiveMethod(), cond.GetDataType(), true);
    }
}
#endif // SEND_ASYNC_REPLY_TEST

void TestClientMultiSend(nn::svc::Handle namedPort, TestTag testTag)
{
    TestThreadLeak testThreadHandle;
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetClientSession, testTag, &info);

        nn::svc::Handle readEvent;
        ReceiveMoveHandle(&readEvent, clientSession);
        AutoHandleClose rEventCloser(readEvent);

        // クライアントプロセスを待つ
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::TerminateProcess(clientProcess);
        ASSERT_RESULT_SUCCESS(result);
    }
}

void TestClientMultiReceive(nn::svc::Handle namedPort, TestTag testTag)
{
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetClientSession, testTag, &info);

        nn::svc::Handle readEvent;
        ReceiveMoveHandle(&readEvent, clientSession);
        AutoHandleClose rEventCloser(readEvent);

        // 複数の送信を待つ
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = ReceiveTestData(targetServerSession, cond.GetReceiveMethod(), cond.GetDataType());
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::TerminateProcess(clientProcess);
        ASSERT_RESULT_SUCCESS(result);

        // 受信確認
        CheckInfo checkInfo;
        checkInfo.processHandle = clientProcess;
        CheckClientData(&checkInfo, cond.GetReceiveMethod(), cond.GetDataType(), false);

        // 送信データの作成
        nn::Bit32* pMsgBuffer;
        size_t msgSize;
        GetReceiveMethodBuffer(&pMsgBuffer, &msgSize, cond.GetReceiveMethod());
        MakeServerIpcData(&checkInfo, pMsgBuffer, msgSize, cond.GetDataType());

        result = ReplyRequest(targetServerSession, cond.GetReceiveMethod());
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

        CheckAfterReply(&checkInfo, cond.GetReceiveMethod(), cond.GetDataType(), false);
    }
}

struct MultiSendData
{
    SendMethod method;
    IpcDataType dataType;
    nn::svc::Handle session;
    nn::svc::Handle writeEvent;
    int32_t index;
};

void MultiSendThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    MultiSendData* data = reinterpret_cast<MultiSendData*>(arg);

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    CheckInfo checkInfo;
    GetSendBuffer(&pMsgBuffer, &msgSize, data->method);

    // 送信データ作成
    MakeClientIpcData(&checkInfo, pMsgBuffer, msgSize, data->dataType);

    result = nn::svc::SignalEvent(data->writeEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = SendWithMethod(data->session, data->method);

    if (data->method == SendMethod_UseAsync)
    {
        NN_ASSERT_RESULT_SUCCESS(result);
    }
    else
    {
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
    }

    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(pMsgBuffer));
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT(info.attribute == 0);

    CheckAfterSend(&checkInfo, data->method, data->dataType, false);
}

void TestServerMultiRequest(nn::svc::Handle namedPort, TestTag testTag)
{
    TestThreadLeak testHandle;
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetServerSession, testTag, &info, false);

        nn::svc::Handle writeEvent;
        ReceiveMoveHandle(&writeEvent, clientSession);

        TestHeap heap(DefaultStackSize * 2);

        nn::svc::Handle readEvents[2];
        MultiSendData data[2];
        data[0].session = targetClientSession;
        data[0].method = cond.GetSendMethod();
        data[0].dataType = cond.GetDataType();
        data[0].index = 0;
        data[1].session = targetClientSession;
        data[1].method = SendMethod_UseTls;
        data[1].dataType = IpcDataType_ProcessId;
        data[1].index = 1;

        result = nn::svc::CreateEvent(&data[0].writeEvent, &readEvents[0]);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser0(data[0].writeEvent);
        AutoHandleClose rEventCloser0(readEvents[0]);
        result = nn::svc::CreateEvent(&data[1].writeEvent, &readEvents[1]);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser1(data[1].writeEvent);
        AutoHandleClose rEventCloser1(readEvents[1]);

        uintptr_t pc = reinterpret_cast<uintptr_t>(MultiSendThread);
        uintptr_t sp = heap.GetAddress() + heap.GetSize();
        uintptr_t param;
        int32_t priority;
        int32_t idealCore = 0;

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);

        param = reinterpret_cast<uintptr_t>(&data[0]);
        TestThread thread1(pc, param, sp, priority, idealCore);
        thread1.Start();

        param = reinterpret_cast<uintptr_t>(&data[1]);
        sp -= DefaultStackSize;
        TestThread thread2(pc, param, sp, priority, idealCore);
        thread2.Start();

        // 送信中であることを伝える
        result = nn::svc::SignalEvent(writeEvent);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(writeEvent);
        ASSERT_RESULT_SUCCESS(result);

        // サーバープロセスが終了するのを待つ
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &clientProcess, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        processCloser.Close();

        // スレッドが送信するのを待つ
        for (int32_t i = 0; i < static_cast<int32_t>(sizeof(readEvents) / sizeof(nn::svc::Handle)); i++)
        {
            result = nn::svc::WaitSynchronization(&index, &readEvents[i], 1, -1);
            ASSERT_RESULT_SUCCESS(result);
        }

        // サーバーセッションがクローズされるまで帰ってこない
        nn::svc::Handle handles[2];
        handles[0] = thread1.GetHandle();
        handles[1] = thread2.GetHandle();
        result = nn::svc::WaitSynchronization(&index, handles, 2, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        result = targetServerSessionCloser.Close();
        ASSERT_RESULT_SUCCESS(result);

        thread1.Wait();
        thread2.Wait();
    }
}

void TestServerMultiRequestAndMoveSession(nn::svc::Handle namedPort, TestTag testTag)
{
    TestThreadLeak testHandle;
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetServerSession, testTag, &info);

        nn::svc::Handle writeEvent;
        ReceiveMoveHandle(&writeEvent, clientSession);

        TestHeap heap(DefaultStackSize * 2);

        MultiSendData data[2];
        nn::svc::Handle readEvents[2];
        data[0].session = targetClientSession;
        data[0].method = cond.GetSendMethod();
        data[0].index = 0;
        data[0].dataType = cond.GetDataType();
        data[1].session = targetClientSession;
        data[1].method = SendMethod_UseTls;
        data[1].index = 1;
        data[1].dataType = IpcDataType_ProcessId;

        result = nn::svc::CreateEvent(&data[0].writeEvent, &readEvents[0]);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser0(data[0].writeEvent);
        AutoHandleClose rEventCloser0(readEvents[0]);
        result = nn::svc::CreateEvent(&data[1].writeEvent, &readEvents[1]);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose wEventCloser1(data[1].writeEvent);
        AutoHandleClose rEventCloser1(readEvents[1]);

        uintptr_t pc = reinterpret_cast<uintptr_t>(MultiSendThread);
        uintptr_t sp = heap.GetAddress() + heap.GetSize();
        uintptr_t param;
        int32_t priority;
        int32_t idealCore = 0;

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);

        param = reinterpret_cast<uintptr_t>(&data[0]);
        TestThread thread1(pc, param, sp, priority, idealCore);
        thread1.Start();

        param = reinterpret_cast<uintptr_t>(&data[1]);
        sp -= DefaultStackSize;
        TestThread thread2(pc, param, sp, priority, idealCore);
        thread2.Start();

        // スレッドが送信するのを待つ
        int32_t index;
        for (int32_t i = 0; i < static_cast<int32_t>(sizeof(readEvents) / sizeof(nn::svc::Handle)); i++)
        {
            result = nn::svc::WaitSynchronization(&index, &readEvents[i], 1, -1);
            ASSERT_RESULT_SUCCESS(result);
        }

        // 送信中であることを伝える
        result = nn::svc::SignalEvent(writeEvent);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(writeEvent);
        ASSERT_RESULT_SUCCESS(result);

        // サーバーセッションを受け取る
        ReceiveMoveHandle(&targetServerSession, clientSession);

        // サーバープロセスが終了するのを待つ
        result = nn::svc::WaitSynchronization(&index, &clientProcess, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        processCloser.Close();

        // サーバーセッションがクローズされるまで帰ってこない
        nn::svc::Handle handles[2];
        handles[0] = thread1.GetHandle();
        handles[1] = thread2.GetHandle();
        result = nn::svc::WaitSynchronization(&index, handles, 2, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        result = nn::svc::CloseHandle(targetServerSession);
        ASSERT_RESULT_SUCCESS(result);

        thread1.Wait();
        thread2.Wait();
    }
}

void TestClientSend(nn::svc::Handle namedPort, TestTag testTag)
{
    TestThreadLeak testThreadHandle;
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetClientSession, testTag, &info);

        nn::svc::Handle readEvent;
        ReceiveMoveHandle(&readEvent, clientSession);
        AutoHandleClose rEventCloser(readEvent);

        // 相手がSend中になるのを待つ
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::TerminateProcess(clientProcess);
        ASSERT_RESULT_SUCCESS(result);

        nn::Bit32* pMsgBuffer;
        size_t msgSize;
        GetReceiveMethodBuffer(&pMsgBuffer, &msgSize, cond.GetReceiveMethod());

        result = ReceiveTestData(targetServerSession, cond.GetReceiveMethod(), cond.GetDataType());
        ASSERT_TRUE(result <= nn::svc::ResultInvalidHandle() ||
                    result <= nn::svc::ResultSessionClosed());
    }
}

void TestKillIpcProcess(nn::svc::Handle namedPort, TestTag testTag)
{
    TestThreadLeak testThreadHandle;
    nn::Result result;
    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    TestCondition cond;

    while(cond.Next())
    {
        SCOPED_TRACE(cond.GetScopedString());
        StartBinTerminateProcess();
        nn::svc::Handle targetServerSession;
        nn::svc::Handle targetClientSession;

        MakeTestSession(&targetServerSession, &targetClientSession, cond.GetSessionType());
        AutoHandleClose targetServerSessionCloser(targetServerSession);
        AutoHandleClose targetClientSessionCloser(targetClientSession);

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        TestInfo info;
        info.sendMethod = cond.GetSendMethod();
        info.receiveMethod = cond.GetReceiveMethod();
        info.sessionType = cond.GetSessionType();
        info.dataType = cond.GetDataType();

        PrepareIpcTest(clientSession, targetClientSession, testTag, &info);

        // 相手が終了するのを待つ
        WaitProcess(clientProcess);
    }
}

// TEST 123-15
// SendSyncRequestLight の通信中に TerminateProcess を呼び、プロセスを終了させることが出来る
void TestSendSyncRequestLightPort(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test SendSyncRequestLight and ReplyAndReceive Start (Port) +++\n");
    StartBinTerminateProcess();
    nn::Result result;
    int32_t index;

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle lightClientSession;
    result = nn::svc::ConnectToPort(&lightClientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_SendSyncLightPort, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 0, 1);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = ipcMsg.SetHandle(offset, lightClientSession);

    result = nn::svc::ReplyAndReceive(&index, &clientSession, 0, clientSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    result = ServerReceiveLightRequest(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    NN_LOG("Receive Light Request\n");

    // 相手のプロセスがIPCの返信を待っている

    // クライアントプロセスを終了させる
    result = nn::svc::TerminateProcess(clientProcess);
    ASSERT_RESULT_SUCCESS(result);

    NN_LOG("Finish to Terminate Process\n");

    result = ServerReplyLightRequest(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    NN_LOG("--- Test SendSyncRequestLight and ReplyAndReceiveLight End (Port) ---\n");
}

// TEST 123-16
// SendSyncRequestLight の通信中に TerminateProcess を呼び、プロセスを終了させることが出来る
void TestSendSyncRequestLightSession(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test SendSyncRequestLight and ReplyAndReceive Start (Session) +++\n");
    StartBinTerminateProcess();
    nn::Result result;
    int32_t index;

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::svc::Handle serverSession;
    nn::svc::Handle lightClientSession;
    result = nn::svc::CreateSession(&serverSession, &lightClientSession, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(lightClientSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_SendSyncLightSession, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 0, 1);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = ipcMsg.SetHandle(offset, lightClientSession);

    result = nn::svc::ReplyAndReceive(&index, &clientSession, 0, clientSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = ServerReceiveLightRequest(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // 相手のプロセスがIPCの返信を待っている

    // クライアントプロセスを終了させる
    result = nn::svc::TerminateProcess(clientProcess);
    ASSERT_RESULT_SUCCESS(result);

    result = ServerReplyLightRequest(serverSession);
    ASSERT_RESULT_FAILURE(result);

    NN_LOG("--- Test SendSyncRequestLight and ReplyAndReceiveLight End (Session) ---\n");
}

// TEST 123-10
// スレッドの終了待ちをしているスレッドがいる状態で TerminateProcess を呼び、
// プロセスを終了させることが出来る
void TestThreadHandle(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test Thread Start +++\n");
    TestThreadLeak testThreadHandle;
    StartBinTerminateProcess();
    nn::Result result;


    {
        nn::svc::Handle clientSession;
        nn::svc::Handle clientProcess;

        WaitClient(&clientSession, &clientProcess, namedPort);
        AutoHandleClose sessionCloser(clientSession);
        AutoHandleClose processCloser(clientProcess);

        nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_Thread, 0, 0, 0, 0, 0, 0, 0);
        ipcMsg.Set(ipcHeader);

        int32_t index;
        result = nn::svc::ReplyAndReceive(&index, &clientSession, 1, clientSession, -1);
        ASSERT_RESULT_SUCCESS(result);

        // 相手のプロセスがスレッドの生成を終え、IPCの返信を待っている

        // クライアントプロセスを終了させる
        result = nn::svc::TerminateProcess(clientProcess);
        ASSERT_RESULT_SUCCESS(result);
    }

    NN_LOG("--- Test Thread END ---\n");
}

// TEST 123-11
// 割り込み待ちをしているスレッドがいる状態で TerminateProcess を呼び、
// プロセスを終了させることが出来る
void TestInterrupt(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test InterruptEvent Start +++\n");
    StartBinTerminateProcess();
    nn::Result result;

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_Interrupt, 0, 0, 0, 0, 0, 0, 0);
    ipcMsg.Set(ipcHeader);

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &clientSession, 1, clientSession, -1);
    ASSERT_RESULT_SUCCESS(result);

    // 相手のプロセスが割り込みイベントの生成を終え、IPCの返信を待っている

    // クライアントプロセスを終了させる
    result = nn::svc::TerminateProcess(clientProcess);
    ASSERT_RESULT_SUCCESS(result);

#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
    //
    // TMR1 を利用する
    //

    TestTmrDriver driver;

    // 再度イベントを作成することが出来る
    nn::svc::Handle interruptHandle;
    result = nn::svc::CreateInterruptEvent(
                &interruptHandle, static_cast<nn::svc::Interrupt>(driver.GetInterruptNumber()),
                nn::svc::InterruptType_Level);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose intCloser(interruptHandle);

    driver.SetTimer(1000); // 1000ms
    driver.StartTimer();

    result = nn::svc::WaitSynchronization(&index, &interruptHandle, 1, 1000 * 1000 * 1000);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    result = nn::svc::WaitSynchronization(&index, &interruptHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    driver.ClearInterrupt();

    result = nn::svc::ClearEvent(interruptHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &interruptHandle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
#endif

    NN_LOG("--- Test InterruptEvent END ---\n");
}

// TEST 123-12
// 共有メモリを利用しているプロセスを TerminateProcess で終了させることが出来る
void TestShared(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test Shared Start +++\n");
    StartBinTerminateProcess();
    nn::Result result;
    nn::svc::MemoryInfo info;

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::svc::Handle sharedHandle;
    nn::svc::MemoryPermission permission = nn::svc::MemoryPermission_ReadWrite;
    result = nn::svc::CreateSharedMemory(&sharedHandle, SharedSize, permission, permission);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(sharedHandle);

    result = nn::svc::MapSharedMemory(sharedHandle, g_FreeAreaBegin, SharedSize, permission);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&info, g_FreeAreaBegin);
    ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
    ASSERT_TRUE(info.attribute == 0);
    ASSERT_TRUE(info.permission == permission);

    int32_t* ptr = reinterpret_cast<int32_t*>(g_FreeAreaBegin);
    *ptr = 0;

    // 共有メモリのハンドルをクライアントにコピー
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_Shared, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 1, 0);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = ipcMsg.SetHandle(offset, sharedHandle);

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &clientSession, 0, clientSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    nn::svc::Handle readEvent;
    ReceiveMoveHandle(&readEvent, clientSession);
    AutoHandleClose rEventCloser(readEvent);

    // クライアントが共有メモリにアクセスするのを待つ
    result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // クライアントプロセスを終了させる
    result = nn::svc::TerminateProcess(clientProcess);
    ASSERT_RESULT_SUCCESS(result);

    // 終了処理
    result = nn::svc::UnmapSharedMemory(sharedHandle, g_FreeAreaBegin, SharedSize);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&info, g_FreeAreaBegin);
    ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(info.attribute == 0);
    ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);

    NN_LOG("--- Test Shared END ---\n");
}

// TEST 123-21
// Sleep しているスレッドがいるプロセスをTerminateProcess で終了させることが出来る
void TestSleepThread(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test SleepThread Start +++\n");
    StartBinTerminateProcess();
    nn::Result result;

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_Sleep, 0, 0, 0, 0, 0, 0, 0);
    ipcMsg.Set(ipcHeader);


    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &clientSession, 1, clientSession, -1);
    ASSERT_RESULT_SUCCESS(result);

    // 相手のプロセスがスレッドの生成を終え、IPCの返信を待っている

    // クライアントプロセスを終了させる
    result = nn::svc::TerminateProcess(clientProcess);
    ASSERT_RESULT_SUCCESS(result);

    NN_LOG("--- Test SleepThread END ---\n");
}

void TestTransfer(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test Transfer Start +++\n");
    StartBinTerminateProcess();
    nn::Result result;

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::svc::Handle transfer;
    TestHeap heap(HeapAlign);
    nn::svc::MemoryInfo info;

    result = nn::svc::CreateTransferMemory(
                &transfer, heap.GetAddress(), SharedSize, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose transferCloser(transfer);

    GetMemoryInfo(&info, heap.GetAddress());
    ASSERT_TRUE(info.attribute == nn::svc::MemoryAttribute_Locked);

    // TransferMemory のハンドルをクライアントにムーブ
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_Transfer, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 0, 1);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = ipcMsg.SetHandle(offset, transfer);

    int32_t index;
    result = nn::svc::ReplyAndReceive(&index, &clientSession, 0, clientSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // クライアントプロセスが終了するのを待つ
    WaitProcess(clientProcess);
    processCloser.Close();

    GetMemoryInfo(&info, heap.GetAddress());
    ASSERT_TRUE(info.attribute == 0);

    NN_LOG("--- Test Transfer END ---\n");
}

#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
void TestDeviceAddressSpace(nn::svc::Handle namedPort)
{
    NN_LOG("+++ Test DeviceAddressSpace Start +++\n");
    StartBinTerminateProcess();
    nn::Result result;

    nn::svc::Handle clientSession;
    nn::svc::Handle clientProcess;

    WaitClient(&clientSession, &clientProcess, namedPort);
    AutoHandleClose sessionCloser(clientSession);
    AutoHandleClose processCloser(clientProcess);

    nn::svc::Handle handle;

    uint64_t spaceSize = 0x80000000;
    uint64_t spaceAddr = 0;

    result = nn::svc::CreateDeviceAddressSpace(&handle, spaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    // DeviceAddressSpace のハンドルをクライアントにコピー
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    int32_t index;

    {
        nn::svc::ipc::MessageBuffer::MessageHeader
            ipcHeader(TestTag_DeviceAddress, 1, 0, 0, 0, 0, 0, 0);
        nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 0, 1);
        int offset = ipcMsg.Set(ipcHeader);
        offset = ipcMsg.Set(ipcSpecial);
        offset = ipcMsg.SetHandle(offset, handle);

        result = nn::svc::ReplyAndReceive(&index, &clientSession, 0, clientSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    // クライアントプロセスが終了するのを待つ
    WaitProcess(clientProcess);

    result = processCloser.Close();
    ASSERT_RESULT_SUCCESS(result);

    NN_LOG("--- Test DeviceAddressSpace END ---\n");
}
#endif

} // namespace


/*
   Test Method
 */

// TEST 123-8
// イベントのシグナル待ちをしているスレッドがいる状態で TerminateProcess を呼び、
// プロセスを終了させることが出来る
TEST(TerminateProcess, EventTest)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestEventLeak leakTest;
    TestReadEvent(namedPortHandle);
    TestWriteEvent(namedPortHandle);
}

// TEST 123-34 から 123-44
// クライアントプロセスが要求を送信し、
// サーバープロセスが要求を受信する前にサーバープロセスが終了する
TEST(TerminateProcess, KillServerBeforeReceving)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestServerSend(namedPortHandle, TestTag_Ipc_Server_Send);
}

// TEST 123-45 から 123-55
// クライアントプロセスが要求を送信し、
// サーバープロセスが要求を受信する前にクライアントプロセスが終了する
TEST(TerminateProcess, KillClientBeforeReceiving)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestClientSend(namedPortHandle, TestTag_Ipc_Client_Send);
}

// TEST 123-23 から 123-33
// クライアントプロセスが要求を送信し、
// サーバープロセスが要求を受信し、Replyを返す前にクライアントプロセスが終了する
TEST(TerminateProcess, KillClientBeforeReplying)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestClientReceive(namedPortHandle, TestTag_Ipc_Client_Receive);
}

// TEST 123-78 から 123-88
// クライアントプロセスが要求を送信し、
// サーバープロセスが要求を受信し、Replyを返す前にサーバープロセスが終了する
TEST(TerminateProcess, KillServerBeforeReplying)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestServerSend(namedPortHandle, TestTag_Ipc_Server_Receive);
}

// TEST 123-56 から 123-66
// クライアントプロセスが要求を送信し、
// サーバープロセスが結果を返信後、
// クライアントプロセスが受信する前にサーバープロセスが終了する
TEST(TerminateProcess, KillServerAfterReplying)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestServerReply(namedPortHandle, TestTag_Ipc_Server_Reply);
}

#ifdef SEND_ASYNC_REPLY_TEST
// TEST 123-67 から 123-77
// クライアントプロセスが要求を送信し、
// サーバープロセスが結果を返信後、
// クライアントプロセスが受信する前にクライアントプロセスが終了する
TEST(TerminateProcess, KillClientAfterReplying)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestClientReply(namedPortHandle, TestTag_Ipc_Client_Reply);
}
#endif

// TEST 123-100 から 123-110
// サーバーが要求を受信する前にクライアントプロセスが終了する
// 複数の要求有
TEST(TerminateProcess, KillClientMultiSendNotReceive)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestClientMultiSend(namedPortHandle, TestTag_Ipc_Client_MultiSend);
}

// TEST 123-122 から 123-132
// サーバーが要求を受信した後、結果を返す前にクライアントプロセスが終了する
// 複数の要求有
TEST(TerminateProcess, KillClientMultiSendReceive)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestClientMultiReceive(namedPortHandle, TestTag_Ipc_Client_MultiReceive);
}

// TEST 123-89 から 123-99
// サーバーが要求を受信する前にサーバープロセスが終了する
// 複数の要求有
TEST(TerminateProcess, KillServerMultiSendNotReceive)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestServerMultiRequest(namedPortHandle, TestTag_Ipc_Server_MultiSend);
}

// TEST 123-111 から 123-121
// サーバーが要求を受信した後、結果を返す前にサーバープロセスが終了する
// 複数の要求有
TEST(TerminateProcess, KillServerMultiSendReceive)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestServerMultiRequest(namedPortHandle, TestTag_Ipc_Server_MultiReceive);
}

// TEST 123-132 から 123-142
// サーバープロセスが終了する際にサーバーセッションを他のプロセスに渡しておくと、参照カウントが0にならない。
// サーバープロセスが要求を受信する前
TEST(TerminateProcess, KillServerMultiSendNotReceiveWithPassingSession)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestServerMultiRequestAndMoveSession(namedPortHandle, TestTag_Ipc_Server_MultiSendMoveSession); // 18 case
}

// TEST 123-143 から 123-153
// サーバープロセスが終了する際にサーバーセッションを他のプロセスに渡しておくと、参照カウントが0にならない。
// サーバープロセスが要求を受信した後
TEST(TerminateProcess, KillServerMultiSendReceiveWithPassingSession)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestServerMultiRequestAndMoveSession(namedPortHandle, TestTag_Ipc_Server_MultiReceiveMoveSession); // 18 case
}

// TEST 123-15, 123-16
// SendSyncRequestLight の通信中に TerminateProcess を呼び、プロセスを終了させることが出来る
TEST(TerminateProcess, LightSession)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestLightSessionLeak leakTest;
    TestSendSyncRequestLightPort(namedPortHandle);
    TestSendSyncRequestLightSession(namedPortHandle);
}

TEST(TerminateProcess, Thread)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    // TEST 123-10
    // スレッドの終了待ちをしているスレッドがいる状態で TerminateProcess を呼び、
    // プロセスを終了させることが出来る
    TestThreadHandle(namedPortHandle);

    // TEST 123-21
    // Sleep しているスレッドがいるプロセスをTerminateProcess で終了させることが出来る
    TestSleepThread(namedPortHandle);
}

// TEST 123-11
// 割り込み待ちをしているスレッドがいる状態で TerminateProcess を呼び、
// プロセスを終了させることが出来る
TEST(TerminateProcess, Interrupt)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestInterruptEventLeak leakTest;
    TestInterrupt(namedPortHandle);
}

// TEST 123-12
// 共有メモリを利用しているプロセスを TerminateProcess で終了させることが出来る
TEST(TerminateProcess, SharedMemory)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestSharedMemoryLeak leakTest;
    TestShared(namedPortHandle);
}

// TEST 123-155
// TransferMemory を利用しているプロセスを TerminateProcess で終了させることが出来る
TEST(TerminateProcess, TransferMemory)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestTransferMemoryLeak leakTest;
    TestTransfer(namedPortHandle);
}

#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
// TEST 123-156
// DeviceAddressSpace を利用しているプロセスを TerminateProcess で終了させることが出来る
TEST(TerminateProcess, sMMU)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestDeviceAddressSpaceLeak leakTest;
    TestDeviceAddressSpace(namedPortHandle);
}
#endif

TEST(TerminateProcess, TestKillIpcProcessWhileSending)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestKillIpcProcess(namedPortHandle, TestTag_Ipc_Process_Sending);
}

TEST(TerminateProcess, TestKillIpcProcessWhileReceiving)
{
    nn::Result result;

    NamedPortManager namedPort(PortName, 1);
    nn::svc::Handle namedPortHandle = namedPort.GetHandle();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinTerminate_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinTerminate_end);
    TestLoader loader(BinTerminate_begin, end - begin);
    g_TestLoader = &loader;

    TestThreadLeak testHandle;
    TestKillIpcProcess(namedPortHandle, TestTag_Ipc_Process_Receiving);
}
