﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestIpc.h"
#include "test_TerminateProcess.h"
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <cstring>
#include <nn/init.h>

#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
#include "test_TestTmrDevice_Jetson.h"
#endif // defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1)

extern nn::Result ClientSendLightRequest(nn::svc::Handle handle);

extern "C" void nndiagStartup()
{
}

extern "C" void nninitStartup()
{
}

namespace {

const int64_t WaitTime = 1000 * 1000 * 1000; // 1 sec
const int64_t SleepTime = 100 * 1000 * 1000; // 100 msec
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));
char g_IpcBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_IpcMapBuffer[0x3000] __attribute__((aligned(0x1000)));
char g_IpcRecvBuffer[2][0x1000] __attribute__((aligned(0x1000)));
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));

void MakeNamedPortSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    const char* portName = "NamedPort";
    result = nn::svc::ManageNamedPort(&serverPort, portName, 1);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);

    result = nn::svc::ConnectToNamedPort(pOutClientSession, portName);
    NN_ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(pOutServerSession, serverPort);
    NN_ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ManageNamedPort(&serverPort, portName, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void MakePortSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    result = nn::svc::ConnectToPort(pOutClientSession, clientPort);
    NN_ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(pOutServerSession, serverPort);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void MakeSession(nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession)
{
    nn::Result result;
    result = nn::svc::CreateSession(pOutServerSession, pOutClientSession, false, 0);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void MakeTestSession(
        nn::svc::Handle *pOutServerSession, nn::svc::Handle *pOutClientSession,
        SessionType sessionType)
{
    switch(sessionType)
    {
        case SessionType_NamedPort:
            MakeNamedPortSession(pOutServerSession, pOutClientSession);
            break;
        case SessionType_Port:
            MakePortSession(pOutServerSession, pOutClientSession);
            break;
        case SessionType_Session:
            MakeSession(pOutServerSession, pOutClientSession);
            break;
        default: NN_UNEXPECTED_DEFAULT;
    }
}


void MakeSendIpcAllData(nn::Bit32* pMsgBuffer)
{
    nn::svc::Handle copyHandle;
    nn::svc::Handle moveHandle;

    char* recvBuffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };
    size_t recvBufferSizes[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };

    MakeAllData(
            &copyHandle, &moveHandle,
            pMsgBuffer,
            SendPointData, SendPointDataSize, SendPointDataNum,
            g_IpcMapBuffer, MapDefaultSize, SendMapSendData,
            g_IpcMapBuffer + MapDefaultSize, MapDefaultSize, SendMapRecvData,
            g_IpcMapBuffer + MapDefaultSize * 2, MapDefaultSize, SendMapExchData,
            SendRawData, SendRawDataSize,
            recvBuffers, recvBufferSizes, 2
            );
}

void MakeReplyIpcAllData(nn::Bit32* pMsgBuffer, uintptr_t sendMapAddr, uintptr_t recvMapAddr, uintptr_t exchMapAddr)
{
    nn::svc::Handle copyHandle;
    nn::svc::Handle moveHandle;

    char* recvBuffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };
    size_t recvBufferSizes[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };

    MakeAllData(
            &copyHandle, &moveHandle,
            pMsgBuffer,
            SendPointData, SendPointDataSize, SendPointDataNum,
            reinterpret_cast<char*>(sendMapAddr), MapDefaultSize,
            reinterpret_cast<char*>(recvMapAddr), MapDefaultSize, SendMapRecvData,
            reinterpret_cast<char*>(exchMapAddr), MapDefaultSize, SendMapExchData,
            SendRawData, SendRawDataSize,
            recvBuffers, recvBufferSizes, 2
            );
}

void MakeSendProcessIdData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyProcessIdData(pMsgBuffer);
}

void MakeSendCopyHandleData(nn::Bit32* pMsgBuffer)
{
    nn::svc::Handle threadHandle;
    MakeOnlyCopyHandleData(&threadHandle, pMsgBuffer);
    // ハンドルがプロセス終了時に閉じることを確認するために、
    // この時点では何もしない
}

void MakeSendMoveHandleData(nn::Bit32* pMsgBuffer)
{
    nn::svc::Handle threadHandle;
    MakeOnlyMoveHandleData(&threadHandle, pMsgBuffer);
}

void MakeSendPointerDataWithIpcBuffer(nn::Bit32* pMsgBuffer)
{
    MakeOnlyPointerWithIpcBufferData(
            pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum);
}

void MakeSendPointerDataWithOneUserBufferData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyPointerWithOneUserBuffer(
            pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
            g_IpcRecvBuffer[0], sizeof(g_IpcRecvBuffer[0]));
}

void MakeSendPointerDataWithMultiUserBufferData(nn::Bit32* pMsgBuffer)
{
    size_t bufferSize[2] = { sizeof(g_IpcRecvBuffer[0]), sizeof(g_IpcRecvBuffer[1]) };
    char* buffers[2] = { g_IpcRecvBuffer[0], g_IpcRecvBuffer[1] };
    MakeOnlyPointerWithMultiUserBuffer(
            pMsgBuffer, SendPointData, SendPointDataSize, SendPointDataNum,
            buffers, bufferSize);
}

void MakeSendMapSendData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapSendData(pMsgBuffer, g_IpcMapBuffer, MapSendSize, SendMapSendData);
}

void MakeSendMapRecvData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapReceiveData(pMsgBuffer, g_IpcMapBuffer, MapRecvSize, SendMapRecvData);
}

void MakeReplyMapRecvData(nn::Bit32* pMsgBuffer, uintptr_t addr)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    WriteMapData(reinterpret_cast<char*>(addr), MapRecvSize, SendMapRecvData);
}

void MakeSendMapExchData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyMapExchangeData(pMsgBuffer, g_IpcMapBuffer, MapExchSize, SendMapExchData);
}

void MakeReplyMapExchData(nn::Bit32* pMsgBuffer, uintptr_t addr)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();
    WriteMapData(reinterpret_cast<char*>(addr), MapExchSize, SendMapExchData);
}

void MakeSendRawData(nn::Bit32* pMsgBuffer)
{
    MakeOnlyRawData(pMsgBuffer, SendRawData, SendRawDataSize);
}

void TestSleepThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    for(;;)
    {
        nn::svc::SleepThread(WaitTime);
    }
}

void GetTag(TestTag* tag)
{
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);

    *tag = static_cast<TestTag>(ipcHeader.GetTag());
}

bool InitTest(nn::svc::Handle* clientSession, TestTag* testTag)
{
    nn::Result result;

    int32_t i;
    for(i = 0; i < 10; i++)
    {
        result = nn::svc::ConnectToNamedPort(clientSession, PortName);
        if (result.IsSuccess())
        {
            break;
        }
        nn::svc::SleepThread(WaitTime);
    }
    if (i == 10)
    {
        return false;
    }

    // 自プロセスのハンドルを送信する
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(TestTag_Init, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(false, 1, 0);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(ipcSpecial);
    offset = ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    result = nn::svc::SendSyncRequest(*clientSession);
    NN_ASSERT_RESULT_SUCCESS(result);

    GetTag(testTag);
    return true;
}

void GetSendMsgBuffer(nn::Bit32** pMsgBuffer, size_t* pMsgSize, SendMethod method)
{
    if (method == SendMethod_UseTls)
    {
        *pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        *pMsgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    else
    {
        *pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_IpcBuffer);
        *pMsgSize = sizeof(g_IpcBuffer);
    }
}

void GetReceiveMsgBuffer(nn::Bit32** pMsgBuffer, size_t* pMsgSize, ReceiveMethod method)
{
    if (method == ReceiveMethod_UseTls)
    {
        *pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        *pMsgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    else
    {
        *pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_IpcBuffer);
        *pMsgSize = sizeof(g_IpcBuffer);
    }

}

void SendWithMethod(
        nn::svc::Handle clientSession, SendMethod method, uintptr_t ipcBuffer, size_t msgSize)
{
    nn::Result result;
    nn::svc::Handle eventHandle;

    switch (method)
    {
    case SendMethod_UseTls:
        result = nn::svc::SendSyncRequest(clientSession);
        break;
    case SendMethod_UseBuffer:
        result = nn::svc::SendSyncRequestWithUserBuffer(ipcBuffer, msgSize, clientSession);
        break;
    case SendMethod_UseAsync:
        result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandle, ipcBuffer, msgSize, clientSession);
        NN_ASSERT_RESULT_SUCCESS(result);
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    // ここに来ること自体がおかしい
    DebugResult(result);
    NN_ASSERT(false);
}

nn::Result ReceiveRequest(nn::svc::Handle clientSession, ReceiveMethod method)
{
    nn::Result result;
    int32_t index;

    if (method == ReceiveMethod_UseTls)
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &clientSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    else
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, reinterpret_cast<uintptr_t>(g_IpcBuffer), sizeof(g_IpcBuffer),
                    &clientSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    return result;
}

nn::Result ReplyRequest(nn::svc::Handle clientSession, bool isBuffer)
{
    nn::Result result;
    int32_t index;

    if (isBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, reinterpret_cast<uintptr_t>(g_IpcBuffer), sizeof(g_IpcBuffer),
                    &clientSession, 0, clientSession, 0);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(&index, &clientSession, 0, clientSession, 0);
    }
    return result;
}

void ReceiveIpcInfo(
        TestInfo* info, nn::svc::Handle* targetSession,
        TestTag testTag, nn::svc::Handle clientSession)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetSendMsgBuffer(&pMsgBuffer, &msgSize, SendMethod_UseTls);

    // 送信
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader
            ipcHeader(testTag, 0, 0, 0, 0, 0, 0, 0);
        ipcMsg.Set(ipcHeader);

        result = nn::svc::SendSyncRequest(clientSession);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    // 受信
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

        size_t bit32Size = sizeof(nn::Bit32);
        int rawNum = ((sizeof(TestInfo) + bit32Size - 1) & ~(bit32Size - 1)) / bit32Size;
        NN_ASSERT(ipcHeader.GetRawNum() == rawNum);
        NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
        NN_ASSERT((ipcSpecial.GetMoveHandleNum() == 1 && ipcSpecial.GetCopyHandleNum() == 0)
                || (ipcSpecial.GetMoveHandleNum() == 0 && ipcSpecial.GetCopyHandleNum() == 1));

        int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
        *targetSession = ipcMsg.GetHandle(offset);
        offset = ipcMsg.GetRawDataOffset(ipcHeader, ipcSpecial);
        std::memcpy(info, &pMsgBuffer[offset], sizeof(TestInfo));
    }
}

void TestReadEvent(nn::svc::Handle clientSession)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
    nn::svc::Handle readableEvent = ipcMsg.GetHandle(offset);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);

    // ここにはこないはず
    NN_ASSERT(false);
}

void TestWriteEvent(nn::svc::Handle clientSession)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
    nn::svc::Handle writableEvent = ipcMsg.GetHandle(offset);

    result = nn::svc::SignalEvent(writableEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    ipcMsg.SetNull();
    result = nn::svc::SendSyncRequest(clientSession);

    // ここにはこないはず
    NN_ASSERT(false);
}

void SwitchIpcTestData(nn::Bit32* pMsgBuffer, IpcDataType dataType, bool isServer)
{
    uintptr_t sendMapAddr;
    uintptr_t recvMapAddr;
    uintptr_t exchMapAddr;
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
    int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
    int mapDelta =  nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32);

    switch(dataType)
    {
    case IpcDataType_All:
        if (!isServer)
        {
            MakeSendIpcAllData(pMsgBuffer);
        }
        else
        {
            GetMapAddr(&sendMapAddr, &ipcMsg, offset);
            GetMapAddr(&recvMapAddr, &ipcMsg, offset + mapDelta);
            GetMapAddr(&exchMapAddr, &ipcMsg, offset + mapDelta * 2);
            MakeReplyIpcAllData(pMsgBuffer, sendMapAddr, recvMapAddr, exchMapAddr);
        }
        break;
    case IpcDataType_ProcessId:
        MakeSendProcessIdData(pMsgBuffer);
        break;
    case IpcDataType_CopyHandle:
        MakeSendCopyHandleData(pMsgBuffer);
        break;
    case IpcDataType_MoveHandle:
        MakeSendMoveHandleData(pMsgBuffer);
        break;
    case IpcDataType_Pointer_IpcBuffer:
        MakeSendPointerDataWithIpcBuffer(pMsgBuffer);
        break;
    case IpcDataType_Pointer_OneBuffer:
        MakeSendPointerDataWithOneUserBufferData(pMsgBuffer);
        break;
    case IpcDataType_Pointer_MultiBuffer:
        MakeSendPointerDataWithMultiUserBufferData(pMsgBuffer);
        break;
    case IpcDataType_Map_Send:
        if (!isServer)
        {
            MakeSendMapSendData(pMsgBuffer);
        }
        break;
    case IpcDataType_Map_Receive:
        if (isServer)
        {
            GetMapAddr(&recvMapAddr, &ipcMsg, offset);
            MakeReplyMapRecvData(pMsgBuffer, recvMapAddr);
        }
        else
        {
            MakeSendMapRecvData(pMsgBuffer);
        }
        break;
    case IpcDataType_Map_Exchange:
        if (isServer)
        {
            GetMapAddr(&exchMapAddr, &ipcMsg, offset);
            MakeReplyMapExchData(pMsgBuffer, exchMapAddr);
        }
        else
        {
            MakeSendMapExchData(pMsgBuffer);
        }
        break;
    case IpcDataType_Raw:
        MakeSendRawData(pMsgBuffer);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT(readability/fn_size)

struct SendData
{
    nn::svc::Handle session;
    nn::svc::Handle writeEvent;
    IpcDataType dataType;
    SendMethod method;
};

void SendThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    SendData *data = reinterpret_cast<SendData*>(arg);

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetSendMsgBuffer(&pMsgBuffer, &msgSize, data->method);

    SwitchIpcTestData(pMsgBuffer, data->dataType, false);

    nn::Result result;
    result = nn::svc::SignalEvent(data->writeEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    SendWithMethod(data->session, data->method, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize);
}

void ReceiveData(nn::svc::Handle targetSession, ReceiveMethod method)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetReceiveMsgBuffer(&pMsgBuffer, &msgSize, method);

    int offset;
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 4);
    offset = ipcMsg.Set(ipcHeader);
    for (int i = 0; i < 2; i ++)
    {
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvEntry(&g_IpcRecvBuffer[i], sizeof(g_IpcRecvBuffer[i]));
        offset = ipcMsg.Set(offset, recvEntry);
    }

    result = ReceiveRequest(targetSession, method);
    NN_ASSERT_RESULT_SUCCESS(result);
}

void TestClientSend(nn::svc::Handle clientSession, TestTag testTag)
{
    nn::Result result;

    nn::svc::Handle targetSession;
    TestInfo testInfo;
    ReceiveIpcInfo(&testInfo, &targetSession, testTag, clientSession);

    nn::svc::Handle writeEvent;
    nn::svc::Handle readEvent;
    result = nn::svc::CreateEvent(&writeEvent, &readEvent);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writeEvent);
    SendMoveHandle(readEvent, clientSession);

    int32_t curPriority;
    result = nn::svc::GetThreadPriority(&curPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle writeEvent1;
    nn::svc::Handle readEvent1;
    result = nn::svc::CreateEvent(&writeEvent1, &readEvent1);
    NN_ASSERT_RESULT_SUCCESS(result);

    SendData data;
    data.session = targetSession;
    data.method = testInfo.sendMethod;
    data.writeEvent = writeEvent1;
    data.dataType = testInfo.dataType;

    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&data);
    int32_t priority = curPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    // スレッドが送信済みになるのを待つ
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &readEvent1, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    // サーバーに準備が出来たことを伝える
    result = nn::svc::SignalEvent(writeEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    thread.Wait();
}

void TestServerSend(nn::svc::Handle clientSession, TestTag tag)
{
    nn::Result result;

    TestInfo testInfo;
    nn::svc::Handle targetSession;

    ReceiveIpcInfo(&testInfo, &targetSession, tag, clientSession);

    nn::svc::Handle writeEvent;
    nn::svc::Handle readEvent;
    result = nn::svc::CreateEvent(&writeEvent, &readEvent);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose rEventCloser(readEvent);
    SendMoveHandle(writeEvent, clientSession);

    // 相手がデータを送信してくるのを待つ
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::TerminateProcess(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    // ここには来ない
    NN_ASSERT(false);
}

void TestServerSendAndPassSession(nn::svc::Handle clientSession, TestTag tag)
{
    nn::Result result;

    nn::svc::Handle targetSession;
    TestInfo testInfo;

    ReceiveIpcInfo(&testInfo, &targetSession, tag, clientSession);

    nn::svc::Handle writeEvent;
    nn::svc::Handle readEvent;
    result = nn::svc::CreateEvent(&writeEvent, &readEvent);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose rEventCloser(readEvent);
    SendMoveHandle(writeEvent, clientSession);

    // 相手がデータを送信してくるのを待つ
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    // targetSession を終了する前に渡す
    SendMoveHandle(targetSession, clientSession);

    result = nn::svc::TerminateProcess(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    // ここには来ない
    NN_ASSERT(false);
}

void TestServerReceive(nn::svc::Handle clientSession, TestTag testTag)
{
    nn::Result result;

    TestInfo testInfo;
    nn::svc::Handle targetSession;
    ReceiveIpcInfo(&testInfo, &targetSession, testTag, clientSession);

    nn::svc::Handle writeEvent;
    nn::svc::Handle readEvent;
    result = nn::svc::CreateEvent(&writeEvent, &readEvent);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose rEventCloser(readEvent);
    SendMoveHandle(writeEvent, clientSession);

    // 相手がすべての要求を出し尽くしたかを待つ
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    ReceiveData(targetSession, testInfo.receiveMethod);

    result = nn::svc::TerminateProcess(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    // ここには来ない
    NN_ASSERT(false);
}

void TestServerReceiveAndPassSesion(nn::svc::Handle clientSession, TestTag testTag)
{
    nn::Result result;

    TestInfo testInfo;
    nn::svc::Handle targetSession;
    ReceiveIpcInfo(&testInfo, &targetSession, testTag, clientSession);

    nn::svc::Handle writeEvent;
    nn::svc::Handle readEvent;
    result = nn::svc::CreateEvent(&writeEvent, &readEvent);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose rEventCloser(readEvent);
    SendMoveHandle(writeEvent, clientSession);

    // 相手がすべての要求を出し尽くしたかを待つ
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
    NN_ASSERT_RESULT_SUCCESS(result);

    ReceiveData(targetSession, testInfo.receiveMethod);

    // targetSession を終了する前に渡す
    SendMoveHandle(targetSession, clientSession);

    result = nn::svc::TerminateProcess(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    // ここには来ない
    NN_ASSERT(false);
}

void TestServerReply(nn::svc::Handle clientSession, TestTag testTag)
{
    nn::Result result;

    TestInfo testInfo;
    nn::svc::Handle targetSession;
    ReceiveIpcInfo(&testInfo, &targetSession, testTag, clientSession);

    ReceiveData(targetSession, testInfo.receiveMethod);

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetReceiveMsgBuffer(&pMsgBuffer, &msgSize, testInfo.receiveMethod);
    SwitchIpcTestData(pMsgBuffer, testInfo.dataType, true);

    result = ReplyRequest(targetSession, testInfo.receiveMethod);
    NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::TerminateProcess(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    // ここには来ない
    NN_ASSERT(false);
}

struct MultiSendClientData
{
    nn::svc::Handle clientSession;
    nn::svc::Handle writeEvent;
    uintptr_t ipcBuffer;
    size_t bufSize;
    IpcDataType dataType;
    SendMethod method;
};

void TestClientMultiSendThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    MultiSendClientData* data = reinterpret_cast<MultiSendClientData*>(arg);
    nn::svc::Handle eventHandle;
    if (data->method == SendMethod_UseTls)
    {
        data->ipcBuffer = reinterpret_cast<uintptr_t>(nn::svc::ipc::GetMessageBuffer());
        data->bufSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    SwitchIpcTestData(reinterpret_cast<nn::Bit32*>(data->ipcBuffer), data->dataType, false);

    result = nn::svc::SignalEvent(data->writeEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    SendWithMethod(data->clientSession, data->method, data->ipcBuffer, data->bufSize);

    // ここには来ない
    NN_ASSERT(false);
}

void TestClientMultiSend(nn::svc::Handle clientSession, TestTag testTag)
{
    nn::Result result;

    const int32_t MethodNum = sizeof(SendMethodArray) / sizeof(SendMethod);
    size_t bufSize = 0x1000;
    TestHeap heap((bufSize + DefaultStackSize) * MethodNum);
    MultiSendClientData data[MethodNum];
    nn::svc::Handle threads[MethodNum];

    TestInfo testInfo;
    nn::svc::Handle targetSession;
    ReceiveIpcInfo(&testInfo, &targetSession, testTag, clientSession);

    nn::svc::Handle writeEvent;
    nn::svc::Handle readEvent;
    result = nn::svc::CreateEvent(&writeEvent, &readEvent);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writeEvent);
    SendMoveHandle(readEvent, clientSession);

    uintptr_t spBase = heap.GetAddress() + heap.GetSize();
    uintptr_t ipcBuffer = heap.GetAddress();
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestClientMultiSendThread);
    uintptr_t sp = spBase;
    uintptr_t param = 0;
    int32_t priority;
    int32_t idealCore = 0;
    int32_t index;

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    NN_ASSERT_RESULT_SUCCESS(result);

    for (int32_t i = 0; i < MethodNum; i++)
    {
        nn::svc::Handle readEvent;
        result = nn::svc::CreateEvent(&data[i].writeEvent, &readEvent);
        NN_ASSERT_RESULT_SUCCESS(result);

        data[i].dataType = testInfo.dataType;
        data[i].method = SendMethodArray[(i + testInfo.sendMethod) % MethodNum];
        data[i].ipcBuffer = ipcBuffer + bufSize * i;
        data[i].bufSize = bufSize;
        data[i].clientSession = targetSession;

        sp = spBase - DefaultStackSize * i;
        param = reinterpret_cast<uintptr_t>(&data[i]);
        idealCore = i % NumCore;
        result = nn::svc::CreateThread(&threads[i], pc, param, sp, priority, idealCore);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(threads[i]);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(data[i].writeEvent);
        NN_ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readEvent);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    // 送信済みであることを伝える
    result = nn::svc::SignalEvent(writeEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    for (int32_t i = 0; i < MethodNum; i++)
    {
        result = nn::svc::WaitSynchronization(&index, &threads[i], 1, -1);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    // ここにはこないはず
    NN_ASSERT(false);
}

void TestKillIpcProcessWhileSending(nn::svc::Handle clientSession, TestTag testTag)
{
    nn::Result result;

    const int32_t MethodNum = sizeof(SendMethodArray) / sizeof(SendMethod);
    size_t bufSize = 0x1000;
    TestHeap heap((bufSize + DefaultStackSize) * MethodNum);
    MultiSendClientData data[MethodNum];
    nn::svc::Handle threads[MethodNum];

    TestInfo testInfo;
    nn::svc::Handle targetSession;
    ReceiveIpcInfo(&testInfo, &targetSession, testTag, clientSession);

    // 使わないので閉じる
    nn::svc::CloseHandle(targetSession);

    uintptr_t spBase = heap.GetAddress() + heap.GetSize();
    uintptr_t ipcBuffer = heap.GetAddress();
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestClientMultiSendThread);
    uintptr_t sp = spBase;
    uintptr_t param = 0;
    int32_t priority;
    int32_t idealCore = 0;
    int32_t index;

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;

    MakeTestSession(&serverSession, &targetSession, testInfo.sessionType);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(targetSession);

    for (int32_t i = 0; i < MethodNum; i++)
    {
        nn::svc::Handle readEvent;
        result = nn::svc::CreateEvent(&data[i].writeEvent, &readEvent);
        NN_ASSERT_RESULT_SUCCESS(result);

        data[i].dataType = testInfo.dataType;
        data[i].method = SendMethodArray[(i + testInfo.sendMethod) % MethodNum];
        data[i].ipcBuffer = ipcBuffer + bufSize * i;
        data[i].bufSize = bufSize;
        data[i].clientSession = targetSession;

        sp = spBase - DefaultStackSize * i;
        param = reinterpret_cast<uintptr_t>(&data[i]);
        idealCore = i % NumCore;
        result = nn::svc::CreateThread(&threads[i], pc, param, sp, priority, idealCore);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(threads[i]);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(data[i].writeEvent);
        NN_ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readEvent);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    // 受信せずに終了する
    nn::svc::ExitProcess();

    // ここにはこないはず
    NN_ASSERT(false);
}

void TestKillIpcProcessWhileReceiving(nn::svc::Handle clientSession, TestTag testTag)
{
    nn::Result result;

    const int32_t MethodNum = sizeof(SendMethodArray) / sizeof(SendMethod);
    size_t bufSize = 0x1000;
    TestHeap heap((bufSize + DefaultStackSize) * MethodNum);
    MultiSendClientData data[MethodNum];
    nn::svc::Handle threads[MethodNum];

    TestInfo testInfo;
    nn::svc::Handle targetSession;
    ReceiveIpcInfo(&testInfo, &targetSession, testTag, clientSession);

    // 使わないので閉じる
    nn::svc::CloseHandle(targetSession);

    uintptr_t spBase = heap.GetAddress() + heap.GetSize();
    uintptr_t ipcBuffer = heap.GetAddress();
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestClientMultiSendThread);
    uintptr_t sp = spBase;
    uintptr_t param = 0;
    int32_t priority;
    int32_t idealCore = 0;
    int32_t index;

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;

    MakeTestSession(&serverSession, &targetSession, testInfo.sessionType);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(targetSession);

    for (int32_t i = 0; i < MethodNum; i++)
    {
        nn::svc::Handle readEvent;
        result = nn::svc::CreateEvent(&data[i].writeEvent, &readEvent);
        NN_ASSERT_RESULT_SUCCESS(result);

        data[i].dataType = testInfo.dataType;
        data[i].method = SendMethodArray[(i + testInfo.sendMethod) % MethodNum];
        data[i].ipcBuffer = ipcBuffer + bufSize * i;
        data[i].bufSize = bufSize;
        data[i].clientSession = targetSession;

        sp = spBase - DefaultStackSize * i;
        param = reinterpret_cast<uintptr_t>(&data[i]);
        idealCore = i % NumCore;
        result = nn::svc::CreateThread(&threads[i], pc, param, sp, priority, idealCore);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(threads[i]);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readEvent, 1, -1);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(data[i].writeEvent);
        NN_ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readEvent);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    ReceiveData(serverSession, testInfo.receiveMethod);

    // 返信せずに終了する
    nn::svc::ExitProcess();

    // ここにはこないはず
    NN_ASSERT(false);
}

void TestSendSyncRequestLight(nn::svc::Handle clientSession)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
    nn::svc::Handle testSession = ipcMsg.GetHandle(offset);

    result = ClientSendLightRequest(testSession);

    // ここにはこないはず
    NN_ASSERT(false);
}

void LoopThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    for(;;)
    {
        nn::svc::SleepThread(WaitTime);
    }
}

void TestThreadFunc(nn::svc::Handle clientSession)
{
    nn::Result result;

    int32_t numThread = 10;
    TestHeap heap((sizeof(nn::svc::Handle) + DefaultStackSize) * numThread);;

    nn::svc::Handle* threads = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
    uintptr_t pc = reinterpret_cast<uintptr_t>(LoopThread);
    uintptr_t sp = heap.GetAddress() + heap.GetSize();
    uintptr_t param = 0;
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    for (int32_t i = 0; i < numThread; i++)
    {
        idealCore = i % NumCore;
        result = nn::svc::CreateThread(
                    &threads[i], pc, param, sp - i * DefaultStackSize, priority, idealCore);
        NN_ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(threads[i]);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    ipcMsg.SetNull();
    result = nn::svc::SendSyncRequest(clientSession);

    // ここにはこないはず
    NN_ASSERT(false);
}

void TestInterrupt(nn::svc::Handle clientSession)
{
    nn::Result result;
    nn::svc::Handle interruptHandle;

    // 割り込みイベントを作成する
#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1) \
    || defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) \
    || defined (NN_BUILD_CONFIG_HARDWARE_NX)
    //
    // TMR2 を利用する
    //

    {
        int32_t index;
        TestTmrDriver driver(2);
        NN_ASSERT(!driver.IsUsedTimer());

        result = nn::svc::CreateInterruptEvent(
                &interruptHandle, static_cast<nn::svc::Interrupt>(driver.GetInterruptNumber()),
                nn::svc::InterruptType_Level);
        NN_ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose intCloser(interruptHandle);

        result = nn::svc::WaitSynchronization(&index, &interruptHandle, 1, 0);
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        driver.SetTimer(100); // 1000ms
        driver.StartTimer();

        result = nn::svc::WaitSynchronization(&index, &interruptHandle, 1, 1000UL * 1000 * 1000);
        NN_ASSERT_RESULT_SUCCESS(result);
        NN_ASSERT(index == 0);

        result = nn::svc::WaitSynchronization(&index, &interruptHandle, 1, 0);
        NN_ASSERT_RESULT_SUCCESS(result);

        driver.ClearInterrupt();

        result = nn::svc::ClearEvent(interruptHandle);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &interruptHandle, 1, 0);
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

#endif

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    ipcMsg.SetNull();
    result = nn::svc::SendSyncRequest(clientSession);

    // ここにはこないはず
    NN_ASSERT(false);
}

void TestShared(nn::svc::Handle clientSession)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(ipcSpecial.GetCopyHandleNum() == 1);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
    nn::svc::Handle sharedHandle = ipcMsg.GetHandle(offset);

    result = nn::svc::MapSharedMemory(
                sharedHandle, g_FreeAreaBegin, SharedSize, nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle writeEvent;
    nn::svc::Handle readEvent;
    result = nn::svc::CreateEvent(&writeEvent, &readEvent);
    NN_ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writeEvent);
    SendMoveHandle(readEvent, clientSession);

    result = nn::svc::SignalEvent(writeEvent);
    NN_ASSERT_RESULT_SUCCESS(result);

    for(int32_t i = 0;;i++)
    {
        int32_t* ptr = reinterpret_cast<int32_t*>(g_FreeAreaBegin);
        *ptr = i;
        nn::svc::SleepThread(SleepTime);
    }

    // ここにはこないはず
    NN_ASSERT(false);
}

void TestSleep(nn::svc::Handle clientSession)
{
    nn::Result result;

    int32_t numThread = 10;
    TestHeap heap((sizeof(nn::svc::Handle) + DefaultStackSize) * numThread);;

    nn::svc::Handle* threads = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestSleepThread);
    uintptr_t sp = heap.GetAddress() + heap.GetSize();
    uintptr_t param = 0;
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    for (int32_t i = 0; i < numThread; i++)
    {
        idealCore = i % NumCore;
        result = nn::svc::CreateThread(
                    &threads[i], pc, param, sp - i * DefaultStackSize, priority, idealCore);
        NN_ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(threads[i]);
        NN_ASSERT_RESULT_SUCCESS(result);
    }

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    ipcMsg.SetNull();
    result = nn::svc::SendSyncRequest(clientSession);

    // ここにはこないはず
    NN_ASSERT(false);
}

void TestTransfer(nn::svc::Handle clientSession)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
    nn::svc::Handle transferHandle = ipcMsg.GetHandle(offset);

    result = nn::svc::MapTransferMemory(
                transferHandle, g_FreeAreaBegin, SharedSize, nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapTransferMemory(transferHandle, g_FreeAreaBegin, SharedSize);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::TerminateProcess(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
}

void TestDeviceAddressSpace(nn::svc::Handle clientSession)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

    NN_ASSERT(ipcHeader.GetSpecialNum() == 1);
    NN_ASSERT(ipcSpecial.GetMoveHandleNum() == 1);

    int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
    nn::svc::Handle handle = ipcMsg.GetHandle(offset);

    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    uint64_t deviceAddress = 0;
    size_t size = 0x1000;
    size_t mappedSize = 0;

    result = nn::svc::MapDeviceAddressSpace(
            &mappedSize, handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT(mappedSize > 0 && mappedSize <= size);
    GetMemoryInfo(&blockInfo, addr);
    NN_ASSERT(blockInfo.state == nn::svc::MemoryState_CodeData);
    NN_ASSERT(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    // マップした状態で自殺する
    nn::svc::TerminateProcess(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
}

} // namespace

uintptr_t g_HeapAreaBegin;
uintptr_t g_HeapAreaEnd;
uintptr_t g_ReservedAreaBegin;
uintptr_t g_ReservedAreaEnd;
uintptr_t g_FreeAreaBegin;
uintptr_t g_FreeAreaEnd;

extern "C" void nnMain()
{
    InitTestMemory();

    nn::Result result;

    nn::svc::Handle clientSession;
    TestTag testTag;

    if (!InitTest(&clientSession, &testTag))
    {
        NN_LOG("Server is already end\n");
        return;
    }
    AutoHandleClose cSessionCloser(clientSession);

    switch(testTag)
    {
    case TestTag_ReadEvent:
        TestReadEvent(clientSession);
        break;
    case TestTag_WriteEvent:
        TestWriteEvent(clientSession);
        break;
    case TestTag_Ipc_Client_Receive:
    case TestTag_Ipc_Client_Send:
        TestClientSend(clientSession, testTag);
        break;
    case TestTag_Ipc_Server_Send:
        TestServerSend(clientSession, testTag);
        break;
    case TestTag_Ipc_Server_Receive:
        TestServerReceive(clientSession, testTag);
        break;
    case TestTag_Ipc_Server_Reply:
        TestServerReply(clientSession, testTag);
        break;
    case TestTag_Ipc_Client_Reply:
        TestClientSend(clientSession, testTag);
        break;
    case TestTag_Ipc_Client_MultiSend:
        TestClientMultiSend(clientSession, testTag);
        break;
    case TestTag_Ipc_Client_MultiReceive:
        TestClientMultiSend(clientSession, testTag);
        break;
    case TestTag_Ipc_Server_MultiSend:
        TestServerSend(clientSession, testTag);
        break;
    case TestTag_Ipc_Server_MultiReceive:
        TestServerReceive(clientSession, testTag);
        break;
    case TestTag_Ipc_Server_MultiSendMoveSession:
        TestServerSendAndPassSession(clientSession, testTag);
        break;
    case TestTag_Ipc_Server_MultiReceiveMoveSession:
        TestServerReceiveAndPassSesion(clientSession, testTag);
        break;
    case TestTag_Ipc_Process_Sending:
        TestKillIpcProcessWhileSending(clientSession, testTag);
        break;
    case TestTag_Ipc_Process_Receiving:
        TestKillIpcProcessWhileReceiving(clientSession, testTag);
        break;
    case TestTag_SendSyncLightPort:
    case TestTag_SendSyncLightSession:
        TestSendSyncRequestLight(clientSession);
        break;
    case TestTag_Thread:
        TestThreadFunc(clientSession);
        break;
    case TestTag_Interrupt:
        TestInterrupt(clientSession);
        break;
    case TestTag_Shared:
        TestShared(clientSession);
        break;
    case TestTag_Sleep:
        TestSleep(clientSession);
        break;
    case TestTag_Transfer:
        TestTransfer(clientSession);
        break;
    case TestTag_DeviceAddress:
        TestDeviceAddressSpace(clientSession);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT(readability/fn_size)

