﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <cstring>

extern "C" void nnMain();

namespace {

const nn::svc::MemoryPermission TestPermission[] = {
    nn::svc::MemoryPermission_None,
    nn::svc::MemoryPermission_Read,
    nn::svc::MemoryPermission_ReadExecute,
    nn::svc::MemoryPermission_ReadWrite,
};

const uint32_t ReadCodes[] = {
    0xe3a03602, // mov r3, #2097152; 0x200000
    0xe3a01a01, // mov r1, #4096   ; 0x1000
    0xe0833001, // add r3, r3, r1
    0xe5930000, // ldr r0, [r3]
    0xef000000 | NN_SVC_ID_EXIT_PROCESS,
};

const uint32_t WriteCodes[] = {
    0xe3a03602, // mov r3, #2097152; 0x200000
    0xe3a01a01, // mov r1, #4096   ; 0x1000
    0xe0833001, // add r3, r3, r1
    0xe5830000, // str r0, [r3]
    0xef000000 | NN_SVC_ID_EXIT_PROCESS,
};

void CheckRead(bool canRead, nn::svc::MemoryPermission permission)
{
    nn::Result result;

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);
    TestProcess process(
            2, nn::svc::CreateProcessParameterFlag_EnableJitDebug, flag, DefaultCapabilityFlagNum);
    nn::svc::Handle processHandle = process.GetHandle();
    SetExecuteCode(
            processHandle, ReadCodes, sizeof(ReadCodes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    uint64_t size = 0x1000;
    uint64_t addr = 0x201000;

    result = nn::svc::SetProcessMemoryPermission(processHandle, addr, size, permission);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartProcess(
            processHandle, TestLowestThreadPriority, g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    int64_t info;
    result = nn::svc::GetProcessInfo(
                &info, processHandle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    bool canAccess = (static_cast<nn::svc::ProcessState>(info) != nn::svc::ProcessState_WaitAttach);
    ASSERT_TRUE(canRead == canAccess);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckWrite(bool canWrite, nn::svc::MemoryPermission permission)
{
    nn::Result result;

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);
    TestProcess process(
            2, nn::svc::CreateProcessParameterFlag_EnableJitDebug, flag, DefaultCapabilityFlagNum);
    nn::svc::Handle processHandle = process.GetHandle();
    SetExecuteCode(
            processHandle, WriteCodes, sizeof(WriteCodes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    uint64_t size = 0x1000;
    uint64_t addr = 0x201000;

    result = nn::svc::SetProcessMemoryPermission(processHandle, addr, size, permission);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartProcess(
            processHandle, TestLowestThreadPriority, g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    int64_t info;
    result = nn::svc::GetProcessInfo(
                &info, processHandle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    bool canAccess = (static_cast<nn::svc::ProcessState>(info) != nn::svc::ProcessState_WaitAttach);
    ASSERT_TRUE(canWrite == canAccess);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckExecute(bool canExecute, nn::svc::MemoryPermission permission)
{
    nn::Result result;

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);
    TestProcess process(
            2, nn::svc::CreateProcessParameterFlag_EnableJitDebug, flag, DefaultCapabilityFlagNum);
    nn::svc::Handle processHandle = process.GetHandle();
    AssignExitCode(processHandle, process.GetCodeAddress(), process.GetCodeAreaSize());

    uint64_t addr = process.GetCodeAddress();
    uint64_t size = 0x1000;

    result = nn::svc::SetProcessMemoryPermission(
            processHandle, addr, size, permission);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartProcess(
            processHandle, TestLowestThreadPriority, g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    int64_t info;
    result = nn::svc::GetProcessInfo(
                &info, processHandle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    bool canAccess = (static_cast<nn::svc::ProcessState>(info) != nn::svc::ProcessState_WaitAttach);
    ASSERT_TRUE(canAccess == canExecute);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

TEST(SetProcessMemoryPermission, NoneTest)
{
    TestProcessLeak testProcessLeak;
    CheckRead(false, nn::svc::MemoryPermission_None);
    CheckWrite(false, nn::svc::MemoryPermission_None);
    CheckExecute(false, nn::svc::MemoryPermission_None);
}

TEST(SetProcessMemoryPermission, ReadTest)
{
    TestProcessLeak testProcessLeak;
    CheckRead(true, nn::svc::MemoryPermission_Read);
    CheckWrite(false, nn::svc::MemoryPermission_Read);
    CheckExecute(false, nn::svc::MemoryPermission_Read);
}

TEST(SetProcessMemoryPermission, ReadWriteTest)
{
    TestProcessLeak testProcessLeak;
    CheckRead(true, nn::svc::MemoryPermission_ReadWrite);
    CheckWrite(true, nn::svc::MemoryPermission_ReadWrite);
    CheckExecute(false, nn::svc::MemoryPermission_ReadWrite);
}

TEST(SetProcessMemoryPermission, ReadExecuteTest)
{
    TestProcessLeak testProcessLeak;
    CheckRead(true, nn::svc::MemoryPermission_ReadExecute);
    CheckWrite(false, nn::svc::MemoryPermission_ReadExecute);
    CheckExecute(true, nn::svc::MemoryPermission_ReadExecute);
}

// TEST 115-41 から115-56
// パーミッションの遷移
TEST(SetProcessMemoryPermission, FromToTest)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;

    uint64_t size = 0x1000;

    for (int32_t from = 0;
            from < static_cast<int32_t>(sizeof(TestPermission) / sizeof(nn::svc::MemoryPermission));
            from++)
    {
        for (int32_t to = 0;
                to < static_cast<int32_t>(sizeof(TestPermission) / sizeof(nn::svc::MemoryPermission));
                to++)
        {
            TestProcess process(1);
            nn::svc::Handle processHandle = process.GetHandle();
            uint64_t addr = process.GetCodeAddress();

            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, size, TestPermission[from]);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, size, TestPermission[to]);
            if (TestPermission[from] == nn::svc::MemoryPermission_ReadWrite)
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
            }
            else
            {
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
}

