﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>

namespace {
const int32_t MemoryAlign = 0x1000;
const int32_t MapSize = 0x1000;
char g_Buffer[0x2000] __attribute__((aligned(0x1000)));

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        ASSERT_TRUE(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0));

        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::SetMemoryPermission(
                addr, array[i]->GetSize(), nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);
        CheckMemory(
                addr,
                array[i]->GetState(),
                nn::svc::MemoryPermission_Read,
                array[i]->GetAttribute());

        result = nn::svc::SetMemoryPermission(
                addr, array[i]->GetSize(), nn::svc::MemoryPermission_None);
        ASSERT_RESULT_SUCCESS(result);
        CheckMemory(
                addr,
                array[i]->GetState(),
                nn::svc::MemoryPermission_None,
                array[i]->GetAttribute());

        result = nn::svc::SetMemoryPermission(
                addr, array[i]->GetSize(), nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
        CheckMemory(
                addr,
                array[i]->GetState(),
                nn::svc::MemoryPermission_ReadWrite,
                array[i]->GetAttribute());
        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::SetMemoryPermission(
                addr, array[i]->GetSize(), nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        array[i]->CheckDefaultState();
        array[i]->Close();

        array[i] = nullptr;
    }
}

}

TEST(SetMemoryPermissionTest, HeapInvalidNewMemoryPermissionTest)
{
    uintptr_t addr;
    size_t size;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == addr);
    ASSERT_TRUE(blockInfo.size == size);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

    // TEST 2-27
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_Write);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 2-28
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_Execute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 2-29
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_DontCare);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SetMemoryPermissionTest, BSSInvalidNewMemoryPermissionTest)
{
    uintptr_t addr;
    size_t size;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    size = sizeof(g_Buffer);
    addr = reinterpret_cast<uintptr_t>(g_Buffer);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

    // TEST 2-17
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_None);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 2-18
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 2-24
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_Write);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 2-25
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_Execute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 2-26
    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_DontCare);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    result = nn::svc::SetMemoryPermission(addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SetMemoryPermissionTest, MissAlignmentTest)
{
    uintptr_t addr;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    addr = reinterpret_cast<uintptr_t>(g_Buffer);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);

    // TEST 2-22
    // アドレスが0x1000 にアライメントされていない時にエラーになることを確認する
    for (int i = 1; i < MemoryAlign; i++)
    {
        result = nn::svc::SetMemoryPermission(
                addr + i, MemoryAlign, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // TEST 2-23
    // サイズが 0x1000 にアライメントされていない時にエラーになることを確認する
    for (int i = 1; i < MemoryAlign; i++)
    {
        result = nn::svc::SetMemoryPermission(
                addr, MemoryAlign + i, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }
}

TEST(SetMemoryPermissionTest, MemoryStateTest)
{
    nn::Result result;
    uintptr_t addr;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    TestMemoryInfo* header;
    GenerateMemoryStateList(&header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    nn::svc::MemoryState successStates[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_Alias,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif // SUPPORT_ALIAS_CODE_DATA
    };

    const int NumSuccess = sizeof(successStates) / sizeof(nn::svc::MemoryState);
    TestMemoryInfo* successArea[NumSuccess];
    header->GetTestListWithStates(successArea, successStates, NumSuccess);

    CheckSuccessMemoryArea(successArea, NumSuccess);

    const int NumInvalid = NumTestMemoryState - NumSuccess;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successStates, NumSuccess);

    CheckInvalidMemoryArea(invalidArea, NumInvalid);

    uint32_t successAttribute[] = {
        0,
    };
    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttr[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttr, NumInvalidAttr, successAttribute, NumSuccessAttr);
    CheckInvalidMemoryArea(invalidAttr, NumInvalidAttr);

    addr = reinterpret_cast<uintptr_t>(g_Buffer);

    // TEST 2-30
    // addr/size で指定される領域のメモリ状態が一様でない場合は失敗する
    {
        TestMapMemory map(g_FreeAreaBegin, addr + MapSize, MapSize);

        result = nn::svc::SetMemoryPermission(addr, MapSize * 2, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 2-34
    // BSS 領域の前方の使われていない領域
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, addr);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetMemoryPermission(blockInfo.baseAddress - MapSize, MapSize,
            nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-35
    // addr/size で指定される領域に確保した領域の前方の使われていない領域が含まれている
    result = nn::svc::SetHeapSize(&addr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetMemoryPermission(addr - MapSize, MapSize * 2,
            nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    result = nn::svc::SetHeapSize(&addr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

/*
   オーバーフローするようなaddr/size は利用できない
 */
TEST(SetMemoryPermissionTest, OverFlowTest)
{
    nn::Result result;
    uintptr_t addr = static_cast<uintptr_t>(-1) & ~(0xFFF);

    // TEST 2-54
    result = nn::svc::SetMemoryPermission(addr, 0x2000, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-55
    result = nn::svc::SetMemoryPermission(addr, 0x2000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 2-56
    result = nn::svc::SetMemoryPermission(addr, 0x2000, nn::svc::MemoryPermission_None);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
}

