﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>

namespace {

char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

void ReceiveAndCloseSession(nn::svc::Handle session)
{
    nn::Result result;
    int32_t index;
    nn::svc::Handle handles[1] = { session };
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceive(
                &index, handles, sizeof(handles) / sizeof(*handles),
                nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);

    result = nn::svc::CloseHandle(session);
    ASSERT_RESULT_SUCCESS(result);
}

void SendIpcThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);
    ASSERT_TRUE(clientSession != NULL);

    // リプライが送信される前にセッションが閉じられると失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(*clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}

} //namespace

TEST(SendSyncRequest, NamedPortSessionCloseTest)
{
    TestSessionLeak leakTest;
    const char* PortName = "Test2820";
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendIpcThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-20
    // リプライが送信される前にセッションが閉じられると失敗する
    result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&clientSession),
            sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    ReceiveAndCloseSession(serverSession);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequest, PortSessionCloseTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendIpcThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    int32_t index;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-21
    // リプライが送信される前にセッションが閉じられると失敗する
    result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&clientSession),
            sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    ReceiveAndCloseSession(serverSession);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequest, SessionCloseTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendIpcThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-22
    // リプライが送信される前にセッションが閉じられると失敗する
    result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&clientSession),
            sp, TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    ReceiveAndCloseSession(serverSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequest, SessionCloseTestOnNamedPort)
{
    TestSessionLeak leakTest;
    nn::Result result;
    TestHeap heap(HeapAlign);
    const char* name = "AsyncReq";

    nn::svc::Handle namedPort;
    result = nn::svc::ManageNamedPort(&namedPort, name, 1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToNamedPort(&clientSession, name);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &namedPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, namedPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-135
    // サーバーセッションが先に閉じられると失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    result = nn::svc::CloseHandle(namedPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&namedPort, name, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(namedPort == nn::svc::INVALID_HANDLE_VALUE);
}

TEST(SendSyncRequest, SessionCloseTestOnPort)
{
    TestSessionLeak leakTest;
    nn::Result result;
    TestHeap heap(HeapAlign);

    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    // TEST 28-136
    // サーバーセッションが先に閉じられると失敗する
    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}

TEST(SendSyncRequest, SessionCloseTestOnSession)
{
    TestSessionLeak leakTest;
    nn::Result result;
    TestHeap heap(HeapAlign);

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 28-137
    // サーバーセッションが先に閉じられると失敗する
    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}

