﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Tcb.h>

extern "C" void IpcHeaderAsm();

namespace {

const int64_t WaitTime = 1000 * 1000 * 1000; // 1 sec
const int DefaultAlign = 0x1000;
nn::Bit32 g_UserBuffer0[0x2000 / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));

bool g_ThreadLoop;

void ReplyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);
    int32_t index;

    while(g_ThreadLoop)
    {
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceive(
                &index, handle, 1, nn::svc::INVALID_HANDLE_VALUE, WaitTime);
        if (!g_ThreadLoop)
        {
            break;
        }
        NN_ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
        result = nn::svc::ReplyAndReceive(
                &index, handle, 0, *handle, 0);
        NN_ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
}

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;
    nn::Bit32 msgBuffer[2];
    SetOnlyIpcTag(msgBuffer, DefaultIpcTag_Send);

    for (int i = 0; i < numState; i++)
    {
        size_t msgSize = 0x1000;

        ASSERT_TRUE(array[i]->GetSize() >= msgSize && ((array[i]->GetSize() & 0xfff) == 0));
        array[i]->Initialize(msgBuffer, sizeof(msgBuffer));

        uintptr_t msgBufferAddr = array[i]->GetAddress();

        result = nn::svc::SendSyncRequestWithUserBuffer(msgBufferAddr, msgSize, handle);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(GetIpcTag(reinterpret_cast<nn::Bit32*>(msgBufferAddr)), DefaultIpcTag_Reply);

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;
    nn::Bit32 msgBuffer[2];
    SetOnlyIpcTag(msgBuffer, DefaultIpcTag_Send);

    for (int i = 0; i < numState; i++)
    {
        size_t msgSize = 0x1000;
        if (!(array[i]->GetSize() >= msgSize && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize(msgBuffer, sizeof(msgBuffer));

        uintptr_t msgBufferAddr = array[i]->GetAddress();
        if (array[i]->GetState() == nn::svc::MemoryState_Code)
        {
            msgBufferAddr = reinterpret_cast<uintptr_t>(IpcHeaderAsm);
            nn::Bit32* ptr = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
            ASSERT_EQ(ptr[0], 0x00000001);
            ASSERT_EQ(ptr[1], 0x00000000);

            nn::svc::MemoryInfo info;
            GetMemoryInfo(&info, msgBufferAddr);
            ASSERT_EQ(info.state, nn::svc::MemoryState_Code);
            ASSERT_EQ(info.permission, nn::svc::MemoryPermission_ReadExecute);
            ASSERT_LE(info.baseAddress, msgBufferAddr);
            ASSERT_GE(info.baseAddress + info.size - 1, msgBufferAddr + msgSize - 1);
        }

        result = nn::svc::SendSyncRequestWithUserBuffer(msgBufferAddr, msgSize, handle);
        if (array[i]->CanWriteData() || array[i]->GetState() == nn::svc::MemoryState_Code)
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
        else
        {
            ASSERT_RESULT_FAILURE(result);
        }

        array[i]->Close();
    }
}



} // namespace

TEST(SendSyncRequestWithUserBuffer, InvalidAlignmentTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    std::memset(g_UserBuffer0, 0, sizeof(g_UserBuffer0));

    // 前処理
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    for (uintptr_t i = 1; i < DefaultAlign; i++)
    {
        // TEST 29-7
        // 4KB にアライメントされていないアドレスは受け付けない
        result = nn::svc::SendSyncRequestWithUserBuffer(
                pMessage + i, DefaultAlign, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());

        // TEST 29-8
        // 4KB にアライメントされていないサイズは受け付けない
        result = nn::svc::SendSyncRequestWithUserBuffer(
                pMessage, DefaultAlign + i, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // 終了処理
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, InvalidMemoryAreaTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t heapPtr;

    // 前処理
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetHeapSize(&heapPtr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-9
    // メモリ領域のパーミッションが一様ではない
    {
        nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(heapPtr);
        SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);

        result = nn::svc::SetMemoryPermission(
                heapPtr + DefaultAlign, DefaultAlign, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SendSyncRequestWithUserBuffer(
                heapPtr, DefaultAlign * 2, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::SetMemoryPermission(
                heapPtr + DefaultAlign, DefaultAlign, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 29-10
    // メモリ領域の状態が一様ではない
    {
        nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(heapPtr);
        SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);

        result = nn::svc::MapMemory(g_FreeAreaBegin, heapPtr + DefaultAlign, DefaultAlign);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SendSyncRequestWithUserBuffer(
                heapPtr, DefaultAlign * 2, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        result = nn::svc::UnmapMemory(g_FreeAreaBegin, heapPtr + DefaultAlign, DefaultAlign);
        ASSERT_RESULT_SUCCESS(result);
    }

#if defined NN_BUILD_CONFIG_CPU_SVC_32
    // TEST 29-11
    // pMessage + Size が pMessage より小さいと失敗する
    {
        result = nn::svc::SendSyncRequestWithUserBuffer(0xFFFFF000, 0x2000, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
#endif
#if defined NN_BUILD_CONFIG_CPU_SVC_64 && defined NN_BUILD_CONFIG_ABI_ILP32
    // TEST 29-12
    // pMessage + Size が pMessage より小さいと失敗する
    {
        result = nn::svc::SendSyncRequestWithUserBuffer(0xFFFFF000, 0x2000, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
#endif
#if defined NN_BUILD_CONFIG_CPU_SVC_64 && defined NN_BUILD_CONFIG_ABI_LP64
    // TEST 29-13
    // pMessage + Size が pMessage より小さいと失敗する
    {
        result = nn::svc::SendSyncRequestWithUserBuffer(
                0xFFFFFFFFFFFFF000, 0x2000, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
#endif

    // 終了処理
    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, MemoryStateTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    // 前処理
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // Thread
    uintptr_t pc = reinterpret_cast<uintptr_t>(ReplyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&serverSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    g_ThreadLoop = true;
    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_CodeData,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState, clientSession);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid, clientSession);

    uint32_t successAttribute[] = {
        0,
    };
    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr, clientSession);

    const int NumInvalidAttrs = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrs[NumInvalidAttrs];
    header->GetTestListExceptAttributes(invalidAttrs, NumInvalidAttrs, successAttribute, NumSuccessAttr);
    CheckInvalidMemoryArea(invalidAttrs, NumInvalidAttrs, clientSession);

    g_ThreadLoop = false;
    result = nn::svc::CancelSynchronization(thread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);
    thread.Wait();

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, InvalidHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    size_t size = sizeof(g_UserBuffer0);

    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);

    // TEST 29-14
    // INVALID_HANDLE_VALUE を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(
            pMessage, size, nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 29-15
    // PSEUDO_HANDLE_CURRENT_THREAD を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(
            pMessage, size, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 29-16
    // PSEUDO_HANDLE_CURRENT_PROCESS を渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(
            pMessage, size, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(SendSyncRequestWithUserBuffer, InvalidLightIPCHandleTest)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    // TEST 29-17
    // Light系のセッションを渡すと失敗する
    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, true, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, InvalidNamedPortHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    const char* PortName = "Test2918";
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-18
    // NamedPort のサーバーポートを渡すと失敗する
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-19
    // NamedPort のサーバーセッショを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 29-20
    // 閉じられたNamedPort のクライアントセッションを渡すと失敗する
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, InvalidPortHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    result = nn::svc::CreatePort(&serverPort, &clientPort, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-21
    // Port のサーバーポートを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-22
    // Port のサーバーセッションを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 29-23
    // Port のクライアントポートを渡すと失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 29-24
    // 閉じられた Port のクライアントセッションを渡すと失敗する
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SendSyncRequestWithUserBuffer, InvalidSessionHandleTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    uintptr_t pMessage = reinterpret_cast<uintptr_t>(g_UserBuffer0);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(pMessage);
    size_t size = sizeof(g_UserBuffer0);

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-25
    // サーバーセッションが渡されると失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 29-26
    // 閉じられた クライアントセッションを渡されると失敗する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequestWithUserBuffer(pMessage, size, clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}

