﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/svc_ServerId.autogen.h>
#include <nn/svc/svc_DdId.autogen.h>
#include <nn/svc/svc_DmntId.autogen.h>
#include <nn/svc/svc_TcbId.autogen.h>
#include <cstring>

extern "C" void nnMain();
extern "C" void CheckReturnRegisters();

namespace {

const int64_t WaitTime = 100ull * 1000 * 1000 * 1000;

} // namespace

TEST(ReturnFromException, InvalidLoadAccess32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000003u, // bne 1f
        0xe3a03000, // mov r3, #0
        0xe5930000, // ldr r0, [r3]
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004101, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 2f
        0xe5915028, // ldr r5, [r1, #40]
        0xe2855008, // add r5, #8
        0xe5815028, // str r5, [r1, #40]
        0xe3a00000, // mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 2f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-84
    // 作成したハンドルを使用後に閉じることが出来る
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(ReturnFromException, InvalidLoadAccess64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x540000a1, // b.ne 1f
        0xd2800003, // mov x3, #0
        0xf9400060, // ldr x0, [x3]
        0x14000000, // b 2b
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2802024, // mov x4, #0x101
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0xf9402c25, // ldr x5, [x1, #88]
        0x910020a5, // add r5, #8
        0xf9002c25, // str x5, [x1, #88]
        0xd2800000, // mov x0, #0
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

TEST(ReturnFromException, InvalidLoadAccess32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t invSvc = 0xdf00;
    uint16_t mov = 0x2300; // movs r3, #0
    uint16_t ldr = 0x6818; // ldr r0, [r3, #0]
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint32_t invalidAccess = mov | ldr << 16;
    uint32_t exitCode = invSvc | exitSvc << 16;
    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        invalidAccess,
        exitCode,
        // 2:
        0xe3004101, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 3f
        0xe5915028, // ldr r5, [r1, #40]
        0xe2855004, // add r5, #4
        0xe5815028, // str r5, [r1, #40]
        0xe3a00000, // mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        // 3:
        0xeafffffe, // b 3f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(ReturnFromException, InvalidStoreAccess32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000003u, // bne 1f
        0xe3a03000, // mov r3, #0
        0xe5830000, // ldr r0, [r3]
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004101, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 2f
        0xe5915028, // ldr r5, [r1, #40]
        0xe2855008, // add r5, #8
        0xe5815028, // str r5, [r1, #40]
        0xe3a00000, // mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 2f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(ReturnFromException, InvalidStoreAccess64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x54000081, // b.ne 1f
        0xd2800003, // mov x3, #0
        0xf9000060, // ldr x0, [x3]
        0x14000000, // b 2b
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2802024, // mov x4, #0x101
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0xf9402c25, // ldr x5, [x1, #88]
        0x910010c5, // add x5, x5, #4
        0xf9002c25, // str x5, [x1, #88]
        0xd2800000, // mov x0, #0
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

TEST(ReturnFromException, InvalidStoreAccess32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t mov = 0x2300; // movs r3, #0
    uint16_t ldr = 0x6018; // str r0, [r3, #0]
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint32_t invalidAccess = mov | ldr << 16;
    uint16_t invSvc = 0xdf00;
    uint32_t exitCode = invSvc | exitSvc << 16;
    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        invalidAccess,
        exitCode,
        // 2:
        0xe3004101, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 3f
        0xe5915028, // ldr r5, [r1, #40]
        0xe2855004, // add r5, #4
        0xe5815028, // str r5, [r1, #40]
        0xe3a00000, // mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        // 3:
        0xeafffffe, // b 3f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(ReturnFromException, InvalidMemoryInstruction32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000003u, // bne 1f
        0xe3a03000, // mov r3, #0
        0xe1a0f003, // mov pc, r3
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004100, // movw r4, #0x100
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 2f
        0xe1a0500f, // mov r5, pc
        0xe2455018, // sub r5, #6 * 4
        0xe5815028, // str r5, [r1, #40]
        0xe3a00000, // mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 2f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(ReturnFromException, InvalidMemoryInstruction64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x54000081, // b.ne 1f
        0xd2800003, // mov x3, #0
        0xd61f0060, // br x3
        0x14000000, // b 2b
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2802004, // mov x4, #0x100
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0x10000005, // adr x5, .
        0xd10040a5, // sub x5, x5, #4 * 4
        0xf9002c25, // str x5, [x1, #88]
        0xd2800000, // mov x0, #0
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

TEST(ReturnFromException, InvalidMemoryInstruction32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t mov = 0x2300; // movs r3, #0
    uint16_t changePc = 0x469f; // movs pc, r3
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint32_t invalidInstruction = mov | changePc << 16;
    uint32_t exitCode = exitSvc;
    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        invalidInstruction,
        exitCode,
        // 2:
        0xe3004100, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 3f
        0xe1a0500f, // mov r5, pc
        0xe2455018, // sub r5, #6 * 4
        0xe5815028, // str r5, [r1, #40]
        0xe3a00000, // mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        // 3:
        0xeafffffe, // b 3f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(ReturnFromException, UndefinedInstruction32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 1f
        0xe7f000f0, // permanentaly undefine instruction
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004104, // movw r4, #0x104
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 2f
        0xe1a0500f, // mov r5, pc
        0xe2455018, // sub r5, #6 * 4
        0xe5815028, // str r5, [r1, #40]
        0xe3a00000, // mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 2f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(ReturnFromException, UndefinedInstruction64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x54000061, // b.ne 1f
        0xe7f000f0, // permanentaly undefine instruction
        0x14000000, // b 2b
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2802084, // mov x4, #0x104
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0xf9402c25, // ldr x5, [x1, #88]
        0x910010c5, // add x5, x5, #4
        0xf9002c25, // str x5, [x1, #88]
        0xd2800000, // mov x0, #0
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

TEST(ReturnFromException, UndefinedInstruction32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t udf = 0xde00; // movs r3, #0
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint32_t undefInstruction = udf;
    uint16_t invSvc = 0xdf00;
    uint32_t exitCode = invSvc | exitSvc << 16;
    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        undefInstruction,
        exitCode,
        // 2:
        0xe3004104, // movw r4, #0x104
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 3f
        0xe5915028, // ldr r5, [r1, #40]
        0xe2855006, // add r5, #4
        0xe5815028, // str r5, [r1, #40]
        0xe3a00000, // mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        // 3:
        0xeafffffe, // b 3f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(ReturnFromException, MultipleExceptions32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    int pageNum = 5;
    size_t codeSize = 0x2000;
    size_t dataSize = 0x3000;
    TestProcess process(pageNum, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    result = nn::svc::SetProcessMemoryPermission(
            processHandle, process.GetCodeAddress() + codeSize, dataSize,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    uint32_t codes[] = {
        // sp -------
        //     handle1
        //    -------
        //     handle2
        //    -------
        //     i
        //    -------

        0xe3500000, //  cmp r0, #0
        0x1a000033, //  bne 200450 <userExcept>
        0xe24dd00c, //  sub sp, sp, #12
        0xe59f0148, //  ldr r0, [pc, #296] ; 2004b4 <Flag0+0x4>
        0xe5900000, //  ldr r0, [r0]
        0xe3a01000, //  mov r1, #0
        0xe5801000, //  str r1, [r0]
        0xe3a02000, //  mov r2, #0
        0xe58d2008, //  str r2, [sp, #8]

        // loop1:

        // スレッドの作成
        // r0: priority
        // r1: pc
        // r2: param ( for 文の index が入る )
        // r3: sp
        // r4: idealCore
        // r5: 0x1000 ( スタックサイズ )
        // r6: i ( for 文の index )
        0xe3a0003f, //  mov r0, #63 ; 0x3f
        0xe59f1130, //  ldr r1, [pc, #304] ; 2004d8 <Flag0+0x8>
        0xe59f3130, //  ldr r3, [pc, #304] ; 2004dc <Flag0+0xc>
        0xe5933000, //  ldr r3, [r3]
        0xe59d6008, //  ldr r6, [sp, #8]
        0xe3015000, //  movw r5, #4096 ; 0x1000
        0xe0060596, //  mul r6, r6, r5
        0xe0433006, //  sub r3, r3, r6
        0xe1a04002, //  mov r4, r2
        0xe1a05000, //  mov r5, r0
        (0xef000000 | NN_SVC_ID_CREATE_THREAD), // CreateThread
        0xe3500000, //  cmp r0, #0
        0x1a00003a, //  bne 2004bc <fail>
        0xe59d6008, //  ldr r6, [sp, #8]
        0xe3a05004, //  mov r5, #4
        0xe0050695, //  mul r5, r5, r6
        0xe78d1005, //  str r1, [sp, r5]

        // スレッドの開始
        // r0: handle
        0xe1a00001, //  mov r0, r1
        (0xef000000 | NN_SVC_ID_START_THREAD), // StartThread
        0xe3500000, //  cmp r0, #0
        0x1a000032, //  bne 2004bc <fail>

        // for の更新式
        0xe59d2008, //  ldr r2, [sp, #8]
        0xe2822001, //  add r2, r2, #1
        0xe58d2008, //  str r2, [sp, #8]
        0xe3520002, //  cmp r2, #2
        0x1affffe5, //  bne 20039c <loop1>

        // 次のループのための初期化
        0xe3a00000, //  mov r0, #0
        0xe58d0008, //  str r0, [sp, #8]

        // loop2:

        // スレッドを待つ
        // r0: ns ( -1 )
        // r1: handle へのポインタ
        // r2: numHandle (1)
        // r3: ns ( -1 )
        0xe59d0008, //  ldr r0, [sp, #8]
        0xe3a02004, //  mov r2, #4
        0xe0000290, //  mul r0, r0, r2
        0xe080100d, //  add r1, r0, sp
        0xe3e00000, //  mvn r0, #0
        0xe3a02001, //  mov r2, #1
        0xe1a03000, //  mov r3, r0
        (0xef000000 | NN_SVC_ID_WAIT_SYNCHRONIZATION), // WaitSynchronization
        0xe3500000, //  cmp r0, #0
        0x1a000021, //  bne 2004bc <fail>

        // for の更新式
        0xe59d0008, //  ldr r0, [sp, #8]
        0xe2800001, //  add r0, r0, #1
        0xe58d0008, //  str r0, [sp, #8]
        0xe3500002, //  cmp r0, #2
        0x1afffff0, //  bne 20040c <loop2>

        // 終了
        0xe3a00000, //  mov r0, #0
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS),

        // userExcept:

        // PC の再設定
        0xe5915028, //  ldr r5, [r1, #40] ; 0x28
        0xe2855004, //  add r5, r5, #4
        0xe5815028, //  str r5, [r1, #40] ; 0x28

        // スタックの切り替え
        // スレッドのスタックを使う
        0xe5916020, //  ldr r6, [r1, #32]
        0xe1a0700d, //  mov r7, sp
        0xe1a0d006, //  mov sp, r6
        0xe50d7004, //  str r7, [sp, #-4]

        // 同時に実行されないかを確認するために、変数をインクリメントし、スタックとグローバル変数に保存
        0xe59f006c, //  ldr r0, [pc, #108] ; 2004e0 <Flag0+0x10>
        0xe5900000, //  ldr r0, [r0]
        0xe5905000, //  ldr r5, [r0]
        0xe2855001, //  add r5, r5, #1
        0xe5805000, //  str r5, [r0]
        0xe50d5008, //  str r5, [sp, #-8]

        // 一時休眠
        0xe30003e8, //  movw r0, #1000
        0xe0000090, //  mul r0, r0, r0
        0xe3a01000, //  mov r1, #0
        (0xef000000 | NN_SVC_ID_SLEEP_THREAD), // SleepThread

        // グローバル変数の値が変化していないか比較
        0xe59f0048, //  ldr r0, [pc, #72] ; 2004e4 <Flag0+0x14>
        0xe5900000, //  ldr r0, [r0]
        0xe5900000, //  ldr r0, [r0]
        0xe51d7004, //  ldr r7, [sp, #-4]
        0xe51d5008, //  ldr r5, [sp, #-8]
        0xe1a0d007, //  mov sp, r7
        0xe1500005, //  cmp r0, r5
        0x1a000001, //  bne 200498 <fail>

        // ユーザー例外終了
        0xe3a00000, //  mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION),

        // fail
        0xeafffffe, //  b 200498 <fail>

        // TestThread
        0xe3a00000, //  mov r0, #0
        0xe5900000, //  ldr r0, [r0]
        (0xef000000 | NN_SVC_ID_EXIT_THREAD),

        // Stack
        static_cast<uint32_t>(process.GetCodeAddress() + process.GetCodeAreaSize()),

        // Flag0
        static_cast<uint32_t>(process.GetCodeAddress() + codeSize),
        0x00200158, // Flag が保存されているアドレス
        0x00200148, // TestThread のアドレス
        0x00200154, // Stack が保存されているアドレス
        0x00200158, // Flag が保存されているアドレス
        0x00200158, // Flag が保存されているアドレス
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), codeSize);

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
} // NOLINT(impl/function_size)

TEST(ReturnFromException, MultipleExceptions32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    int pageNum = 5;
    size_t codeSize = 0x2000;
    size_t dataSize = 0x3000;
    TestProcess process(pageNum, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    result = nn::svc::SetProcessMemoryPermission(
            processHandle, process.GetCodeAddress() + codeSize, dataSize,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    uint16_t invSvc = 0xdf00;
    uint16_t mov = 0x2300; // movs r3, #0
    uint16_t ldr = 0x6818; // ldr r0, [r3, #0]
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_THREAD;
    uint32_t invalidAccess = mov | ldr << 16;
    uint32_t exitCode = invSvc | exitSvc << 16;

    uint32_t codes[] = {
        // sp -------
        //     handle1
        //    -------
        //     handle2
        //    -------
        //     i
        //    -------

        0xe3500000, //  cmp r0, #0
        0x1a000033, //  bne 200450 <userExcept>
        0xe24dd00c, //  sub sp, sp, #12
        0xe59f0148, //  ldr r0, [pc, #296] ; 2004b4 <Flag0+0x4>
        0xe5900000, //  ldr r0, [r0]
        0xe3a01000, //  mov r1, #0
        0xe5801000, //  str r1, [r0]
        0xe3a02000, //  mov r2, #0
        0xe58d2008, //  str r2, [sp, #8]

        // loop1:

        // スレッドの作成
        // r0: priority
        // r1: pc
        // r2: param ( for 文の index が入る )
        // r3: sp
        // r4: idealCore
        // r5: 0x1000 ( スタックサイズ )
        // r6: i ( for 文の index )
        0xe3a0003f, //  mov r0, #63 ; 0x3f
        0xe59f1130, //  ldr r1, [pc, #304] ; 2004d8 <Flag0+0x8>
        0xe59f3130, //  ldr r3, [pc, #304] ; 2004dc <Flag0+0xc>
        0xe5933000, //  ldr r3, [r3]
        0xe59d6008, //  ldr r6, [sp, #8]
        0xe3015000, //  movw r5, #4096 ; 0x1000
        0xe0060596, //  mul r6, r6, r5
        0xe0433006, //  sub r3, r3, r6
        0xe1a04002, //  mov r4, r2
        0xe1a05000, //  mov r5, r0
        (0xef000000 | NN_SVC_ID_CREATE_THREAD), // CreateThread
        0xe3500000, //  cmp r0, #0
        0x1a00003a, //  bne 2004bc <fail>
        0xe59d6008, //  ldr r6, [sp, #8]
        0xe3a05004, //  mov r5, #4
        0xe0050695, //  mul r5, r5, r6
        0xe78d1005, //  str r1, [sp, r5]

        // スレッドの開始
        // r0: handle
        0xe1a00001, //  mov r0, r1
        (0xef000000 | NN_SVC_ID_START_THREAD), // StartThread
        0xe3500000, //  cmp r0, #0
        0x1a000032, //  bne 2004bc <fail>

        // for の更新式
        0xe59d2008, //  ldr r2, [sp, #8]
        0xe2822001, //  add r2, r2, #1
        0xe58d2008, //  str r2, [sp, #8]
        0xe3520002, //  cmp r2, #2
        0x1affffe5, //  bne 20039c <loop1>

        // 次のループのための初期化
        0xe3a00000, //  mov r0, #0
        0xe58d0008, //  str r0, [sp, #8]

        // loop2:

        // スレッドを待つ
        // r0: ns ( -1 )
        // r1: handle へのポインタ
        // r2: numHandle (1)
        // r3: ns ( -1 )
        0xe59d0008, //  ldr r0, [sp, #8]
        0xe3a02004, //  mov r2, #4
        0xe0000290, //  mul r0, r0, r2
        0xe080100d, //  add r1, r0, sp
        0xe3e00000, //  mvn r0, #0
        0xe3a02001, //  mov r2, #1
        0xe1a03000, //  mov r3, r0
        (0xef000000 | NN_SVC_ID_WAIT_SYNCHRONIZATION), // WaitSynchronization
        0xe3500000, //  cmp r0, #0
        0x1a000021, //  bne 2004bc <fail>

        // for の更新式
        0xe59d0008, //  ldr r0, [sp, #8]
        0xe2800001, //  add r0, r0, #1
        0xe58d0008, //  str r0, [sp, #8]
        0xe3500002, //  cmp r0, #2
        0x1afffff0, //  bne 20040c <loop2>

        // 終了
        0xe3a00000, //  mov r0, #0
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS),

        // userExcept:

        // PC の再設定
        0xe5915028, //  ldr r5, [r1, #40] ; 0x28
        0xe2855004, //  add r5, r5, #4
        0xe5815028, //  str r5, [r1, #40] ; 0x28

        // スタックの切り替え
        // スレッドのスタックを使う
        0xe5916020, //  ldr r6, [r1, #32]
        0xe1a0700d, //  mov r7, sp
        0xe1a0d006, //  mov sp, r6
        0xe50d7004, //  str r7, [sp, #-4]

        // 同時に実行されないかを確認するために、変数をインクリメントし、スタックとグローバル変数に保存
        0xe59f006c, //  ldr r0, [pc, #108] ; 2004e0 <Flag0+0x10>
        0xe5900000, //  ldr r0, [r0]
        0xe5905000, //  ldr r5, [r0]
        0xe2855001, //  add r5, r5, #1
        0xe5805000, //  str r5, [r0]
        0xe50d5008, //  str r5, [sp, #-8]

        // 一時休眠
        0xe30003e8, //  movw r0, #1000
        0xe0000090, //  mul r0, r0, r0
        0xe3a01000, //  mov r1, #0
        (0xef000000 | NN_SVC_ID_SLEEP_THREAD), // SleepThread

        // グローバル変数の値が変化していないか比較
        0xe59f0048, //  ldr r0, [pc, #72] ; 2004e4 <Flag0+0x14>
        0xe5900000, //  ldr r0, [r0]
        0xe5900000, //  ldr r0, [r0]
        0xe51d7004, //  ldr r7, [sp, #-4]
        0xe51d5008, //  ldr r5, [sp, #-8]
        0xe1a0d007, //  mov sp, r7
        0xe1500005, //  cmp r0, r5
        0x1a000001, //  bne 200498 <fail>

        // ユーザー例外終了
        0xe3a00000, //  mov r0, #0
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION),

        // fail
        0xeafffffe, //  b 200498 <fail>

        // TestThread
        0xfaffffff, //  blx 1f
        invalidAccess,
        exitCode,

        // Stack
        static_cast<uint32_t>(process.GetCodeAddress() + process.GetCodeAreaSize()),

        // Flag0
        static_cast<uint32_t>(process.GetCodeAddress() + codeSize),
        0x00200158, // Flag が保存されているアドレス
        0x00200148, // TestThread のアドレス
        0x00200154, // Stack が保存されているアドレス
        0x00200158, // Flag が保存されているアドレス
        0x00200158, // Flag が保存されているアドレス
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), codeSize);

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
} // NOLINT(impl/function_size)

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(ReturnFromException, MultipleExceptions64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    int pageNum = 5;
    size_t codeSize = 0x2000;
    size_t dataSize = 0x3000;
    TestProcess process(pageNum, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    result = nn::svc::SetProcessMemoryPermission(
            processHandle, process.GetCodeAddress() + codeSize, dataSize,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    uint32_t codes[] = {
        // sp -------
        //     handle1
        //    -------
        //     handle2
        //    -------
        //     i
        //    -------

        0xf100001f, //    cmp x0, #0x0
        0x540006c1, //    b.ne    200454 <userExcept>
        0xd10063ff, //    sub sp, sp, #0x18
        0x58000aa0, //    ldr x0, 2004d8 <Flag0+0x4>
        0xb9400000, //    ldr w0, [x0]
        0xd2800001, //    mov x1, #0x0                    // #0
        0xf9000001, //    str x1, [x0]
        0xd2800002, //    mov x2, #0x0                    // #0
        0xf9000be2, //    str x2, [sp,#8 * 2]

        // loop1:

        // スレッドの作成
        // x0: priority ( 念のため )
        // x1: pc
        // x2: param ( for 文の index が入る )
        // x3: sp
        // x4: priority
        // x5: idealCore
        // x6: pOut
        0xd28007e0, //    mov x0, #0x3f                   // #63
        0x58000a01, //    ldr x1, 2004e0 <Flag0+0xc>
        0x58000a23, //    ldr x3, 2004e8 <Flag0+0x14>
        0xb9400063, //    ldr w3, [x3]
        0xf9400be6, //    ldr x6, [sp,#8 * 2]
        0xd2820005, //    mov x5, #0x1000                 // #4096
        0x9b057cc6, //    mul x6, x6, x5
        0xcb060063, //    sub x3, x3, x6
        0xaa0203e5, //    mov x5, x2
        0xaa0003e4, //    mov x4, x0
        (0xd4000001 | (NN_SVC_ID_CREATE_THREAD << 5)), // svc CreateThread
        0xf100001f, //    cmp x0, #0x0
        0x540007a1, //    b.ne    2004c0 <fail>
        0xf9400be6, //    ldr x6, [sp, #16]
        0xd2800105, //    mov x5, #0x8
        0x9b057cc6, //    mul x6, x6, x5
        0xf8266be1, //    str x1, [sp,x6]

        // スレッドの開始
        // x0: handle
        0xaa0103e0, //    mov x0, x1
        (0xd4000001 | (NN_SVC_ID_START_THREAD << 5)), // svc StartThread
        0xf100001f, //    cmp x0, #0x0
        0x540006a1, //    b.ne    2004c0 <fail>

        // for の更新式
        0xf9400be2, //    ldr x2, [sp,#8 * 2]
        0x91000442, //    add x2, x2, #0x1
        0xf9000be2, //    str x2, [sp,#8 * 2]
        0xf100085f, //    cmp x2, #0x2
        0x54fffce1, //    b.ne    20039c <loop1>

        // 次のループのための初期化
        0xd2800000, //    mov x0, #0x0                    // #0
        0xf9000be0, //    str x0, [sp,#8 * 2]

        // loop2:

        // スレッドを待つ
        // x0: ns ( -1 )
        // x1: handle へのポインタ
        // x2: numHandle (1)
        // x3: ns ( -1 )
        0xf9400be0, //    ldr x0, [sp,#8 * 2]
        0xd2800102, //    mov x2, #0x8                    // #8
        0x9b027c00, //    mul x0, x0, x2
        0x910003e2, //    mov x2, sp
        0x8b020001, //    add x1, x0, x2
        0x92800000, //    mov x0, #0xffffffffffffffff   // #-1
        0xd2800022, //    mov x2, #0x1                    // #1
        0xaa0003e3, //    mov x3, x0
        (0xd4000001 | (NN_SVC_ID_WAIT_SYNCHRONIZATION << 5)), // svc WaitSynchronization
        0xf100001f, //    cmp x0, #0x0
        0x54000461, //    b.ne    2004c0 <fail>

        // for の更新式
        0xf9400be0, //    ldr x0, [sp,#8 * 2]
        0x91000400, //    add x0, x0, #0x1
        0xf9000be0, //    str x0, [sp,#8 * 2]
        0xf100081f, //    cmp x0, #0x2
        0x54fffe21, //    b.ne    2004a8 <loop2>

        // 終了
        0xd2800000, //    mov x0, #0x0                    // #0
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)), // svc ExitProcess

        // userExcept:

        // PC の再設定

        0xf9402c25, //    ldr x5, [x1, #88]
        0x910010a5, //    add x5, x5, #4
        0xf9002c25, //    str x5, [x1, #88]

        // スタックの切り替え
        // スレッドのスタックを使う
        0xf9402826, //    ldr x6, [x1,#80]
        0x910003e5, //    mov x5, sp
        0x910000df, //    mov sp, x6
        0xf81f80c5, //    str x5, [x6,#-8]

        // 同時に実行されないかを確認するために、変数をインクリメントし、スタックとグローバル変数に保存
        0x58000340, //    ldr x0, 2004d8 <Flag0+0x8>
        0xb9400000, //    ldr w0, [x0]
        0xf9400005, //    ldr x5, [x0]
        0x910004a5, //    add x5, x5, #0x1
        0xf9000005, //    str x5, [x0]
        0xf81f03e5, //    str x5, [sp,#-16]

        // 一時休眠
        0xd2807d00, //    mov x0, #0x3e8                  // #1000
        0x9b007c00, //    mul x0, x0, x0
        0xd2800001, //    mov x1, #0x0                    // #0
        (0xd4000001 | (NN_SVC_ID_SLEEP_THREAD << 5)), // svc SleepThread

        // 値が変化していないか比較
        0x58000200, //    ldr x0, 2004d8 <Flag0+0x8>
        0xb9400000, //    ldr w0, [x0]
        0xf9400000, //    ldr x0, [x0]
        0xf85f83e6, //    ldr x6, [sp,#-8]
        0xf85f03e5, //    ldr x5, [sp,#-16]
        0x910000df, //    mov sp, x6
        0xeb05001f, //    cmp x0, x5
        0x54000061, //    b.ne    2004c0 <fail>

        // ユーザー例外終了
        0xd2800000, //    mov x0, #0x0                    // #0
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)), // svc ReturnFromException

        // fail
        0x14000000, //    b   2004c0 <fail>

        // TestThread
        0xd2800000, //    mov x0, #0x0                    // #0
        0xf9400000, //    ldr x0, [x0]
        (0xd4000001 | (NN_SVC_ID_EXIT_THREAD << 5)), // svc ExitThread

        // Stack
        static_cast<uint32_t>(process.GetCodeAddress() + process.GetCodeAreaSize()),

        // Flag0
        static_cast<uint32_t>(process.GetCodeAddress() + codeSize),
        0x0020015c, // Flag が保存されているアドレス
        0x00000000,
        0x0020014c, // TestThread のアドレス
        0x00000000,
        0x00200158, // Stack が保存されているアドレス
        0x00000000,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), codeSize);

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
} // NOLINT(impl/function_size)
#endif // NN_BUILD_CONFIG_CPU_ARM_V8A

TEST(ReturnFromException, CallInNormalMode32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 1f
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004301, // movw r4, #0x301
        0xe1500004, // cmp r0, r4
        0x1a000000u, // bne 2f
        // ExceptionInfo のレジスタが破壊されている可能性があるので、ここで終了する
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xeafffffe, // b 2f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(ReturnFromException, CallInNormalMode32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t returnFromException = 0xdf00 | NN_SVC_ID_RETURN_FROM_EXCEPTION;
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint32_t callSvc = returnFromException | exitSvc << 16;
    NN_LOG("callSvc: 0x%x\n", callSvc);
    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000001u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        callSvc,
        // 2:
        0xe3004301, // movw r4, #0x301
        0xe1500004, // cmp r0, r4
        0x1a000000u, // bne 3f
        // ExceptionInfo のレジスタが破壊されている可能性があるので、ここで終了する
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        // 3:
        0xeafffffe, // b 3f
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(ReturnFromException, CallInNormalMode64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t infinitLoop = 0x14000000;
    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x54000081, // b.ne 1f
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        infinitLoop,
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2806024, // mov r4, #0x301
        0xeb04001f, // cmp x0, x4
        0x54000041, // b.ne 2f
        // ExceptionInfo のレジスタが破壊されている可能性があるので、ここで終了する
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 2:
        infinitLoop, // b 2b
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif // NN_BUILD_CONFIG_CPU_ARM_V8A

TEST(ReturnFromException, ResultNotHandled32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000003u, // bne 1f
        0xe3a03000, // mov r3, #0
        0xe5930000, // ldr r0, [r3]
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004101, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000005u, // bne 2f
        0xe5915028, // ldr r5, [r1, #40]
        0xe2855008, // add r5, #8
        0xe5815028, // str r5, [r1, #40]
        0xe59f0008, // ldr r0, [pc, #8]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0x00000000,
        0xeafffffe, // b 2f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    int64_t procInfo;
    result = nn::svc::GetProcessInfo(&procInfo, processHandle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(procInfo) == nn::svc::ProcessState_WaitAttach);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-84
    // 作成したハンドルを使用後に閉じることが出来る
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(ReturnFromException, ResultNotHandled32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t invSvc = 0xdf00;
    uint16_t mov = 0x2300; // movs r3, #0
    uint16_t ldr = 0x6818; // ldr r0, [r3, #0]
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint32_t invalidAccess = mov | ldr << 16;
    uint32_t exitCode = invSvc | exitSvc << 16;
    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        invalidAccess,
        exitCode,
        // 2:
        0xe3004101, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000004u, // bne 3f
        0xe5915028, // ldr r5, [r1, #40]
        0xe2855004, // add r5, #4
        0xe5815028, // str r5, [r1, #40]
        0xe59f0008, // ldr r0, [pc, #8]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 3f
        0x00000000,
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    int64_t procInfo;
    result = nn::svc::GetProcessInfo(&procInfo, processHandle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(procInfo) == nn::svc::ProcessState_WaitAttach);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(ReturnFromException, ResultNotHandled64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit
        | nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x540000a1, // b.ne 1f
        0xd2800003, // mov x3, #0
        0xf9400060, // ldr x0, [x3]
        0x14000000, // b 2b
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2802024, // mov x4, #0x101
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0xf9402c25, // ldr x5, [x1, #88]
        0x910020a5, // add r5, #8
        0xf9002c25, // str x5, [x1, #88]
        0x58000080, // ldr x0, =result
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
        0x00000000,
        // result:
        nn::svc::ResultNotHandled().InnerValue,
        0x00000000,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    int64_t procInfo;
    result = nn::svc::GetProcessInfo(&procInfo, processHandle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(procInfo) == nn::svc::ProcessState_WaitAttach);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A && defined NN_BUILD_CONFIG_ABI_LP64
TEST(ReturnFromException, Register64bitChecks)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit
        | nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();
    uint32_t* codes = reinterpret_cast<uint32_t*>(CheckReturnRegisters);

    SetExecuteCode(
            processHandle, codes, 0x1000,
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    int64_t procInfo;
    result = nn::svc::GetProcessInfo(&procInfo, processHandle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(procInfo) == nn::svc::ProcessState_Terminated ||
            static_cast<nn::svc::ProcessState>(procInfo) == nn::svc::ProcessState_Terminating);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

#if (defined NN_BUILD_CONFIG_CPU_ARM_V8A && defined NN_BUILD_CONFIG_ABI_ILP32) || \
    (defined NN_BUILD_CONFIG_CPU_ARM_V7A)
TEST(ReturnFromException, Register32bitChecks)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();
    uint32_t* codes = reinterpret_cast<uint32_t*>(CheckReturnRegisters);

    SetExecuteCode(
            processHandle, codes, 0x1000,
            process.GetCodeAddress(), process.GetCodeAreaSize());

    process.Start();
    nn::svc::ResetSignal(processHandle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    int64_t procInfo;
    result = nn::svc::GetProcessInfo(&procInfo, processHandle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(procInfo) == nn::svc::ProcessState_Terminated ||
            static_cast<nn::svc::ProcessState>(procInfo) == nn::svc::ProcessState_Terminating);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif
