﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"

extern nn::Result ClientSendLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReceiveLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReplyLightRequest(nn::svc::Handle handle);

namespace {

char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));

void ClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);

    result = ClientSendLightRequest(*handle);
    NN_ASSERT_RESULT_SUCCESS(result);
}

} // namespace

TEST(ReplyAndReceiveLight, CloseClientSessionBeforeReceiving)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    int32_t index;


    {
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 1);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);
        AutoHandleClose cPortCloser(clientPort);

        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(clientSession);

        result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::AcceptSession(&serverSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);

        result = cSessionCloser.Close();
        ASSERT_RESULT_SUCCESS(result);

        // TEST 66-16
        // 受信する前にクライアントセッションが閉じられていると失敗する
        result = ServerReceiveLightRequest(serverSession);
        ASSERT_RESULT_FAILURE(result);
    }

    {
        result = nn::svc::CreateSession(&serverSession, &clientSession, true, 1);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        result = cSessionCloser.Close();
        ASSERT_RESULT_SUCCESS(result);

        // TEST 66-17
        // 受信する前にクライアントセッションが閉じられていると失敗する
        result = ServerReceiveLightRequest(serverSession);
        ASSERT_RESULT_FAILURE(result);
    }
}

TEST(ReplyAndReceiveLight, CloseClientSessionForReceving)
{
    TestLightSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    int32_t index;

    {
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 1);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);
        AutoHandleClose cPortCloser(clientPort);

        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(clientSession);

        result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::AcceptSession(&serverSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);

        uintptr_t pc = reinterpret_cast<uintptr_t>(ClientThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        result = ServerReceiveLightRequest(serverSession);
        ASSERT_RESULT_SUCCESS(result);

        cSessionCloser.Close();

        // TEST 66-18
        // 要求を受信中にクライアントセッションを閉じると失敗する
        result = ServerReplyLightRequest(serverSession);
        ASSERT_RESULT_FAILURE(result);

        thread.Wait();
    }

    {
        result = nn::svc::CreateSession(&serverSession, &clientSession, true, 1);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        uintptr_t pc = reinterpret_cast<uintptr_t>(ClientThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        result = ServerReceiveLightRequest(serverSession);
        ASSERT_RESULT_SUCCESS(result);

        cSessionCloser.Close();

        // TEST 66-19
        // 要求を受信中にクライアントセッションを閉じると失敗する
        result = ServerReplyLightRequest(serverSession);
        ASSERT_RESULT_FAILURE(result);

        thread.Wait();
    }
}

