﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <cstring>

namespace {

void CreateAndSetupProcess(nn::svc::Handle* pOut, const nn::svc::CreateProcessParameter& param,
                           uint32_t* codePtr, size_t codeSize)
{
    nn::Result result;
    nn::svc::Handle handle;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + 1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(
            g_FreeAreaBegin, handle, param.memoryAddress, param.memoryNumPages * 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    uint32_t* ptr = reinterpret_cast<uint32_t*>(g_FreeAreaBegin);
    ::std::memcpy(ptr, codePtr, codeSize);

    result = nn::svc::UnmapProcessMemory(
            g_FreeAreaBegin, handle, param.memoryAddress, param.memoryNumPages * 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetProcessMemoryPermission(
            handle, param.memoryAddress, param.memoryNumPages * 0x1000,
            nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_SUCCESS(result);

    *pOut = handle;
}

void CreateRunnableProcess(nn::svc::Handle* pOut)
{
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);
    uint32_t code = 0xef000000 | NN_SVC_ID_EXIT_PROCESS; // svc NN_SVC_ID_EXIT_PROCESS
    CreateAndSetupProcess(pOut, param, &code, sizeof(uint32_t));
}


} // namespace

TEST(GetProcessInfo, RunningAndJit)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int64_t info;
    uint32_t codes[] = {
        0xe3a00000, // mov r0, #0
        0xe5803000, // str r3, [r0] ; ここでNULL アクセス
        0xef000000 | NN_SVC_ID_EXIT_PROCESS, // ここにはこないはず
    };

    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);
    param.flags = nn::svc::CreateProcessParameterFlag_EnableJitDebug;
    CreateAndSetupProcess(&handle, param, codes, sizeof(codes));
    AutoHandleClose processCloser(handle);

    int32_t priority = TestLowestThreadPriority;
    int32_t coreNo = g_ProcessIdealCore;
    result = nn::svc::StartProcess(handle, priority, coreNo, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ResetSignal(handle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ResetSignal(handle);

    // TEST 124-15
    // 対象のプロセスが RUNNING 状態で、かつ JIT デバッグとなる状況が発生してから、
    // DebugActiveProcess が呼び出されるまでの間 に GetProcessInfo を呼ぶ
    result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_WaitAttach);

    nn::Bit64 pid;
    result = nn::svc::GetProcessId(&pid, handle);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle debugHandle;
    result = nn::svc::DebugActiveProcess(&debugHandle, pid);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose debugCloser(debugHandle);

    // TEST 124-16
    // 対象のプロセスが RUNNING 状態で、かつ JIT デバッグとなる状況が発生し、
    // DebugActiveProcess を呼び出した後に GetProcessInfo を呼ぶ
    result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Breaked);

    result = nn::svc::TerminateDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Terminated ||
            static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Terminating);
}

TEST(GetProcessInfo, ProcessStateTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int64_t info;

    {
        CreateRunnableProcess(&handle);
        AutoHandleClose processCloser(handle);

        // 124-18
        // 実行前のプロセスの情報を取得できる
        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Initializing);

        // テストプロセスと同じコア上で低優先度で動かすことで、プリエンプトさせない
        int32_t priority = TestLowestThreadPriority;
        int32_t coreNo = g_ProcessIdealCore;
        result = nn::svc::StartProcess(handle, priority, coreNo, DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);

        // 124-19
        // 実行中のプロセスの情報を取得できる
        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Running);

        nn::svc::ResetSignal(handle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        // 124-20
        // 実行後のプロセスの情報を取得できる
        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Terminated ||
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Terminating);
    }

    {
        CreateRunnableProcess(&handle);
        AutoHandleClose processCloser(handle);

        nn::Bit64 pid;
        result = nn::svc::GetProcessId(&pid, handle);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle debug;
        result = nn::svc::DebugActiveProcess(&debug, pid);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 124-22
        // 実行前のデバッグプロセスの情報が取得できる
        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_PreAttached);

        result = nn::svc::CloseHandle(debug);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Initializing);

        result = nn::svc::DebugActiveProcess(&debug, pid);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_PreAttached);

        // テストプロセスと同じコア上で低優先度で動かすことで、プリエンプトさせない
        int32_t priority = TestLowestThreadPriority;
        int32_t coreNo = g_ProcessIdealCore;
        result = nn::svc::StartProcess(handle, priority, coreNo, DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 124-23
        // 実行中のデバッグプロセスの情報が取得できる
        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Breaked);

        result = nn::svc::CloseHandle(debug);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Running);

        nn::svc::ResetSignal(handle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Terminated ||
                static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Terminating
                );
    }
}

