﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestIpc.h"
#include "util_TestLoader.h"
#include "test_DebugProcess_Register.h"
#include "util_DebugProcess.h"
#include <nn/svc/svc_Thread.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <cstring>
#include <nn/init.h>

extern char BinDebugProcess_begin[];
extern char BinDebugProcess_end[];

extern "C" void nndiagStartup()
{
}

extern "C" void nninitStartup()
{
}

namespace {
const int64_t WaitTime = 100 * 1000 * 1000;
const size_t InstructionSize = 4;

void PrintContextRegisters(nn::svc::ThreadContext context, nn::Bit32 flag)
{
#if defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    if ((nn::svc::ContextFlag_General & flag) > 0)
    {
        for(int r = 0; r < 13; r++)
        {
            NN_LOG("cpuRegisters[%d] = 0x%x\n", r, context.r[r]);
        }
    }
    if ((nn::svc::ContextFlag_Control & flag) > 0)
    {
        NN_LOG("sp = 0x%x\n", context.sp);
        NN_LOG("lr = 0x%x\n", context.lr);
        NN_LOG("pc = 0x%x\n", context.pc);
        NN_LOG("cpsr = 0x%x\n", context.cpsr);
    }
    if ((nn::svc::ContextFlag_Fpu & flag) > 0)
    {
        for(int f = 0; f < 16; f++)
        {
            NN_LOG("fpuRegisters[%d] = 0x%x\n", f, context.fpuRegisters[f]);
        }
    }
    if ((nn::svc::ContextFlag_FpuControl & flag) > 0)
    {
        NN_LOG("fpscr = 0x%x\n", context.fpscr);
        NN_LOG("fpexc = 0x%x\n", context.fpexc);
    }
#elif  defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    if ((nn::svc::ContextFlag_General & flag) > 0)
    {
        for(int r = 0; r < 29; r++)
        {
            NN_LOG("cpuRegisters[%d] = 0x%llx\n", r, context.r[r]);
        }
    }
    if ((nn::svc::ContextFlag_Control & flag) > 0)
    {
        NN_LOG("fp = 0x%llx\n", context.fp);
        NN_LOG("sp = 0x%llx\n", context.sp);
        NN_LOG("lr = 0x%llx\n", context.lr);
        NN_LOG("pc = 0x%llx\n", context.pc);
        NN_LOG("pstate = 0x%x\n", context.pstate);
    }
    if ((nn::svc::ContextFlag_Fpu & flag) > 0)
    {
        nn::Bit64* fpu = reinterpret_cast<nn::Bit64*>(&context.v[0]);
        for(int f = 0; f < 64; f++, fpu++)
        {
            NN_LOG("fpuRegisters[%d] = 0x%llx\n", f, *fpu);
        }
    }
    if ((nn::svc::ContextFlag_FpuControl & flag) > 0)
    {
        NN_LOG("fpcr = 0x%x\n", context.fpcr);
        NN_LOG("fpsr = 0x%x\n", context.fpsr);
    }
#endif
}

void PrintContextAllRegisters(nn::svc::Handle debugHandle, nn::Bit64 tid)
{
    nn::Result result;
    nn::svc::ThreadContext context;
    nn::Bit32 contextFlags =
        nn::svc::ContextFlag_General |
        nn::svc::ContextFlag_Control |
        //nn::svc::ContextFlag_Fpu |
        nn::svc::ContextFlag_FpuControl;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlags);
    ASSERT_RESULT_SUCCESS(result);

    NN_LOG("+++ All Registers +++\n");
    PrintContextRegisters(context, contextFlags);
    NN_LOG("--- All Registers ---\n");
}

void CheckDebugBreak(nn::svc::DebugEventInfo debugInfo)
{
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    nn::svc::DebugInfoException except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);
}

void CheckNormalState(nn::svc::Handle debugHandle, nn::Bit64 tid)
{
    nn::Result result;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::ThreadContext context;
    nn::svc::ThreadContext cmpContext;
    nn::Bit32 contextFlag;

    // セットしない場合
    nn::svc::SleepThread(WaitTime);

    result = nn::svc::BreakDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    CheckDebugBreak(debugInfo);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    // セットする場合
    nn::svc::SleepThread(WaitTime);

    result = nn::svc::BreakDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    CheckDebugBreak(debugInfo);

    contextFlag = nn::svc::ContextFlag_Control;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    context.pc += InstructionSize;

    result = nn::svc::SetDebugThreadContext(debugHandle, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetDebugThreadContext(&cmpContext, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(::std::memcmp(&context, &cmpContext, sizeof(context)) == 0);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckSvcState(nn::svc::Handle debugHandle, nn::Bit64 tid, nn::svc::Handle serverSession)
{
    nn::Result result;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::ThreadContext context;
    nn::svc::ThreadContext cmpContext;
    nn::Bit32 contextFlag;
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    int32_t index;

    ipcMsg.SetNull();
    result = nn::svc::ReplyAndReceive(
            &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::BreakDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    CheckDebugBreak(debugInfo);

    contextFlag = nn::svc::ContextFlag_Control;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    context.lr += InstructionSize;

    // SVC 中はセットできない
    result = nn::svc::SetDebugThreadContext(debugHandle, tid, context, contextFlag);
    ASSERT_RESULT_FAILURE(result);

    result = nn::svc::GetDebugThreadContext(&cmpContext, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    context.lr -= InstructionSize;
    ASSERT_TRUE(::std::memcmp(&context, &cmpContext, sizeof(context)) == 0);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    result = nn::svc::ReplyAndReceive(
            &index, &serverSession, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // ここで、PC + 4 の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckReturnFromExceptionState(nn::svc::Handle debugHandle, nn::Bit64 tid)
{
    nn::Result result;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::ThreadContext context;
    nn::svc::ThreadContext cmpContext;
    nn::Bit32 contextFlag;
    nn::svc::DebugInfoException except;

    // セットなし
    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UndefinedSystemCall);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    // セットあり
    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UndefinedSystemCall);

    contextFlag = nn::svc::ContextFlag_Control;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    context.pc += InstructionSize;

    result = nn::svc::SetDebugThreadContext(debugHandle, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetDebugThreadContext(&cmpContext, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(::std::memcmp(&context, &cmpContext, sizeof(context)) == 0);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckBreakState(nn::svc::Handle debugHandle, nn::Bit64 tid)
{
    nn::Result result;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::ThreadContext context;
    nn::svc::ThreadContext cmpContext;
    nn::Bit32 contextFlag;

    // セットなし
    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    nn::svc::DebugInfoException except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UserBreak);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    // セットあり
    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UserBreak);

    contextFlag = nn::svc::ContextFlag_Control;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    context.pc += InstructionSize;

    result = nn::svc::SetDebugThreadContext(debugHandle, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetDebugThreadContext(&cmpContext, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(::std::memcmp(&context, &cmpContext, sizeof(context)) == 0);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckExceptionState(nn::svc::Handle debugHandle, nn::Bit64 tid)
{
    nn::Result result;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::ThreadContext context;
    nn::svc::ThreadContext cmpContext;
    nn::Bit32 contextFlag;

    // セットなし
    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    nn::svc::DebugInfoException except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AccessViolationData);
    ASSERT_TRUE(except.exceptionAddress == 0x4);
    ASSERT_TRUE(except.detail.accessViolationData.faultAddress == 0x0);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    // セットあり
    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AccessViolationData);
    ASSERT_TRUE(except.exceptionAddress == 0x4);
    ASSERT_TRUE(except.detail.accessViolationData.faultAddress == 0x0);

    contextFlag = nn::svc::ContextFlag_Control;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    context.pc += InstructionSize;

    result = nn::svc::SetDebugThreadContext(debugHandle, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetDebugThreadContext(&cmpContext, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(::std::memcmp(&context, &cmpContext, sizeof(context)) == 0);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckNoUpdate(nn::svc::Handle debugHandle, nn::Bit64 tid)
{
    nn::Result result;
    nn::svc::DebugEventInfo debugInfo;
    nn::svc::ThreadContext context;
    nn::Bit32 contextFlag;

    // セットなし
    nn::svc::SleepThread(WaitTime);

    result = nn::svc::BreakDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    CheckDebugBreak(debugInfo);

    contextFlag = nn::svc::ContextFlag_Control;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    PrintContextAllRegisters(debugHandle, tid);

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    // セットあり
    nn::svc::SleepThread(WaitTime);

    result = nn::svc::BreakDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);
    CheckDebugBreak(debugInfo);

    contextFlag = nn::svc::ContextFlag_Control;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);


    context.pc += InstructionSize;

#if defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    nn::Bit32 beforeCpsr = context.cpsr;
    context.cpsr = ~context.cpsr;
#elif  defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    nn::Bit32 beforePstate = context.pstate;
    context.pstate = ~context.pstate;
#endif

    result = nn::svc::SetDebugThreadContext(debugHandle, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    PrintContextAllRegisters(debugHandle, tid);

    NN_LOG("+++ %s(line: %d) +++\n", __func__, __LINE__);
    PrintContextAllRegisters(debugHandle, tid);
#if defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    NN_LOG("before cpsr: 0x%x\n", beforeCpsr);
#elif  defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    NN_LOG("before pstate: 0x%x\n", beforePstate);
#endif

    // ここで、PC の位置にハードブレイクポイントを張る
    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

uintptr_t g_HeapAreaBegin;
uintptr_t g_HeapAreaEnd;
uintptr_t g_ReservedAreaBegin;
uintptr_t g_ReservedAreaEnd;
uintptr_t g_FreeAreaBegin;
uintptr_t g_FreeAreaEnd;
int32_t g_ProcessIdealCore;

extern "C" void nnMain()
{
    InitTestMemory();

    nn::Result result;

    int32_t idealCore;
    nn::Bit64 mask;

    NamedPortManager namedManager(DebugPortName, 1);
    nn::svc::Handle portHandle = namedManager.GetHandle();

    result = nn::svc::GetThreadCoreMask(&idealCore, &mask, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    NN_ASSERT_RESULT_SUCCESS(result);

    int32_t childCore = (idealCore + 1) % NumCore;
    uintptr_t begin = reinterpret_cast<uintptr_t>(BinDebugProcess_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinDebugProcess_end);
    TestLoader loader(BinDebugProcess_begin, end - begin);
    loader.SetProcessorId(childCore);

    uint32_t flag;
    MakeNo16Flag(&flag, 0x1);
    loader.SetAdditionalCapability(&flag, 1);

    nn::svc::Handle processHandle;
    loader.SpawnProcess(&processHandle);
    loader.StartProcess(processHandle);

    nn::Bit64 processId;
    result = nn::svc::GetProcessId(&processId, processHandle);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle debugHandle;
    result = nn::svc::DebugActiveProcess(&debugHandle, processId);
    NN_ASSERT_RESULT_SUCCESS(result);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoCreateThread createThreadInfo;
    nn::Bit64 tid;
    bool attached = false;
    for(int i = 0; i < 2 || !attached; )
    {
        result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
        ASSERT_RESULT_SUCCESS(result);
        if(debugInfo.event == nn::svc::DebugEvent_CreateThread)
        {
            createThreadInfo = debugInfo.info.createThread;
            tid = createThreadInfo.id;
            i++;
        }
        if(debugInfo.event == nn::svc::DebugEvent_Exception)
        {
            nn::svc::DebugInfoException except = debugInfo.info.exception;
            ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AttachBreak);
            attached = true;
        }
        if (attached)
        {
            result = ContinueDebugProcess(debugHandle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &portHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, portHandle);
    ASSERT_RESULT_SUCCESS(result);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    // 通常時
    CheckNormalState(debugHandle, tid);

    // SVC 中
    CheckSvcState(debugHandle, tid, serverSession);

    // ReturnFromException
    CheckReturnFromExceptionState(debugHandle, tid);

    // Break
    CheckBreakState(debugHandle, tid);

    // 例外中
    CheckExceptionState(debugHandle, tid);

    // ユーザーが書き換えられないことを確認
    CheckNoUpdate(debugHandle, tid);

    result = WaitAndGetDebugEvent(&debugInfo, debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    // cpsr が書き換えられたことによって、プロセスが正常終了しなくなるので、強制終了させる
    result = nn::svc::TerminateDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

