﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestLoader.h"
#include "util_TestIpc.h"
#include "test_TransferMemory_Server.h"

extern "C" void nnMain();

namespace {

#ifdef ENABLE_MAX_HANDLE_TEST
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));
#endif // ENABLE_MAX_HANDLE_TEST
char g_UnmapArea[0x1000] __attribute__((aligned(0x1000)));

} // namespace

#ifdef ENABLE_MAX_HANDLE_TEST
TEST(CreateTransferMemory, MaxHandleTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t size = 0x1000;
    nn::svc::MemoryInfo blockInfo;

    nn::svc::Handle handle;

    {
        ConsumeHandle maxHandle;

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        // TEST 126-37
        // ハンドル数が上限に達している状態で呼び出すことが出来ない
        result = nn::svc::CreateTransferMemory(
                &handle, addr, size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());

        // 126-41
        // 呼び出しに失敗した後、addr/size のメモリ状態が元に戻っている
        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 126-38
    // ハンドル数が上限よりも少なくなると、呼び出しに成功する
    result = nn::svc::CreateTransferMemory(
            &handle, addr, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);
}
#endif // ENABLE_MAX_HANDLE_TEST

extern char BinTransfer_begin[];
extern char BinTransfer_end[];

// このテストは TransferMemory がアンマップされずに相手プロセスが終了したときにマップ状態が残ることをテストする
// 繰り返すことには対応していない
TEST(CreateTransferMemory, NotUnmapTest)
{
    nn::Result result;
    nn::svc::MemoryInfo info;

    uintptr_t addr = reinterpret_cast<uintptr_t>(g_UnmapArea);
    size_t size = sizeof(g_UnmapArea);

    // 以前実行されているとメモリ属性に Locked が付いている
    GetMemoryInfo(&info, addr);
    if (info.attribute != nn::svc::MemoryAttribute_Locked)
    {
        uintptr_t begin = reinterpret_cast<uintptr_t>(BinTransfer_begin);
        uintptr_t end = reinterpret_cast<uintptr_t>(BinTransfer_end);

        NamedPortManager manager(TransferPortName, 1);
        nn::svc::Handle portHandle = manager.GetHandle();

        nn::svc::Handle processHandle;
        TestLoader loader(BinTransfer_begin, end - begin);
        loader.SpawnProcess(&processHandle);
        loader.StartProcess(processHandle);
        AutoHandleClose processCloser(processHandle);

        nn::svc::ResetSignal(processHandle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &portHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle serverSession;
        result = nn::svc::AcceptSession(&serverSession, portHandle);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);

        nn::svc::Handle transferHandle;
        nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_None;
        result = nn::svc::CreateTransferMemory(&transferHandle, addr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        SendMoveHandleFromServer(transferHandle, serverSession);

        GetMemoryInfo(&info, addr);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(info.permission == perm);
        ASSERT_TRUE(info.attribute == nn::svc::MemoryAttribute_Locked);

        nn::svc::Handle writableEvent;
        nn::svc::Handle readableEvent;
        result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
        ASSERT_RESULT_SUCCESS(result);

        SendMoveHandleFromServer(writableEvent, serverSession);

        result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&info, addr);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(info.permission == perm);
        ASSERT_TRUE(info.attribute == nn::svc::MemoryAttribute_Locked);

        result = nn::svc::CloseHandle(readableEvent);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        int64_t state;
        result = nn::svc::GetProcessInfo(
                &state, processHandle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(static_cast<nn::svc::ProcessState>(state) != nn::svc::ProcessState_WaitAttach);

        GetMemoryInfo(&info, addr);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_CodeData);
        ASSERT_TRUE(info.permission == perm);
        ASSERT_TRUE(info.attribute == nn::svc::MemoryAttribute_Locked);
    }
}

