﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_Version.h>
#include <cstring>

extern "C" void nnMain();

namespace {

#ifdef INVALID_POINTER_TEST
const char ConstVar[sizeof(nn::svc::Handle)] = {0};
#endif

} // namespace

TEST(CreateProcess, pOutTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle* pHandle;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    // TEST 121-1
    // MemoryPermission_ReadWrite を受け付ける
    // local 変数
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // Heap
    {
        TestHeap heap(HeapAlign);
        pHandle = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
        result = nn::svc::CreateProcess(pHandle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(*pHandle);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef INVALID_POINTER_TEST
    // TEST 121-86
    // NULL アドレスを受け付けない
    pHandle = NULL;
    result = nn::svc::CreateProcess(pHandle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif

#ifdef INVALID_POINTER_TEST
    // TEST 121-2
    // MemoryPermission_None は受け付けない
    pHandle = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
    result = nn::svc::CreateProcess(pHandle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif

#ifdef INVALID_POINTER_TEST
    // TEST 121-3
    // MemoryPermission_Read は受け付けない
    uintptr_t addr = reinterpret_cast<uintptr_t>(ConstVar);
    pHandle = reinterpret_cast<nn::svc::Handle*>(addr);
    result = nn::svc::CreateProcess(pHandle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif

#ifdef INVALID_POINTER_TEST
    // TEST 121-4
    // MemoryPermission_ReadExecute は受け付けない
    pHandle = reinterpret_cast<nn::svc::Handle*>(nnMain);
    result = nn::svc::CreateProcess(pHandle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif

}

TEST(CreateProcess, ParamNameTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);
    ::std::memset(param.name, 0, sizeof(param.name));

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    // TEST 121-7
    // param.name は 0 - 12 文字受け付ける
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    for (uint8_t i = 0; i < 12; i++)
    {
        param.name[i] = '0' + (i % 10);
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(CreateProcess, ParamVersionTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    nn::Bit32 validVersion[] = {0, 1, 0x1000, 0xFFFF, 0x10000, 0xFFFFFFFF};

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    // TEST 121-9
    // Bit32 の値を受け付ける
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(validVersion) / sizeof(nn::Bit32)); i++)
    {
        param.version = validVersion[i];
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(CreateProcess, ParamProgramIdTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    nn::Bit64 validProgramId[] = {
        0, 1, 0x1000, 0xFFFF, 0x10000, 0xFFFFFFFF, 0x100000000, 0xFFFFFFFFFFFFFFFF};

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    // TEST 121-10
    // Bit64 の値を受け付ける
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(validProgramId) / sizeof(nn::Bit64)); i++)
    {
        param.version = validProgramId[i];
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(CreateProcess, ParamMemoryAddressTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    // TEST 121-12
    // 2MB にアライメントされていないと失敗する
    for (int32_t i = 1; i < 0x200000; i++)
    {
        param.memoryAddress = i;
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
}

TEST(CreateProcess, ParamMemoryNumPageTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

#ifdef TEST_BY_HAND
    // TEST 121-13
    // 0 を受け付けない
    param.memoryNumPages = 0;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
#endif // TEST_BY_HAND

    // TEST 121-87
    // int32_t の値を受け付ける
    // SmallRegionBegin の領域でテスト
    param.memoryAddress = SmallRegionBegin + 0x00200000;
    int32_t validNum[] = {1, 0x100, 0xFFF, 0x1000};
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(validNum) / sizeof(int32_t)); i++)
    {
        param.memoryNumPages = validNum[i];
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-88
    // 負の値を受け付けない
    param.memoryNumPages = -1;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidSize() ||
            result <= nn::svc::ResultInvalidRegion());

    param.memoryNumPages = 0xFFFF0000;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidSize() ||
            result <= nn::svc::ResultInvalidRegion());
}

TEST(CreateProcess, ParamMemoryAreaTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);
    param.memoryNumPages = 1;

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    // TEST 121-28
    // MemoryState_Inaccessible の領域は取得できない
    // MemoryState_Inaccessible がないアーキテクチャもある
    if (ProcessEnd + 1 != 0)
    {
        param.memoryAddress = ProcessEnd + 1;
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());
    }

    // TEST 121-14
    // MemoryState_Free の領域を指し示していると成功する
    // TEST 121-29
    // MemoryRegion_Small に含まれていると成功する
    param.memoryAddress = 0x200000;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

#ifdef ENABLE_ASLR
    // TEST 121-30
    // MemoryRegion_Small に含まれていないと失敗する
    param.memoryAddress = LargeRegionBegin;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
#endif

    // TEST 121-31
    // オーバーフローする組み合わせを指定できない
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
    param.memoryAddress = static_cast<uint32_t>(-0x200000);
    param.memoryNumPages = 0x201;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());
#endif

#if defined NN_BUILD_CONFIG_CPU_SVC_64
    param.flags = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
    param.memoryAddress = static_cast<uint64_t>(-0x200000ull);
    param.memoryNumPages = 0x201;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());
#endif

#ifdef ENABLE_ASLR
#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
    // 64 bit の場合、オーバーフローするわけではないが、
    // MemoryRegion_Large 領域を指し示すことになり、失敗するはず
    param.memoryAddress = 0x10000000;
    param.memoryNumPages = 0xF0000;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() || result <= nn::svc::ResultInvalidRegion());
#endif // defined NN_BUILD_CONFIG_CPU_ARM_V8A
#endif
}

TEST(CreateProcess, ParamFlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
    // TEST 121-32
    // CreateProcessParameterFlag_64Bit を指定できる
    param.flags = nn::svc::CreateProcessParameterFlag_64Bit;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-33
    // CreateProcessParameterFlag_AddressSpace64Bit を
    // CreateProcessParameterFlag_64Bit と同時に使用できる
    param.flags = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
    if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
    {
        param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
    }
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
#else
    // TEST 121-34
    // 32 bit 環境で、CreateProcessParameter_FlagAddressSpace64Bit を指定できない
    param.flags = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
    if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
    {
        param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
    }
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
#endif

    // TEST 121-35
    // CreateProcessParameterFlag_AddressSpace64Bit を指定しないで
    // CreateProcessParameterFlag_64Bit を指定することが出来ない
    param.flags = nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
    if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
    {
        param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
    }

    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());

    // TEST 121-36
    // CreateProcessParameterFlag_EnableJitDebug を指定できる
    param.flags = nn::svc::CreateProcessParameterFlag_EnableJitDebug;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-37
    // CreateProcessParameterFlag_EnableAslr を指定できる
    param.flags = nn::svc::CreateProcessParameterFlag_EnableAslr;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-89
    // 複数のフラグを一度に指定できる
    param.flags = 0;
    param.flags =
#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
        nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit
        | nn::svc::CreateProcessParameterFlag_EnableJitDebug
        | nn::svc::CreateProcessParameterFlag_EnableAslr;
#else
    nn::svc::CreateProcessParameterFlag_EnableJitDebug
        | nn::svc::CreateProcessParameterFlag_EnableAslr;
#endif // defined NN_BUILD_CONFIG_CPU_ARM_V8A
    if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
    {
        param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
    }
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
#if defined NN_BUILD_CONFIG_CPU_SVC_64
#if 0
    // TEST 121-38
    // CreateProcessParameterFlag_AddressSpace64Bit が指定されていると、
    // 64 bit のアドレスが利用できる
    SetDefaultParam(&param);
    param.flags = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
    param.memoryAddress  = 0x100000000;
    param.memoryNumPages = 0x00001000;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
#endif

    // TEST 121-39
    // CreateProcessParameterFlag_AddressSpace64Bit が指定されていないと、
    // 64 bit のアドレスを指定できない
    // Large 領域を指定することになるが、念のため
    SetDefaultParam(&param);
    param.flags = 0;
    param.memoryAddress  = 0x100000000;
    param.memoryNumPages = 0x00001000;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());
#endif // NN_BUILD_CONFIG_CPU_SVC_64
#if 0
    // TEST 121-102
    // CreateProcessParameterFlag_AddressSpace64Bit が指定されていると、
    // 64 bit のmemoryAddress/memoryNumPages の組み合わせが利用できる
    SetDefaultParam(&param);
    param.flags = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
    param.memoryAddress  = 0xffe00000;
    param.memoryNumPages = 0x00000201;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
#endif

    // TEST 121-40
    // CreateProcessParameterFlag_AddressSpace64Bit が指定されていないと、
    // 64 bit のアドレスを指定できない
    // Large 領域を指定することになるが、念のため
    param.flags = 0;
    param.memoryAddress  = 0xffe00000;
    param.memoryNumPages = 0x00000201;
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());
#endif // defined NN_BUILD_CONFIG_CPU_ARM_V8A

    // TEST 121-91
    // CreateProcessMarameterFlag 以外の値を受け付けない
    SetDefaultParam(&param);
    param.flags = ~(  nn::svc::CreateProcessParameterFlag_64Bit
            | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit
            | nn::svc::CreateProcessParameterFlag_EnableJitDebug
            | nn::svc::CreateProcessParameterFlag_EnableAslr);
    if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
    {
        param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
    }
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidEnum());
} // NOLINT (readability/fn_size)

TEST(CreateProcess, FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    int32_t maxFlagNum = heap.GetSize() / sizeof(nn::Bit32);
    SetDefaultCapability(flags, maxFlagNum);

    // TEST 121-41
    // NULL を受け付ける
    // 廃止

    // TEST 121-42
    // MemoryPermission_ReadWrite を受け付ける
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-43
    // MemoryPermission_None を受け付けない
    {
        TestMemoryPermission perm(heap.GetAddress(), 0x1000, nn::svc::MemoryPermission_None);
        TestMemoryBlock blockInfo(heap.GetAddress());
        blockInfo.CheckPermission(nn::svc::MemoryPermission_None);
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_FAILURE(result);
    }

    // TEST 121-44
    // MemoryPermission_Read を受け付ける
    {
        TestMemoryPermission perm(heap.GetAddress(), 0x1000, nn::svc::MemoryPermission_Read);
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-45
    // MemoryPermission_ReadExecute を受け付ける
    // flag を書き込めないので廃止

    // TEST 121-46
    // KernelData を受け付けない
    // flag を書き込めないので廃止

    // TEST 121-47
    // KernelCode を受け付けない
    // flag を書き込めないので廃止


    // TEST 121-66
    // 許可されている No. 以外のフラグを利用することが出来ない
    {
        SetDefaultParam(&param);
        int32_t validNo[] = {3, 4, 6, 7, 11, 14, 15, 16};
        int32_t validNum = sizeof(validNo) / sizeof(int32_t);
        for (int32_t i = 0; i < 32; i++)
        {
            int32_t j = 0;
            for (; j < validNum; j++)
            {
                if (i == validNo[j])
                {
                    break;
                }
            }
            if (j != validNum)
            {
                continue;
            }

            flags[0] = CapabilitySignature(i);
            result = nn::svc::CreateProcess(&handle, param, flags, 1);
            ASSERT_RESULT_FAILURE(result);
        }
    }

    // TEST 121-100
    // flags を複数種類指定できる
    {
        int32_t flagNum = 0;

#ifdef SUPPORT_CAPABILITY
        // 現状ではすべて許可されいる
#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
        || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
        || defined( NN_BUILD_CONFIG_HARDWARE_NX )
        // 許可する割り込み番号を指定
        uint16_t names[] = {148, 149, 150, 151};
        MakeNo11Flag(flags, &flagNum, maxFlagNum, names, sizeof(names) / sizeof(uint16_t));
        ASSERT_TRUE(flagNum == 2);
        ASSERT_TRUE(flagNum < maxFlagNum);
#endif // defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
#endif // SUPPORT_CAPABILITY

        // スレッドの設定
        MakeNo3Flag(&flags[flagNum], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        flagNum++;
        ASSERT_TRUE(flagNum < maxFlagNum);

        // 一般カテゴリのSVC をすべて許可する
        int svcNum = 40;
        int32_t svcIds[svcNum];
        for (int32_t i = 0; i < svcNum; i++)
        {
            svcIds[i] = i + 1;
        }
        int32_t tmpNum;
        MakeNo4Flag(&flags[flagNum], &tmpNum, maxFlagNum - flagNum, svcIds, svcNum);
        ASSERT_TRUE(flagNum + tmpNum < maxFlagNum);
        flagNum += tmpNum;

        // 物理メモリをプロセスにマップ
        uint64_t ioAddress = GetIoAddress(0x2000);
        ASSERT_TRUE(ioAddress != 0);
        {
            MakeNo6Flag(&flags[flagNum], maxFlagNum - flagNum, ioAddress, 0x1000, false, true);
            flagNum += 2;
            ASSERT_TRUE(flagNum < maxFlagNum);

            // 物理メモリをプロセスにマップ (for 1page IO)
            MakeNo7Flag(&flags[flagNum], ioAddress + 0x1000);
            flagNum++;
            ASSERT_TRUE(flagNum < maxFlagNum);
        }

        // カーネルバージョン
        MakeNo14Flag(&flags[flagNum], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        flagNum++;
        ASSERT_TRUE(flagNum < maxFlagNum);

        // 使用可能ハンドル数
        MakeNo15Flag(&flags[flagNum], (1 << 10) - 1);
        flagNum++;
        ASSERT_TRUE(flagNum < maxFlagNum);

        // 雑多なフラグ
        MakeNo16Flag(&flags[flagNum], 0x3);
        flagNum++;

        result = nn::svc::CreateProcess(&handle, param, flags, flagNum);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}  // NOLINT (readability/fn_size)

TEST(CreateProcess, No3FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    // TEST 121-130
    // No.3 のフラグを指定できる
    {
        MakeNo3Flag(
                flags, TestLowestThreadPriority, TestHighestThreadPriority, 0, 2);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 2);
        ASSERT_RESULT_SUCCESS(result);

        nn::Bit64 coreMask;
        result = nn::svc::GetInfo(&coreMask, nn::svc::InfoType_CoreMask, handle, 0);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(coreMask != 0);
        ASSERT_TRUE(coreMask == ((1ull << 3) - 1));

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-131
    // No.3 のフラグで、システムに存在しないコア番号が指定されると失敗する
    {
        MakeNo3Flag(
                flags, TestLowestThreadPriority, TestHighestThreadPriority, NumCore, NumCore);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());
    }
}

TEST(CreateProcess, No4FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    // TEST 121-50
    // No.4 のフラグを指定できる
    {
        MakeNo4Flag(flags, 1); // SetHeapSize
        MakeNo3Flag(&flags[1], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[2], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef SUPPORT_CAPABILITY
    // TEST 121-51
    // No.4 のフラグに存在しないシステムコール番号を指定できない
    {
        MakeNo4Flag(flags, 130);
        MakeNo3Flag(&flags[1], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[2], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());
    }
    {
        MakeNo4Flag(&flags[0], NN_SVC_ID_EXIT_PROCESS);
        MakeNo4Flag(&flags[1], 130);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());
    }
#endif
}

TEST(CreateProcess, No6FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    int32_t maxFlagNum = heap.GetSize() / sizeof(nn::Bit32);

    uint64_t ioAddress = GetIoAddress(0x1000);

    ASSERT_TRUE(ioAddress != 0);

    // TEST 121-52
    // No.6 のフラグを指定できる
    // TEST 121-95
    // No.6 のフラグで、MemoryPermission_Read のマッピングを指定できる
    // TEST 121-98
    // No.6 のフラグで、MemoryState_Io のマッピングを指定できる
    {
        MakeNo6Flag(flags, maxFlagNum, ioAddress, 0x1000, false, true);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef ENABLE_CREATE_PROCESS_STATIC_MAP
    // TEST 121-97
    // No.6 のフラグで、MemoryState_Static のマッピングを指定できる
    {
        MakeNo6Flag(flags, maxFlagNum, ioAddress, 0x1000, false, false);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
#endif
    // TEST 121-96
    // No.6 のフラグで、MemoryPermission_ReadWrite のマッピングを指定できる
    // TEST 121-98
    // No.6 のフラグで、MemoryState_Io のマッピングを指定できる
    {
        MakeNo6Flag(flags, maxFlagNum, ioAddress, 0x1000, true, true);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-53
    // No.6 のフラグを指定する際に、２つ組み合わせてセットされていないと失敗する
    {
        MakeNo6Flag(flags, maxFlagNum, ioAddress, 0x1000, false, true);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);

        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }

    uint64_t highAddress = (PhysMemoryEnd >= 0x100000000) ? PhysMemoryEnd : 0x100000000;
    ASSERT_TRUE(highAddress + 0x1000 > 0x100000000);
    // TEST 121-54
    // CreateProcessParameterFlag_AddressSapce64Bit が指定されていない状態でも、No.6 で 64bit アドレスになるデータを指定することが出来る
    {
        SetDefaultParam(&param);
        param.flags = 0; // フラグは何も指定しない
        MakeNo6Flag(flags, maxFlagNum, highAddress, 0x1000, false, true);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        result = nn::svc::CreateProcess(&handle, param, flags, 4);
#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
#else
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
#endif // defined NN_BUILD_CONFIG_CPU_ARM_V8A
    }

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
    // TEST 121-55
    // CreateProcessParameterFlag_AddressSapce64Bit が指定されていると、No.6 で 64bit アドレスになるデータを指定することが出来る
    {
        SetDefaultParam(&param);
        param.flags = nn::svc::CreateProcessParameterFlag_64Bit
            | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
        MakeNo6Flag(flags, maxFlagNum, highAddress, 0x1000, false, true);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
#endif // defined NN_BUILD_CONFIG_CPU_ARM_V8A

    // TEST 121-56
    // No.6 のアドレスに 0 を指定することが出来る
    ASSERT_TRUE(0x1000 <= PhysMemoryBegin);
    {
        SetDefaultParam(&param);
        MakeNo6Flag(flags, maxFlagNum, 0, 0x1000, false, true);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-92
    // No.6 のサイズに 0 を指定することが出来ない
    {
        SetDefaultParam(&param);
        MakeNo6Flag(flags, maxFlagNum, ioAddress, 0, false, true);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

#if ENABLE_CREATE_PROCESS_STATIC_MAP
    // TEST 121-57
    // No.6 で１度指定したアドレスに、重複してマッピングすることが出来る
    {
        SetDefaultParam(&param);
        MakeNo6Flag(flags, maxFlagNum, ioAddress, 0x1000, false, true);
        MakeNo6Flag(&flags[2], maxFlagNum - 2, ioAddress, 0x1000, true, false);
        MakeNo3Flag(&flags[4], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[5], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 6);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
    {
        SetDefaultParam(&param);
        MakeNo6Flag(flags, maxFlagNum, ioAddress, 0x1000, false, false);
        MakeNo6Flag(&flags[2], maxFlagNum - 2, ioAddress, 0x1000, false, true);
        MakeNo3Flag(&flags[4], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[5], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 6);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
#endif
    {
        SetDefaultParam(&param);
        MakeNo6Flag(flags, maxFlagNum, ioAddress, 0x1000, false, true);
        MakeNo6Flag(&flags[2], maxFlagNum - 2, ioAddress, 0x1000, false, true);
        MakeNo3Flag(&flags[4], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[5], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

        result = nn::svc::CreateProcess(&handle, param, flags, 6);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-118
    // No.6 に指定する領域が大きすぎると失敗する
    {
        uint64_t bigSize = (SmallRegionEnd - SmallRegionBegin + 0x1000 + 0xfff) & ~(0xfff);
        uint64_t bigAddress = (PhysMemoryBegin >= bigSize) ? 0 : PhysMemoryEnd;

        if (bigAddress > 0)
        {
            ASSERT_TRUE(0x1000000000ULL - PhysMemoryEnd > bigSize);
        }

        {
            SetDefaultParam(&param);
            MakeNo6Flag(flags, maxFlagNum, bigAddress, bigSize, false, true);
            MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
            MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);

            result = nn::svc::CreateProcess(&handle, param, flags, 4);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfMemory());
        }
    }

    // TEST 121-119
    // No.6 にオーバーフローする組み合わせを指定できない
    // TEST 121-122
    // No.6 に 36bit 以上アドレススペースになる IO 領域を許可しない
    {
        SetDefaultParam(&param);

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
        param.flags = nn::svc::CreateProcessParameterFlag_64Bit
            | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
        MakeNo6Flag(flags, maxFlagNum, 0xffffff000, 0x2000, false, true);
#else
        MakeNo6Flag(flags, maxFlagNum, 0xfffff000, 0x2000, false, true);
#endif
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // 物理メモリのアドレスを指定できない
    {
        SetDefaultParam(&param);
        uint64_t physAddr = PhysMemoryBegin;
        MakeNo6Flag(flags, maxFlagNum, physAddr, 0x1000, false, true);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
    {
        SetDefaultParam(&param);
#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
        uint64_t physAddr = PhysMemoryEnd - 0x1000;
#else
        uintptr_t tmpPhysAddr = static_cast<uintptr_t>(PhysMemoryEnd - 0x1000);
        uint64_t physAddr = static_cast<uint64_t>(tmpPhysAddr);
#endif

        MakeNo6Flag(flags, maxFlagNum, physAddr, 0x1000, false, true);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        MakeNo14Flag(&flags[3], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
} // NOLINT (readability/fn_size)

TEST(CreateProcess, No7FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    uint64_t ioAddress = GetIoAddress(0x1000);
    ASSERT_TRUE(ioAddress != 0);

    // TEST 121-58
    // No.7 のフラグを指定できる
    {
        SetDefaultParam(&param);
        MakeNo7Flag(flags, ioAddress);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
    uint64_t highAddress = (PhysMemoryEnd >= 0x100000000) ? PhysMemoryEnd : 0x100000000;
    ASSERT_TRUE(highAddress + 0x1000 > 0x100000000);
    // TEST 121-59
    // CreateProcessParameterFlag_AddressSapce64Bit が指定されていない状態でも、No.7 で 64bit アドレスになるデータを指定することが出来る
    {
        SetDefaultParam(&param);
        param.flags = 0;
        MakeNo7Flag(flags, highAddress);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-60
    // CreateProcessParameterFlag_AddressSapce64Bit が指定されていると、No.7 で 64bit アドレスになるデータを指定することが出来る
    {
        SetDefaultParam(&param);
        param.flags = nn::svc::CreateProcessParameterFlag_64Bit
            | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
        MakeNo7Flag(flags, highAddress);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
#endif // defined NN_BUILD_CONFIG_CPU_ARM_V8A

    // TEST 121-61
    // No.7 のアドレスに 0 を指定することが出来る
    ASSERT_TRUE(0x1000 < PhysMemoryBegin);
    {
        SetDefaultParam(&param);
        MakeNo7Flag(flags, 0);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // No.7 のアドレスに物理メモリの領域を指定することが出来ない
    {
        SetDefaultParam(&param);
        uint64_t physAddr = PhysMemoryBegin;
        MakeNo7Flag(flags, physAddr);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
}

#ifdef SUPPORT_CAPABILITY
#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_NX )
TEST(CreateProcess, No11FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    int32_t maxFlagNum = heap.GetSize() / sizeof(nn::Bit32);

    // TEST 121-48
    // No.11 のフラグを指定できる
    {
        int32_t flagNum = 0;
#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
        || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
        || defined( NN_BUILD_CONFIG_HARDWARE_NX )
        uint16_t names[] = {148, 149, 150, 151};
#endif

        MakeNo11Flag(flags, &flagNum, maxFlagNum, names, sizeof(names) / sizeof(uint16_t));
        ASSERT_TRUE(flagNum == 2);
        MakeNo14Flag(&flags[2], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[3], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);

        result = nn::svc::CreateProcess(&handle, param, flags, flagNum + 2);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-49
    // No.11 のフラグに、存在しない割り込み番号を指定できない
    {
        int32_t flagNum = 0;
        uint16_t names[] = {500};
        MakeNo11Flag(flags, &flagNum, maxFlagNum, names, sizeof(names) / sizeof(uint16_t));
        ASSERT_TRUE(flagNum == 1);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);

        result = nn::svc::CreateProcess(&handle, param, flags, flagNum + 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());
    }

    {
        int32_t flagNum = 0;
#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
        || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
        || defined( NN_BUILD_CONFIG_HARDWARE_NX )
        uint16_t names[] = {148, 149, 150, 500};
#endif

        MakeNo11Flag(flags, &flagNum, maxFlagNum, names, sizeof(names) / sizeof(uint16_t));
        ASSERT_TRUE(flagNum == 2);
        MakeNo14Flag(&flags[2], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[3], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);

        result = nn::svc::CreateProcess(&handle, param, flags, flagNum + 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());
    }

    // TEST 121-128
    // No.11 のフラグに、1023 を指定することが出来る
    {
        int32_t flagNum = 0;
        uint16_t names[] = {1023, 1023};
        MakeNo11Flag(flags, &flagNum, maxFlagNum, names, sizeof(names) / sizeof(uint16_t));
        ASSERT_TRUE(flagNum == 1);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);

        result = nn::svc::CreateProcess(&handle, param, flags, flagNum + 2);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}
#endif // defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
#endif // SUPPORT_CAPABILITY

TEST(CreateProcess, No13FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    // No.13 のフラグを指定できる
    for (uint32_t programType = 0; programType < (1u << No13ProgramType::Width); programType++)
    {
        {
            int index = 0;
            uint32_t reserved = 0;
            SetDefaultParam(&param);
            MakeNo13Flag(&flags[index++], programType, reserved);
            MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
            MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
            result = nn::svc::CreateProcess(&handle, param, flags, index);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
        {
            int index = 0;
            uint32_t reserved = 1;
            SetDefaultParam(&param);
            MakeNo13Flag(&flags[index++], programType, reserved);
            MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
            MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
            result = nn::svc::CreateProcess(&handle, param, flags, index);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultReservedIsUsed());
        }
        {
            int index = 0;
            uint32_t reserved = (1u << No13Reserved::Width) - 1;
            SetDefaultParam(&param);
            MakeNo13Flag(&flags[index++], programType, reserved);
            MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
            MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
            result = nn::svc::CreateProcess(&handle, param, flags, index);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultReservedIsUsed());
        }

        {
            int index = 0;
            uint32_t reserved = 0;
            SetDefaultParam(&param);
            MakeNo13Flag(&flags[index++], programType, reserved);
            MakeNo13Flag(&flags[index++], programType, reserved);
            MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
            MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
            result = nn::svc::CreateProcess(&handle, param, flags, index);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
        }
    }
}


TEST(CreateProcess, No14FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    // No.14 のフラグを指定できる
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo14Flag(&flags[index++], NN_SVC_REQUIRED_VERSION_MAJOR - 1, NN_SVC_REQUIRED_VERSION_MINOR);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }
    if (NN_SVC_REQUIRED_VERSION_MINOR != 0)
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo14Flag(&flags[index++], NN_SVC_REQUIRED_VERSION_MAJOR, 0);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo14Flag(&flags[index++], NN_SVC_REQUIRED_VERSION_MAJOR, NN_SVC_REQUIRED_VERSION_MINOR);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (NN_SVC_REQUIRED_VERSION_MAJOR < NN_SVC_VERSION_MAJOR)
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, 0);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR + 1);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR + 1, 0);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo14Flag(&flags[index++], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }
    {
        int index = 0;
        SetDefaultParam(&param);
        MakeNo3Flag(&flags[index++], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, index);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
    }
}

TEST(CreateProcess, No15FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    // TEST 121-62
    // No.15 のフラグを指定できる
    {
        SetDefaultParam(&param);
        MakeNo15Flag(flags, 0);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        MakeNo15Flag(flags, (1 << 10) - 1);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef SUPPORT_CAPABILITY
    // TEST 121-63
    // No.15 に 10 bit 目以上の位置にビットを立てることが出来ない
    {
        SetDefaultParam(&param);
        MakeNo15Flag(flags, 1 << 10);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultReservedIsUsed());

        MakeNo15Flag(flags, 0xFFFF);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultReservedIsUsed());
    }
#endif // SUPPORT_CAPABILITY
}

TEST(CreateProcess, No16FlagTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    // TEST 121-64
    // No.16 のフラグを指定できる
    {
        SetDefaultParam(&param);
        MakeNo16Flag(flags, 1 << 0);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        MakeNo16Flag(flags, 1 << 1);
        MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
        result = nn::svc::CreateProcess(&handle, param, flags, 3);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 121-129
    // No.16 のフラグで、予約領域にフラグが立っていると失敗する
#ifdef SUPPORT_CAPABILITY
    {
        SetDefaultParam(&param);
        // 予約領域は13 bit
        for (int i = 2; i < 15; i++)
        {
            MakeNo16Flag(flags, (1 << i));
            MakeNo14Flag(&flags[1], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
            MakeNo3Flag(&flags[2], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
            result = nn::svc::CreateProcess(&handle, param, flags, 3);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultReservedIsUsed());
        }
    }
#endif
}

TEST(CreateProcess, FlagsNumTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    // TEST 121-68
    // 0 を指定できる
    // 廃止

    // TEST 121-69
    // 33 以上のフラグを指定できる
    MakeNo14Flag(&flags[0], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
    MakeNo3Flag(&flags[1], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);
    for (int32_t i = 2; i < 33; i++)
    {
        flags[i] = 0xFFFFFFFF;
    }
    result = nn::svc::CreateProcess(&handle, param, flags, 33);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_CAPABILITY
    // TEST 121-70
    // 負の値を指定できない
    {
        result = nn::svc::CreateProcess(&handle, param, flags, -1);
        ASSERT_RESULT_FAILURE(result);

        result = nn::svc::CreateProcess(&handle, param, flags, 0xFFFF0000);
        ASSERT_RESULT_FAILURE(result);
    }
#endif // SUPPORT_CAPABILITY

}

TEST(CreateProcess, FlagsMemoryAreaTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);
    size_t pageSize = 0x1000;

    TestHeap heap(HeapAlign);

    // TEST 121-71
    // メモリ状態が一様でないと失敗する(前方)
    {
        nn::Bit32* flags =
            reinterpret_cast<nn::Bit32*>(heap.GetAddress() + pageSize - sizeof(nn::Bit32));

        MakeNo4Flag(&flags[0], 1);
        MakeNo4Flag(&flags[1], 2);
        MakeNo14Flag(&flags[2], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[3], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);

        TestMemoryPermission perm(heap.GetAddress(), pageSize, nn::svc::MemoryPermission_None);

        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
    }

    // TEST 121-72
    // メモリ状態が一様でないと失敗する(後方)
    {
        nn::Bit32* flags =
            reinterpret_cast<nn::Bit32*>(heap.GetAddress() + pageSize - sizeof(nn::Bit32));

        MakeNo4Flag(&flags[0], 1);
        MakeNo4Flag(&flags[1], 2);
        MakeNo14Flag(&flags[2], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        MakeNo3Flag(&flags[3], TestLowestThreadPriority, TestHighestThreadPriority, 0, NumCore - 1);

        TestMemoryPermission perm(
                heap.GetAddress() + pageSize, pageSize, nn::svc::MemoryPermission_None);

        result = nn::svc::CreateProcess(&handle, param, flags, 4);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
    }

    // TEST 121-73
    // flags に NULL を入れているときに、flagsNum に 1 以上の数を入れると失敗する
    {
        result = nn::svc::CreateProcess(&handle, param, NULL, 1);
        ASSERT_RESULT_FAILURE(result);
    }
}
