﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern nn::Result ClientLightSession(nn::svc::Handle handle);
extern nn::Result ServerLightSession(nn::svc::Handle handle);
extern "C" void nnMain();

namespace {

const size_t StackSize = 0x1000;
#ifdef INVALID_POINTER_TEST
const char ConstVar[sizeof(nn::svc::Handle)] = { 0 };
#endif

char g_Stack[2][StackSize] __attribute__((aligned(0x1000)));

void NormalClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;

    nn::Result result;
    nn::svc::Handle* clientPort = reinterpret_cast<nn::svc::Handle*>(arg);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToPort(&clientSession, *clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sessionCloser(clientSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void NormalServerThread(uintptr_t arg)
{
    AutoThreadExit autoExit;

    nn::Result result;
    nn::svc::Handle* serverPort = reinterpret_cast<nn::svc::Handle*>(arg);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, *serverPort);
    AutoHandleClose sessionCloser(serverSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void LightClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;

    nn::Result result;
    nn::svc::Handle* clientPort = reinterpret_cast<nn::svc::Handle*>(arg);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToPort(&clientSession, *clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sessionCloser(clientSession);

    result = ClientLightSession(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void LightServerThread(uintptr_t arg)
{
    AutoThreadExit autoExit;

    nn::Result result;
    nn::svc::Handle* serverPort = reinterpret_cast<nn::svc::Handle*>(arg);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, *serverPort);
    AutoHandleClose sessionCloser(serverSession);

    result = ServerLightSession(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}

void CheckIpc(nn::svc::Handle serverPort, nn::svc::Handle clientPort, bool isLight)
{
    nn::Result result;

    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    if (isLight)
    {
        pc[0] = reinterpret_cast<uintptr_t>(LightServerThread);
        pc[1] = reinterpret_cast<uintptr_t>(LightClientThread);
    }
    else
    {
        pc[0] = reinterpret_cast<uintptr_t>(NormalServerThread);
        pc[1] = reinterpret_cast<uintptr_t>(NormalClientThread);
    }


    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + StackSize;
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + StackSize;

    param[0] = reinterpret_cast<uintptr_t>(&serverPort);
    param[1] = reinterpret_cast<uintptr_t>(&clientPort);

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    serverThread.Wait();
    clientThread.Wait();
}

} // namespace

// TEST 112-1
// pOutServer は ReadWrite の領域を指定すると成功する
// TEST 112-8
// pOutClient は ReadWrite の領域を指定すると成功する
// TEST 112-18
// true を指定すると、Light セッション用のポートを作成する
// TEST 112-19
// false を指定すると、Normal セッション用のポートを作成する
TEST(CreatePort, NormalCase)
{
    TestPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;

    TestHeap heap(sizeof(nn::svc::Handle) * 4);

    // Normal Session
    {
        // ローカル変数
        {
            result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sPortCloser(serverPort);
            AutoHandleClose cPortCloser(clientPort);

            CheckIpc(serverPort, clientPort, false);
        }
        // ヒープ
        {
            nn::svc::Handle* handles = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
            result = nn::svc::CreatePort(&handles[0], &handles[1], 1, false, 0);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sPortCloser(handles[0]);
            AutoHandleClose cPortCloser(handles[1]);

            CheckIpc(handles[0], handles[1], false);
        }
    }

    // Light Session
    {
        // ローカル変数
        {
            result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sPortCloser(serverPort);
            AutoHandleClose cPortCloser(clientPort);

            CheckIpc(serverPort, clientPort, true);
        }

        // ヒープ
        {
            nn::svc::Handle* handles = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
            result = nn::svc::CreatePort(&handles[2], &handles[3], 1, true, 0);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sPortCloser(handles[2]);
            AutoHandleClose cPortCloser(handles[3]);

            CheckIpc(handles[2], handles[3], true);
        }
    }
}

#ifdef INVALID_POINTER_TEST
TEST(CreatePort, pOutServerTest)
{
    TestPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle* serverPort;
    nn::svc::Handle clientPort;

#ifdef INVALID_POINTER_TEST
    // TEST 112-2
    // NULL の領域を指定すると失敗する
    serverPort = NULL;
    result = nn::svc::CreatePort(serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 112-3
    // MemoryPermission_None の領域を指定すると失敗する
    serverPort = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
    result = nn::svc::CreatePort(serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 112-4
    // MemoryPermission_Read の領域を指定すると失敗する
    uintptr_t tmpAddr = reinterpret_cast<uintptr_t>(&ConstVar);
    serverPort = reinterpret_cast<nn::svc::Handle*>(tmpAddr);
    result = nn::svc::CreatePort(serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 112-5
    // MemoryPermission_ReadExecute の領域を指定すると失敗する
    serverPort = reinterpret_cast<nn::svc::Handle*>(nnMain);
    result = nn::svc::CreatePort(serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
TEST(CreatePort, pOutClientTest)
{
    TestPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle* clientPort;

#ifdef INVALID_POINTER_TEST
    // TEST 112-9
    // NULL の領域を指定すると失敗する
    clientPort = NULL;
    result = nn::svc::CreatePort(&serverPort, clientPort, 1, false, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 112-10
    // MemoryPermission_None の領域を指定すると失敗する
    clientPort = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
    result = nn::svc::CreatePort(&serverPort, clientPort, 1, false, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 112-11
    // MemoryPermission_Read の領域を指定すると失敗する
    {
        TestHeap heap(HeapAlign);
        uintptr_t addr = heap.GetAddress();
        {
            TestMemoryPermission perm(addr, 0x1000, nn::svc::MemoryPermission_Read);
            clientPort = reinterpret_cast<nn::svc::Handle*>(addr);
            result = nn::svc::CreatePort(&serverPort, clientPort, 1, false, 0);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
        }
    }
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 112-12
    // MemoryPermission_ReadExecute の領域を指定すると失敗する
    clientPort = reinterpret_cast<nn::svc::Handle*>(nnMain);
    result = nn::svc::CreatePort(&serverPort, clientPort, 1, false, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
}
#endif // INVALID_POINTER_TEST

TEST(CreatePort, MaxSessionTest)
{
    TestPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;

    // TEST 112-15
    // TEST 112-16
    // int32_t の 1以上の数を受け付ける
    int32_t successData[] = {
        1,
        nn::svc::ArgumentHandleCountMax, nn::svc::ArgumentHandleCountMax + 1,
        0xFFF, 0x1000, 0x1001,
        0xFFFF, 0x10000, 0x10001,
        INT32_MAX
    };

    for (int32_t i = 0; i < static_cast<int>(sizeof(successData) / sizeof(int32_t)); i++)
    {
        // Normal Session
        {
            result = nn::svc::CreatePort(&serverPort, &clientPort, successData[i], false, 0);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sPortCloser(serverPort);
            AutoHandleClose cPortCloser(clientPort);

            CheckIpc(serverPort, clientPort, false);
        }

        // Light Session
        {
            result = nn::svc::CreatePort(&serverPort, &clientPort, successData[i], true, 0);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sPortCloser(serverPort);
            AutoHandleClose cPortCloser(clientPort);

            CheckIpc(serverPort, clientPort, true);
        }
    }

    // 0 は受け付けない
    {
        // TEST 112-17
        // Normal Session
        result = nn::svc::CreatePort(&serverPort, &clientPort, 0, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

        // TEST 112-18
        // Light Session
        result = nn::svc::CreatePort(&serverPort, &clientPort, 0, true, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());
    }

    // 負の値を指定すると失敗する
    {
        // TEST 112-19
        // Normal Session
        result = nn::svc::CreatePort(&serverPort, &clientPort, -1, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

        result = nn::svc::CreatePort(&serverPort, &clientPort, 0xFFFF0000, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

        result = nn::svc::CreatePort(&serverPort, &clientPort, 0x80000000, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

        // TEST 112-20
        // Light Session
        result = nn::svc::CreatePort(&serverPort, &clientPort, -1, true, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

        result = nn::svc::CreatePort(&serverPort, &clientPort, 0xFFFF0000, true, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());
        result = nn::svc::CreatePort(&serverPort, &clientPort, 0xFFFF0000, true, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());
    }
}

TEST(CreatePort, NameTest)
{
    TestPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    uintptr_t testData[] = {0, static_cast<uintptr_t>(-1), 1, 1 << (sizeof(uintptr_t) / 2)};

    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(testData) / sizeof(uintptr_t)); i++)
    {
        // Normal Session
        {
            result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, testData[i]);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sPortCloser(serverPort);
            AutoHandleClose cPortCloser(clientPort);

            CheckIpc(serverPort, clientPort, false);
        }

        // Light Session
        {
            result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, testData[i]);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sPortCloser(serverPort);
            AutoHandleClose cPortCloser(clientPort);

            CheckIpc(serverPort, clientPort, true);
        }
    }
}

