﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>

extern "C" void nnMain();

namespace {
#ifdef INVALID_POINTER_TEST
const char ConstHandle[sizeof(nn::svc::Handle)] = {0};
#endif
const uint64_t SpaceAddr = 0;
} // namespace

TEST(CreateDeviceAddressSpace, OutTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    size_t size = 0x200000; // 2MB
    nn::svc::Handle handle;
    nn::svc::Handle* pHandle;

    // TEST 129-1
    // MemoryPermission_ReadWrite の領域を受け付ける
    {
        // ローカル変数
        result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        // ヒープ変数
        TestHeap heap(HeapAlign);
        pHandle = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
        result = nn::svc::CreateDeviceAddressSpace(pHandle, SpaceAddr, size);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(*pHandle);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef INVALID_POINTER_TEST
    // TEST 129-2
    // NULL を受け付けない
    pHandle = NULL;
    result = nn::svc::CreateDeviceAddressSpace(pHandle, SpaceAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 129-3
    // MemoryPermission_None の領域を指し示していると失敗する
    pHandle = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
    result = nn::svc::CreateDeviceAddressSpace(pHandle, SpaceAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 129-4
    // MemoryPermission_Read の領域を指し示していると失敗する
    uintptr_t tmpAddr = reinterpret_cast<uintptr_t>(&ConstHandle);
    pHandle = reinterpret_cast<nn::svc::Handle*>(tmpAddr);
    result = nn::svc::CreateDeviceAddressSpace(pHandle, SpaceAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 129-4
    // MemoryPermission_ReadExecute の領域を指し示していると失敗する
    pHandle = reinterpret_cast<nn::svc::Handle*>(nnMain);
    result = nn::svc::CreateDeviceAddressSpace(pHandle, SpaceAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}

TEST(CreateDeviceAddressSpace, SizeTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    size_t alignSize = 0x200000; // 2MB
    nn::svc::Handle handle;
    uint64_t size;

    // TEST 129-8
    // 2MB にアライメントされていると成功する
    for (size = alignSize; size <= 0x100000000ul; size += alignSize)
    {
        result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 129-11
    // 0 を受け付けない
    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
    // TEST 129-12
    // 32bit アドレス空間しか持たないとき、64 bit のサイズを指定することが出来ない
    for (size = 0x100000000ul + alignSize; size < 0x400000000ul; size += alignSize)
    {
        result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
    }
#elif defined NN_BUILD_CONFIG_CPU_ARM_V8A
    // TEST 129-18
    // 34bit の値を受け付ける
    for (size = 0x100000000ul + alignSize; size <= 0x400000000ul; size += alignSize)
    {
        result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
#endif

    // TEST 129-19
    // 35bit 以上の値を受け付けない
    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
#ifdef SUPPORT_RESULT_CREATE_DEVICE_ADDRESS_SPACE
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
#else
    ASSERT_RESULT_FAILURE(result);
#endif // SUPPORT_RESULT_CREATE_DEVICE_ADDRESS_SPACE
}

