﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern "C" void nnMain();
extern nn::Result ClientLightSession(nn::svc::Handle handle);
extern nn::Result ServerLightSession(nn::svc::Handle handle);

namespace {

#ifdef INVALID_POINTER_TEST
const char ConstVar[sizeof(nn::svc::Handle)] = { 0 };
#endif
const size_t StackSize = 0x1000;
char g_Stack[StackSize] __attribute__((aligned(0x1000)));

void NormalSessionClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(*clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // LightSessionClientThread に合わせて、こちらもclientSession を閉じる
    result = nn::svc::CloseHandle(*clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void LightSessionClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);

    result = ClientLightSession(*clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // ClientSession が閉じられることが期待されている
    result = nn::svc::CloseHandle(*clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckIpc(nn::svc::Handle serverPort, nn::svc::Handle clientSession, bool isLight)
{
    nn::Result result;

    uintptr_t pc = (isLight)
        ? reinterpret_cast<uintptr_t>(LightSessionClientThread)
        : reinterpret_cast<uintptr_t>(NormalSessionClientThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    if (isLight)
    {
        result = ServerLightSession(serverSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
    }
    else
    {
        nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        ipcMsg.SetNull();

        result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        ipcMsg.SetNull();

        result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    thread.Wait();
}

} // namespace

#ifdef INVALID_POINTER_TEST
TEST(ConnectToPort, InvalidPointerTest)
{
    TestPortLeak leakTest;
    nn::Result result;

    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle *clientSession;

#ifdef INVALID_POINTER_TEST
    // TEST 114-2
    // NULL アドレスを受け付けない
    clientSession = NULL;
    result = nn::svc::ConnectToPort(clientSession, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 114-3
    // MemoryPermission_None の領域を指定すると失敗する
    clientSession = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
    result = nn::svc::ConnectToPort(clientSession, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 114-4
    // MemoryPermission_Read の領域を指定すると失敗する
    uintptr_t addr = reinterpret_cast<uintptr_t>(ConstVar);
    clientSession = reinterpret_cast<nn::svc::Handle*>(addr);
    result = nn::svc::ConnectToPort(clientSession, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 114-5
    // MemoryPermission_ReadExecute の領域を指定すると失敗する
    clientSession = reinterpret_cast<nn::svc::Handle*>(nnMain);
    result = nn::svc::ConnectToPort(clientSession, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}
#endif // INVALID_POINTER_TEST

TEST(ConnectToPort, PortTest)
{
    TestPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle clientSession;

    // Normal Session (Local Var)
    {
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);

        // TEST 114-9
        // Normal Session のサーバーポートを受け付けない
        result = nn::svc::ConnectToPort(&clientSession, serverPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        // TEST 114-1
        // pOut にMemoryPermission_ReadWrite の領域を指定すると成功する
        // TEST 114-8
        // Normal Session のクライアントポートを受け付ける
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);

        // CheckIpc の中で、clientSession は閉じられる
        CheckIpc(serverPort, clientSession, false);

        // TEST 114-12
        // Close した Normal Session のクライアントポートを受け付けない
        result = nn::svc::CloseHandle(clientPort);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    }

    // Normal Session (Heap Var)
    {
        TestHeap heap(sizeof(nn::svc::Handle));
        nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());

        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);

        // TEST 114-9
        // Normal Session のサーバーポートを受け付けない
        result = nn::svc::ConnectToPort(handle, serverPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        // TEST 114-1
        // pOut にMemoryPermission_ReadWrite の領域を指定すると成功する
        // TEST 114-8
        // Normal Session のクライアントポートを受け付ける
        result = nn::svc::ConnectToPort(handle, clientPort);
        ASSERT_RESULT_SUCCESS(result);

        // CheckIpc の中で、clientSession は閉じられる
        CheckIpc(serverPort, *handle, false);

        // TEST 114-12
        // Close した Normal Session のクライアントポートを受け付けない
        result = nn::svc::CloseHandle(clientPort);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::ConnectToPort(handle, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // Light Session (Local Var)
    {
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);

        // TEST 114-11
        // Light Session のサーバーポートを受け付けない
        result = nn::svc::ConnectToPort(&clientSession, serverPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        // TEST 114-1
        // pOut にMemoryPermission_ReadWrite の領域を指定すると成功する
        // TEST 114-10
        // Light Session のクライアントポートを受け付ける
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);

        // CheckIpc の中で、clientSession は閉じられる
        CheckIpc(serverPort, clientSession, true);

        // TEST 114-13
        // Close した Light Session のクライアントポートを受け付けない
        result = nn::svc::CloseHandle(clientPort);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // Light Session (Heap Var)
    {
        TestHeap heap(sizeof(nn::svc::Handle));
        nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());

        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);

        // TEST 114-11
        // Light Session のサーバーポートを受け付けない
        result = nn::svc::ConnectToPort(handle, serverPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        // TEST 114-1
        // pOut にMemoryPermission_ReadWrite の領域を指定すると成功する
        // TEST 114-10
        // Light Session のクライアントポートを受け付ける
        result = nn::svc::ConnectToPort(handle, clientPort);
        ASSERT_RESULT_SUCCESS(result);

        // CheckIpc の中で、clientSession は閉じられる
        CheckIpc(serverPort, *handle, true);

        // TEST 114-13
        // Close した Light Session のクライアントポートを受け付けない
        result = nn::svc::CloseHandle(clientPort);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::ConnectToPort(handle, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    // TEST 114-14
    // 名前付きポートのサーバーポートを受け付けない
    {
        const char* name = "TEST11414";
        result = nn::svc::ManageNamedPort(&serverPort, name, 1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::ConnectToPort(&clientSession, serverPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        result = nn::svc::CloseHandle(serverPort);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::ManageNamedPort(&serverPort, name, 0);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 114-15
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::ConnectToPort(&clientSession, nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 114-16
    // T_CurrentThread を受け付けない
    result = nn::svc::ConnectToPort(&clientSession, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 114-16
    // T_CurrentProcess を受け付けない
    result = nn::svc::ConnectToPort(&clientSession, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

