﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "sdmmc_Result.h"
#include <nn/dd.h>

//#define NN_DETAIL_SDMMC_ADMA2_ENABLE    // エラーハンドリングが不十分のため、使用禁止

namespace nn { namespace sdmmc1 {

    enum Port
    {
        Port_Mmc0,
        Port_SdCard0,
        Port_GcAsic0
    };

    // SMMU 有効時に使用
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        const size_t BufferDeviceVirtualAddressAlignment = 4 * 1024;    // ADMA2 使用想定
    #else
        const size_t BufferDeviceVirtualAddressAlignment = 512 * 1024;  // SDMA 512KB 使用想定
    #endif

    struct BufferInfo {
        void* pBuffer;
        size_t bufferSize;
        nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress;
    };

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        const size_t WorkBufferSizeForHostController = 4096;
    #endif

    const size_t SectorSize = 512;

    enum BusPower {
        BusPower_Off,
        BusPower_1_8V,
        BusPower_3_3V
    };

    enum BusWidth {
        BusWidth_1Bit,
        BusWidth_4Bit,
        BusWidth_8Bit
    };

    enum SpeedMode {
        SpeedMode_Identification,
        SpeedMode_MmcLegacySpeed,
        SpeedMode_MmcHighSpeed,
        SpeedMode_MmcHs200,
        SpeedMode_MmcHs400,
        SpeedMode_SdCardDefaultSpeed,
        SpeedMode_SdCardHighSpeed,
        SpeedMode_SdCardSdr12,
        SpeedMode_SdCardSdr25,
        SpeedMode_SdCardSdr50,
        SpeedMode_SdCardSdr104,
        SpeedMode_SdCardDdr50,
        SpeedMode_GcAsicFpgaSpeed,
        SpeedMode_GcAsicSpeed
    };

    struct HostBusStatus
    {
        BusPower busPower;
        BusWidth busWidth;
        uint32_t deviceClockFrequencyKHz;
    };

    const size_t DeviceCidSize = 16;
    const size_t DeviceCsdSize = 16;

    typedef void (*DeviceDetectionEventCallback)(void* pParameter);

    struct DataTransfer
    {
        void* pDataBuffer;
        size_t dataBufferSize;
        size_t blockSize;
        uint32_t numBlocks;
        bool isRead;
    };

    // 対応していない環境で実行すると ABORT します。
    void Initialize(Port port, const BufferInfo* pBufferInfos, int numBufferInfos) NN_NOEXCEPT;
    void Initialize(Port port) NN_NOEXCEPT;
    void Finalize(Port port) NN_NOEXCEPT;
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        void SetWorkBufferForHostController(Port port, void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT;
    #endif
    void Deactivate(Port port) NN_NOEXCEPT;
    Result Read(void* pOutDataBuffer, size_t dataBufferSize, Port port, uint32_t sectorIndex, uint32_t numSectors) NN_NOEXCEPT;
    Result Write(Port port, uint32_t sectorIndex, uint32_t numSectors, const void* pDataBuffer, size_t dataBufferSize) NN_NOEXCEPT;
    void GetHostBusStatus(HostBusStatus* pOutHostBusStatus, Port port) NN_NOEXCEPT;
    Result GetDeviceMemoryCapacity(uint32_t* pOutNumSectors, Port port) NN_NOEXCEPT;
    Result GetDeviceStatus(uint32_t* pOutDeviceStatus, Port port) NN_NOEXCEPT;
    Result GetDeviceOcr(uint32_t* pOutOcr, Port port) NN_NOEXCEPT;
    Result GetDeviceRca(uint16_t* pOutRca, Port port) NN_NOEXCEPT;
    Result GetDeviceCid(void* pOutCid, size_t cidSize, Port port) NN_NOEXCEPT;
    Result GetDeviceCsd(void* pOutCsd, size_t csdSize, Port port) NN_NOEXCEPT;

    void SetAbortCommandForDebug(Port port, uint32_t commandIndex, uint32_t commandArgument, int64_t abortTimingNanoSeconds) NN_NOEXCEPT;
    void ClearAbortCommandForDebug(Port port) NN_NOEXCEPT;
    Result IssueCommandForDebug(uint32_t* pOutResponse, Port port, uint32_t commandIndex, uint32_t commandArgument,
        DataTransfer* pDataTransfer, bool isBusy) NN_NOEXCEPT;

}} // namespace nn { namespace sdmmc1 {
