﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_SdmmcController.tegra-k1.h"
#include "sdmmc_Log.h"

namespace nn { namespace sdmmc1 {
namespace detail {

namespace
{
    // レジスタログ
    #define NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(basePointer, memberVariable) NN_DETAIL_SDMMC_DEBUG_LOG(#memberVariable "(0x%03x) 0x%08X\n",\
        reinterpret_cast<uintptr_t>(&(basePointer->memberVariable)) - reinterpret_cast<uintptr_t>(basePointer), (basePointer->memberVariable))
}

void SdmmcController::LogRegisters() NN_NOEXCEPT
{
    SdHostStandardController::LogRegisters();

    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorClockCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorSysSwCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorCapOverrides0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorBootCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorBootActTimeout0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, bootDatTimeout0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorDebounceCount0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorMiscCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, maxCurrentOverride0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, maxCurrentOverrideHi0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorClkGateHysteresisCount0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorPresetVal00);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorPresetVal10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorPresetVal20);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, sdmemcomppadctrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, autoCalConfig0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, autoCalInterval0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, autoCalStatus0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, mmcifFifoctrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, timeoutWcalSdmmc0);
}

Result SdmmcController::Startup(BusPower busPower, BusWidth busWidth, SpeedMode speedMode, bool isPowerSavingEnable) NN_NOEXCEPT
{
    detail::ClockResetController::Module module = GetClockResetModule();
    if (detail::ClockResetController::IsAvailable(module))
    {
        // クロックソース変更によるグリッジを防ぐためにクロック供給を止める
        SdHostStandardController::DisableDeviceClock();
    }
    detail::ClockResetController::Reset(module);
    //LogRegisters();

    // ここまで実行したら Shutdown() が可能
    m_IsShutdowned = false;

    Result result = SdHostStandardController::EnableInternalClock();
    if (result.IsFailure())
    {
        return result;
    }

    SdHostStandardController::SetBusWidth(busWidth);
    SdHostStandardController::SetBusPower(busPower);
    result = SetSpeedMode(speedMode);
    if (result.IsFailure())
    {
        return result;
    }
    SdHostStandardController::SetPowerSaving(isPowerSavingEnable);
    SdHostStandardController::EnableDeviceClock();

    return ResultSuccess();
}

void SdmmcController::Shutdown() NN_NOEXCEPT
{
    if (m_IsShutdowned)
    {
        return;
    }
    SdHostStandardController::DisableDeviceClock();
    SdHostStandardController::SetBusPower(BusPower_Off);
    m_IsShutdowned = true;
}

Result SdmmcController::SetSpeedMode(SpeedMode speedMode) NN_NOEXCEPT
{
    bool isClockEnabled = false;

    if ((m_pRegisters->sdHostStandardRegisters.clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) != 0)
    {
        isClockEnabled = true;

        // クロックソースを変更するため、一時的にクロックを停止する
        m_pRegisters->sdHostStandardRegisters.clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
    }

    uint32_t targetSourceClockFrequencyKHz;
    uint16_t x; // Divided by "X = 1 / (2 * N)", 1 分周より大きいものは 2 の倍数
    switch (speedMode)
    {
    case SpeedMode_Identification:
        m_pRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        // 400kHz 以下、25MHz 以下を 64 分周
        targetSourceClockFrequencyKHz = 25000;
        x = 64;
        break;
    case SpeedMode_SdCardDefaultSpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        // 25MHz 以下を 1 分周
        targetSourceClockFrequencyKHz = 25000;
        x = 1;
        break;
    case SpeedMode_MmcLegacySpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        // 26MHz 以下を 1 分周
        targetSourceClockFrequencyKHz = 26000;
        x = 1;
        break;
    case SpeedMode_SdCardHighSpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl1 |= REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE;
        // 50MHz 以下を 1 分周
        targetSourceClockFrequencyKHz = 50000;
        x = 1;
        break;
    case SpeedMode_MmcHighSpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl1 |= REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE;
        // 52MHz 以下を 1 分周
        targetSourceClockFrequencyKHz = 52000;
        x = 1;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // ソースクロックの設定
    uint32_t actualSourceClockFrequencyKHz;
    detail::ClockResetController::SetClockFrequencyKHz(&actualSourceClockFrequencyKHz, GetClockResetModule(), targetSourceClockFrequencyKHz);

    // デバイスクロックの算出（Wait 時間算出に使用するため、繰り上げ計算）
    uint32_t actualDeviceClockFrequencyKHz = (actualSourceClockFrequencyKHz + (x - 1)) / x;
    SdHostStandardController::SetDeviceClockFrequencyKHz(actualDeviceClockFrequencyKHz);

    // SDMMC での分周設定
    NN_ABORT_UNLESS((x == 1) || ((x & 0x1) == 0));
    uint16_t n = x / 2;     // N = 1 / (2 * X), 0: Base Clock
    uint16_t upper_n = 0;
    if (n > 0xFF)
    {
        upper_n = n >> 8;
    }
    uint16_t mask = REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__MASK | REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__MASK;
    m_pRegisters->sdHostStandardRegisters.clockControl = (m_pRegisters->sdHostStandardRegisters.clockControl & (~mask)) | (n << REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__SHIFT) | (upper_n << REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__SHIFT);

    if (isClockEnabled)
    {
        m_pRegisters->sdHostStandardRegisters.clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
    }

    return ResultSuccess();
}

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::InterruptEventType Sdmmc3Controller::m_InterruptEvent;
    #endif
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::InterruptEventType Sdmmc4Controller::m_InterruptEvent;
    #endif
#endif

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
