﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_GcAsicDeviceAccessor.h"
#include "sdmmc_Timer.h"
#include "../sdmmc_GcAsic.h"
#include <nn/result/result_HandlingUtility.h>
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
#endif

namespace nn { namespace sdmmc1 {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_GcAsicDevice.m_DeviceMutex)
#else
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD
#endif

namespace
{
    #if (defined(NN_DETAIL_SDMMC_GC_ASIC_IS_FPGA))
        SpeedMode GcAsicSpeedMode = SpeedMode_GcAsicFpgaSpeed;
    #else
        SpeedMode GcAsicSpeedMode = SpeedMode_GcAsicSpeed;
    #endif
}

Result GcAsicDeviceAccessor::IssueCommandWriteOperation(const void* pOperationBuffer, size_t operationBufferSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOperationBuffer);
    NN_ABORT_UNLESS(operationBufferSize >= GcAsicOperationSize);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // Argument は 0 固定
    ResponseType responseType = ResponseType_R1;
    Command command(60, 0, responseType, true);
    TransferData transferData(const_cast<void*>(pOperationBuffer), GcAsicOperationSize, 1, TransferDirection_WriteToDevice);
    Result result = pHostController->IssueCommand(&command, &transferData);
    if (result.IsFailure())
    {
        return result;
    }

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    return m_GcAsicDevice.CheckDeviceStatus(response);
}

Result GcAsicDeviceAccessor::IssueCommandFinishOperation() const NN_NOEXCEPT
{
    // Argument は 0 固定
    return BaseDeviceAccessor::IssueCommandAndCheckR1(61, 0, true, DeviceState_Tran);
}

Result GcAsicDeviceAccessor::IssueCommandSleep() const NN_NOEXCEPT
{
    // Argument は 0 固定
    return BaseDeviceAccessor::IssueCommandAndCheckR1(62, 0, true, DeviceState_Tran);
}

Result GcAsicDeviceAccessor::IssueCommandUpdateKey() const NN_NOEXCEPT
{
    // Argument は 0 固定
    return BaseDeviceAccessor::IssueCommandAndCheckR1(63, 0, true, DeviceState_Tran);
}

void GcAsicDeviceAccessor::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_IsInitialized)
    {
        return;
    }

    BaseDeviceAccessor::SetDevice(&m_GcAsicDevice);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        m_GcAsicDevice.InitializeRemovedEvent();
        pHostController->PreSetRemovedEvent(m_GcAsicDevice.GetRemovedEvent());
    #endif
    pHostController->Initialize();

    m_IsInitialized = true;
}

void GcAsicDeviceAccessor::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (!m_IsInitialized)
    {
        return;
    }
    m_IsInitialized = false;

    BaseDeviceAccessor::Deactivate();

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Finalize();
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        m_GcAsicDevice.FinalizeRemovedEvent();
    #endif
}

Result GcAsicDeviceAccessor::ActivateGcAsic() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_GcAsicDevice.IsActive())
    {
        // Activate 済みならば何もしない
        return ResultSuccess();
    }

    // HostController 起動
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    Result result = pHostController->Startup(BusPower_1_8V, BusWidth_8Bit, GcAsicSpeedMode, false);
    if (result.IsFailure())
    {
        return result;
    }

    // 10 clock cycles 待ち
    WaitClocks(10, pHostController->GetDeviceClockFrequencyKHz());

    // いきなり Tuning
    NN_ABORT_UNLESS(pHostController->IsSupportedTuning());
    result = pHostController->Tuning(GcAsicSpeedMode, 21);
    if (result.IsFailure())
    {
        return result;
    }

    // Argument は 0 固定のため true, false どちらでも構わないが、
    // IssueCommandMultipleBlock() で IsHighCapacity() を使用できるように、いずれか設定する
    m_GcAsicDevice.SetHighCapacity(false);

    // IssueCommandMultipleBlock() で GetMemoryCapacity() を使用した
    // 終端チェックが無効になるよう便宜上 0 を設定する
    m_GcAsicDevice.SetMemoryCapacity(0);

    // 以降、クロック供給を抑える
    pHostController->SetPowerSaving(true);

    m_GcAsicDevice.SetActive();
    return ResultSuccess();
}

Result GcAsicDeviceAccessor::GetGcAsicSpeedMode(SpeedMode* pOutSpeedMode) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutSpeedMode);

    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    // ゲームカード ASIC 側 Speed Mode は固定で取得機能はない
    *pOutSpeedMode = GcAsicSpeedMode;
    return ResultSuccess();
}

Result GcAsicDeviceAccessor::WriteGcAsicOperation(const void* pOperationBuffer, size_t operationBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    Result result = IssueCommandWriteOperation(pOperationBuffer, operationBufferSize);
    if (result.IsFailure())
    {
        return result;
    }
    return BaseDeviceAccessor::IssueCommandSendStatus();
}

Result GcAsicDeviceAccessor::FinishGcAsicOperation() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    Result result = IssueCommandFinishOperation();
    if (result.IsFailure())
    {
        return result;
    }
    return BaseDeviceAccessor::IssueCommandSendStatus();
}

Result GcAsicDeviceAccessor::SleepGcAsic() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    Result result = IssueCommandSleep();
    if (result.IsFailure())
    {
        return result;
    }
    return BaseDeviceAccessor::IssueCommandSendStatus();
}

Result GcAsicDeviceAccessor::UpdateGcAsicKey() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    Result result = IssueCommandUpdateKey();
    if (result.IsFailure())
    {
        return result;
    }
    return BaseDeviceAccessor::IssueCommandSendStatus();
}

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
