﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/gpio/gpio.h>
#include "../sdmmc_Common.h"

namespace nn { namespace sdmmc1 {
namespace detail {

typedef void (*InsertedCallback)(void* pParameter);
typedef void (*RemovedCallback)(void* pParameter);

struct CallbackInfos
{
    InsertedCallback insertedCallback;
    void* pInsertedCallbackParameter;
    RemovedCallback removedCallback;
    void* pRemovedCallbackParameter;
};

class DeviceDetector
{
private:
    NN_OS_ALIGNAS_THREAD_STACK uint8_t m_DetectorThreadStack[8192];
    nn::os::ThreadType m_DetectorThread;
    const int DetectorThreadPriority = 9;   // TODO: fs ワーカスレッド - 1
    nn::os::EventType m_DetectorThreadEndEvent;

    nn::os::EventType m_ReadyDeviceStatusEvent;

    nn::gpio::GpioPadName m_GpioPadName;
    nn::gpio::GpioValue m_GpioValueInserted;
    int m_GpioDebounceMicroSeconds;
    nn::gpio::GpioPadSession m_GpioPadSession;

    InsertedCallback m_InsertedCallback;
    void* m_pInsertedCallbackParameter;
    RemovedCallback m_RemovedCallback;
    void* m_pRemovedCallbackParameter;

    DeviceDetectionEventCallback m_DeviceDetectionEventCallback;
    void* m_pDeviceDetectionEventCallbackParameter;

    bool IsInserted(bool isNeedToCheckReady) NN_NOEXCEPT;
    void DetectorThread() NN_NOEXCEPT;

    static void DetectorThreadEntry(void* arg) NN_NOEXCEPT
    {
        DeviceDetector* pDeviceDetector = reinterpret_cast<DeviceDetector*>(arg);
        pDeviceDetector->DetectorThread();
    }

public:
    explicit DeviceDetector(nn::gpio::GpioPadName gpioPadName, nn::gpio::GpioValue gpioValueInserted, int gpioDebounceMicroSeconds) NN_NOEXCEPT
        : m_GpioPadName(gpioPadName), m_GpioValueInserted(gpioValueInserted), m_GpioDebounceMicroSeconds(gpioDebounceMicroSeconds)
    {
        m_InsertedCallback = nullptr;
        m_RemovedCallback = nullptr;
        m_DeviceDetectionEventCallback = nullptr;
    }

    void Initialize(CallbackInfos* pCallbackInfos) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    bool IsInserted() NN_NOEXCEPT;
    void RegisterDetectionEventCallback(DeviceDetectionEventCallback callback, void* pParameter) NN_NOEXCEPT;
    void UnregisterDetectionEventCallback() NN_NOEXCEPT;
};

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
