﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_ClockResetController.tegra.h"
#include <nn/TargetConfigs/build_Base.h>
#include <nn/dd.h>
#include <nn/nn_Abort.h>
#include "sdmmc_Timer.h"

namespace nn { namespace sdmmc1 {
namespace detail {

namespace ClockResetController {

namespace
{
    const uint32_t MaxTargetClockFrequencyKHz = 208000;

    struct ClockResetControllerRegisters
    {
        volatile uint32_t   rstSource0;         // 0x0
        volatile uint32_t   rstDevicesL0;       // 0x4
        volatile uint32_t   rstDevicesH0;       // 0x8
        volatile uint32_t   rstDevicesU0;       // 0xc
        volatile uint32_t   clkOutEnbL0;        // 0x10
        volatile uint32_t   clkOutEnbH0;        // 0x14
        volatile uint32_t   clkOutEnbU0;        // 0x18
        volatile uint8_t    dummy0[0x134];
        volatile uint32_t   clkSourceSdmmc1;    // 0x150
        volatile uint32_t   clkSourceSdmmc2;    // 0x154
        volatile uint8_t    dummy1[0xC];
        volatile uint32_t   clkSourceSdmmc4;    // 0x164
        volatile uint8_t    dummy2[0x54];
        volatile uint32_t   clkSourceSdmmc3;    // 0x1bc
        volatile uint8_t    dummy3[0xD8];
        volatile uint32_t   clkOutEnbY0;        // 0x298
        volatile uint32_t   clkEnbYSet0;        // 0x29c
        volatile uint32_t   clkEnbYClr0;        // 0x2a0
        volatile uint8_t    dummy4[0x5C];
        volatile uint32_t   rstDevLSet0;        // 0x300
        volatile uint32_t   rstDevLClr0;        // 0x304
        volatile uint8_t    dummy5[0x8];
        volatile uint32_t   rstDevUSet0;        // 0x310
        volatile uint32_t   rstDevUClr0;        // 0x314
        volatile uint8_t    dummy6[0x8];
        volatile uint32_t   clkEnbLSet0;        // 0x320
        volatile uint32_t   clkEnbLClr0;        // 0x324
        volatile uint8_t    dummy7[0x8];
        volatile uint32_t   clkEnbUSet0;        // 0x330
        volatile uint32_t   clkEnbUClr0;        // 0x334
        volatile uint8_t    dummy8[0x26C];
        volatile uint32_t   pllc4Base0;         // 0x5a4
        volatile uint32_t   pllc4Misc0;         // 0x5a8
        volatile uint8_t    dummy9[0x38];
        volatile uint32_t   pllc4Out0;          // 0x5e4
        volatile uint8_t    dummy10[0xAC];
        volatile uint32_t   clkSourceSdmmcLegacyTm0;    // 0x694
    };

    const nn::dd::PhysicalAddress ClockResetControllerRegistersPhysicalAddress = 0x60006000ull;
    const size_t ClockResetControllerRegistersSize = 0x1000;

    ClockResetControllerRegisters* s_pRegisters = nullptr;

    // 0x4 レジスタ CLK_RST_CONTROLLER_RST_DEVICES_L_0 ビット定義
    const uint32_t REG_RST_DEVICES_L_0__SDMMC4 = 0x1U << 15;
    const uint32_t REG_RST_DEVICES_L_0__SDMMC1 = 0x1U << 14;
    const uint32_t REG_RST_DEVICES_L_0__SDMMC2 = 0x1U << 9;

    // 0xc レジスタ CLK_RST_CONTROLLER_RST_DEVICES_U_0 ビット定義
    const uint32_t REG_RST_DEVICES_U_0__SDMMC3 = 0x1U << 5;

    // 0x10 レジスタ CLK_RST_CONTROLLER_CLK_OUT_ENB_L_0 ビット定義
    const uint32_t REG_CLK_OUT_ENB_L_0__SDMMC4 = 0x1U << 15;
    const uint32_t REG_CLK_OUT_ENB_L_0__SDMMC1 = 0x1U << 14;
    const uint32_t REG_CLK_OUT_ENB_L_0__SDMMC2 = 0x1U << 9;

    // 0x18 レジスタ CLK_RST_CONTROLLER_CLK_OUT_ENB_U_0 ビット定義
    const uint32_t REG_CLK_OUT_ENB_U_0__SDMMC3 = 0x1U << 5;

    // 0x150. 0x154, 0x164, 0x1bc CLK_RST_CONTROLLER_CLK_SOURCE_SDMMCx_0 ビット定義
    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1) && defined(NN_DETAIL_SDMMC_SET_PLLC4_DIRECTLY))
        const uint8_t REG_CLK_SOURCE_SDMMCX_0__CLK_M__SHIFT = 29;
        const uint32_t REG_CLK_SOURCE_SDMMC24_0__CLK_M__PLLC4_OUT2_LJ   = 0x1U << REG_CLK_SOURCE_SDMMCX_0__CLK_M__SHIFT;
        const uint32_t REG_CLK_SOURCE_SDMMCX_0__CLK_M__PLLC4_OUT2       = 0x3U << REG_CLK_SOURCE_SDMMCX_0__CLK_M__SHIFT;
    #endif

    // 0x29c レジスタ CLK_RST_CONTROLLER_CLK_ENB_Y_SET_0 ビット定義
    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1))
        const uint32_t REG_CLK_ENB_Y_SET_0__SDMMC_LEGACY_TM = 0x1U << 1;
    #endif

    // 0x300 レジスタ CLK_RST_CONTROLLER_RST_DEV_L_SET_0 ビット定義
    const uint32_t REG_RST_DEV_L_SET_0__SDMMC4 = 0x1U << 15;
    const uint32_t REG_RST_DEV_L_SET_0__SDMMC1 = 0x1U << 14;
    const uint32_t REG_RST_DEV_L_SET_0__SDMMC2 = 0x1U << 9;

    // 0x304 レジスタ CLK_RST_CONTROLLER_RST_DEV_L_CLR_0 ビット定義
    const uint32_t REG_RST_DEV_L_CLR_0__SDMMC4 = 0x1U << 15;
    const uint32_t REG_RST_DEV_L_CLR_0__SDMMC1 = 0x1U << 14;
    const uint32_t REG_RST_DEV_L_CLR_0__SDMMC2 = 0x1U << 9;

    // 0x310 レジスタ CLK_RST_CONTROLLER_RST_DEV_U_SET_0 ビット定義
    const uint32_t REG_RST_DEV_U_SET_0__SDMMC3 = 0x1U << 5;

    // 0x314 レジスタ CLK_RST_CONTROLLER_RST_DEV_U_CLR_0 ビット定義
    const uint32_t REG_RST_DEV_U_CLR_0__SDMMC3 = 0x1U << 5;

    // 0x320 レジスタ CLK_RST_CONTROLLER_CLK_ENB_L_SET_0 ビット定義
    const uint32_t REG_CLK_ENB_L_SET_0__SDMMC4 = 0x1U << 15;
    const uint32_t REG_CLK_ENB_L_SET_0__SDMMC1 = 0x1U << 14;
    const uint32_t REG_CLK_ENB_L_SET_0__SDMMC2 = 0x1U << 9;

    // 0x324 レジスタ CLK_RST_CONTROLLER_CLK_ENB_L_CLR_0 ビット定義
    const uint32_t REG_CLK_ENB_L_CLR_0__SDMMC4 = 0x1U << 15;
    const uint32_t REG_CLK_ENB_L_CLR_0__SDMMC1 = 0x1U << 14;
    const uint32_t REG_CLK_ENB_L_CLR_0__SDMMC2 = 0x1U << 9;

    // 0x330 レジスタ CLK_RST_CONTROLLER_CLK_ENB_U_SET_0 ビット定義
    const uint32_t REG_CLK_ENB_U_SET_0__SDMMC3 = 0x1U << 5;

    // 0x334 レジスタ CLK_RST_CONTROLLER_CLK_ENB_U_CLR_0 ビット定義
    const uint32_t REG_CLK_ENB_U_CLR_0__SDMMC3 = 0x1U << 5;

    // 0x694 レジスタ CLK_RST_CONTROLLER_CLK_SOURCE_SDMMC_LEGACY_TM_0 ビット定義
    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1))
        const uint8_t REG_CLK_SOURCE_SDMMC_LEGACY_TM_0__SDMMC_LEGACY_TM_CLK_SRC__SHIFT = 29;
        const uint32_t REG_CLK_SOURCE_SDMMC_LEGACY_TM_0__SDMMC_LEGACY_TM_CLK_SRC__PLLP_OUT0 = 4U << REG_CLK_SOURCE_SDMMC_LEGACY_TM_0__SDMMC_LEGACY_TM_CLK_SRC__SHIFT;
    #endif

    // 0x5a4 レジスタ CLK_RST_CONTROLLER_PLLC4_BASE_0 ビット定義
    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1) && defined(NN_DETAIL_SDMMC_SET_PLLC4_DIRECTLY))
        const uint32_t REG_PLLC4_BASE_0__PLLC4_ENABLE = 0x1U << 30;
        const uint32_t REG_PLLC4_BASE_0__PLLC4_LOCK = 0x1U << 27;
        const uint32_t REG_PLLC4_BASE_0__PLLC4_IDDQ = 0x1U << 18;
    #endif

    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1) && defined(NN_DETAIL_SDMMC_SET_PLLC4_DIRECTLY))
        void InitializePllc4()
        {
            const uint32_t ExpectedPllc4Base0Value = 0x58006804;    // PLLC4 998.4MH
            if (s_pRegisters->pllc4Base0 == ExpectedPllc4Base0Value)
            {
                return; // すでに設定済み
            }

            if ((s_pRegisters->pllc4Base0 & REG_PLLC4_BASE_0__PLLC4_ENABLE) != 0)
            {
                s_pRegisters->pllc4Base0 &= (~REG_PLLC4_BASE_0__PLLC4_ENABLE);
            }

            s_pRegisters->pllc4Base0 &= (~REG_PLLC4_BASE_0__PLLC4_IDDQ);
            WaitMicroseconds(5);

            s_pRegisters->pllc4Base0 = ExpectedPllc4Base0Value & (~(REG_PLLC4_BASE_0__PLLC4_ENABLE | REG_PLLC4_BASE_0__PLLC4_IDDQ));

            s_pRegisters->pllc4Base0 |= REG_PLLC4_BASE_0__PLLC4_ENABLE;

            ManualTimer timer(1000);    // 適当に長い 1 秒
            while (true)
            {
                if ((s_pRegisters->pllc4Base0 & REG_PLLC4_BASE_0__PLLC4_LOCK) != 0)
                {
                    break;
                }
                if (!(timer.Update()))
                {
                    // PLLC4 が LOCK されないまま、タイムアウトを過ぎている
                    NN_ABORT("PLLC4 could not be programed.");
                }
            }
        }
    #endif

    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1))
        void InitializeTmclk()
        {
            NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

            uint8_t n = 66; // Divided by "X = (N / 2) + 1", N = (X - 1) * 2, PLLP 408MHz / 34 = 12MHz
            s_pRegisters->clkSourceSdmmcLegacyTm0 = REG_CLK_SOURCE_SDMMC_LEGACY_TM_0__SDMMC_LEGACY_TM_CLK_SRC__PLLP_OUT0 | n;

            s_pRegisters->clkEnbYSet0 = REG_CLK_ENB_Y_SET_0__SDMMC_LEGACY_TM;
        }
    #endif

    bool IsResetReleased(Module module) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

        switch (module)
        {
        case Module_Sdmmc1:
            if ((s_pRegisters->rstDevicesL0 & REG_RST_DEVICES_L_0__SDMMC1) != 0)
            {
                return false;
            }
            else
            {
                return true;
            }
        case Module_Sdmmc2:
            if ((s_pRegisters->rstDevicesL0 & REG_RST_DEVICES_L_0__SDMMC2) != 0)
            {
                return false;
            }
            else
            {
                return true;
            }
        case Module_Sdmmc3:
            if ((s_pRegisters->rstDevicesU0 & REG_RST_DEVICES_U_0__SDMMC3) != 0)
            {
                return false;
            }
            else
            {
                return true;
            }
        case Module_Sdmmc4:
            if ((s_pRegisters->rstDevicesL0 & REG_RST_DEVICES_L_0__SDMMC4) != 0)
            {
                return false;
            }
            else
            {
                return true;
            }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void SetReset(Module module) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

        switch (module)
        {
        case Module_Sdmmc1:
            s_pRegisters->rstDevLSet0 = REG_RST_DEV_L_SET_0__SDMMC1;
            return;
        case Module_Sdmmc2:
            s_pRegisters->rstDevLSet0 = REG_RST_DEV_L_SET_0__SDMMC2;
            return;
        case Module_Sdmmc3:
            s_pRegisters->rstDevUSet0 = REG_RST_DEV_U_SET_0__SDMMC3;
            return;
        case Module_Sdmmc4:
            s_pRegisters->rstDevLSet0 = REG_RST_DEV_L_SET_0__SDMMC4;
            return;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void ClearReset(Module module) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

        switch (module)
        {
        case Module_Sdmmc1:
            s_pRegisters->rstDevLClr0 = REG_RST_DEV_L_CLR_0__SDMMC1;
            return;
        case Module_Sdmmc2:
            s_pRegisters->rstDevLClr0 = REG_RST_DEV_L_CLR_0__SDMMC2;
            return;
        case Module_Sdmmc3:
            s_pRegisters->rstDevUClr0 = REG_RST_DEV_U_CLR_0__SDMMC3;
            return;
        case Module_Sdmmc4:
            s_pRegisters->rstDevLClr0 = REG_RST_DEV_L_CLR_0__SDMMC4;
            return;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    bool IsClockEnabled(Module module) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

        switch (module)
        {
        case Module_Sdmmc1:
            if ((s_pRegisters->clkOutEnbL0 & REG_CLK_OUT_ENB_L_0__SDMMC1) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        case Module_Sdmmc2:
            if ((s_pRegisters->clkOutEnbL0 & REG_CLK_OUT_ENB_L_0__SDMMC2) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        case Module_Sdmmc3:
            if ((s_pRegisters->clkOutEnbU0 & REG_CLK_OUT_ENB_U_0__SDMMC3) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        case Module_Sdmmc4:
            if ((s_pRegisters->clkOutEnbL0 & REG_CLK_OUT_ENB_L_0__SDMMC4) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void SetEnableClock(Module module) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

        switch (module)
        {
        case Module_Sdmmc1:
            s_pRegisters->clkEnbLSet0 = REG_CLK_ENB_L_SET_0__SDMMC1;
            return;
        case Module_Sdmmc2:
            s_pRegisters->clkEnbLSet0 = REG_CLK_ENB_L_SET_0__SDMMC2;
            return;
        case Module_Sdmmc3:
            s_pRegisters->clkEnbUSet0 = REG_CLK_ENB_U_SET_0__SDMMC3;
            return;
        case Module_Sdmmc4:
            s_pRegisters->clkEnbLSet0 = REG_CLK_ENB_L_SET_0__SDMMC4;
            return;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void ClearEnableClock(Module module) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);

        switch (module)
        {
        case Module_Sdmmc1:
            s_pRegisters->clkEnbLClr0 = REG_CLK_ENB_L_CLR_0__SDMMC1;
            return;
        case Module_Sdmmc2:
            s_pRegisters->clkEnbLClr0 = REG_CLK_ENB_L_CLR_0__SDMMC2;
            return;
        case Module_Sdmmc3:
            s_pRegisters->clkEnbUClr0 = REG_CLK_ENB_U_CLR_0__SDMMC3;
            return;
        case Module_Sdmmc4:
            s_pRegisters->clkEnbLClr0 = REG_CLK_ENB_L_CLR_0__SDMMC4;
            return;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_K1))
        void SetClockSorceSdmmc(uint32_t* pOutActualClockFrequencyKHz, Module module, uint32_t targetClockFrequencyKHz) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(pOutActualClockFrequencyKHz);
            uint8_t n;  // Divided by "N = n + 1", n = N - 1, lsb denotes 0.5x
            switch (targetClockFrequencyKHz)
            {
            case 25000:
                n = (15U << 1) | 0x1U;  // PLLP 408MHz / 16.5 = 24.7272..MHz
                *pOutActualClockFrequencyKHz = 24728;
                break;
            case 26000:
                n = (15U << 1);         // PLLP 408MHz / 16 = 25.5MHz
                *pOutActualClockFrequencyKHz = 25500;
                break;
            case 50000:
                n = (7U << 1) | 0x1U;   // PLLP 408MHz / 8.5 = 48MHz
                *pOutActualClockFrequencyKHz = 48000;
                break;
            case 52000:
                n = (7U << 1);          // PLLP 408MHz / 8 = 51MHz
                *pOutActualClockFrequencyKHz = 51000;
                break;
            case 208000:    // MaxTargetClockFrequencyKHz
                n = (1U << 1);          // PLLP 408MHz / 2 = 204MHz
                *pOutActualClockFrequencyKHz = 204000;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            // PLLP 選択固定
            NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);
            switch (module)
            {
            case Module_Sdmmc1:
                s_pRegisters->clkSourceSdmmc1 = n;
                break;
            case Module_Sdmmc2:
                s_pRegisters->clkSourceSdmmc2 = n;
                break;
            case Module_Sdmmc3:
                s_pRegisters->clkSourceSdmmc3 = n;
                break;
            case Module_Sdmmc4:
                s_pRegisters->clkSourceSdmmc4 = n;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    #else   // NN_BUILD_CONFIG_SOC_TEGRA_X1
        void SetClockSorceSdmmc(uint32_t* pOutActualClockFrequencyKHz, Module module, uint32_t targetClockFrequencyKHz) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(pOutActualClockFrequencyKHz);
            uint32_t clkM = 0;  // デフォルト PLLP 選択
            uint8_t n;  // Divided by "X = (N / 2) + 1", N = (X - 1) * 2
            switch (targetClockFrequencyKHz)
            {
            case 25000:
                n = 31; // PLLP 408MHz / 16.5 = 24.7272..MHz
                *pOutActualClockFrequencyKHz = 24728;
                break;
            case 26000:
                n = 30; // PLLP 408MHz / 16 = 25.5MHz
                *pOutActualClockFrequencyKHz = 25500;
                break;
            case 40800:
                n = 18; // PLLP 408MHz / 10 = 40.8MHz
                *pOutActualClockFrequencyKHz = 40800;
                break;
            case 50000:
                n = 15; // PLLP 408MHz / 8.5 = 48MHz
                *pOutActualClockFrequencyKHz = 48000;
                break;
            case 52000:
                n = 14; // PLLP 408MHz / 8 = 51MHz
                *pOutActualClockFrequencyKHz = 51000;
                break;
            case 100000:
                #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1) && defined(NN_DETAIL_SDMMC_SET_PLLC4_DIRECTLY))
                    clkM = REG_CLK_SOURCE_SDMMCX_0__CLK_M__PLLC4_OUT2;
                    n = 2;  // PLLC4_OUT 199.68MHz / 2 = 99.84MHz (PLLC4_OUT2_LJ は分周できない)
                    *pOutActualClockFrequencyKHz = 99840;
                #else
                    n = 7;  // PLLP 408MHz / 4.5 = 90.666..MHz
                    *pOutActualClockFrequencyKHz = 90667;
                #endif
                break;
            case 200000:
                #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1) && defined(NN_DETAIL_SDMMC_SET_PLLC4_DIRECTLY))
                    NN_ABORT_UNLESS((module == Module_Sdmmc2) || (module == Module_Sdmmc4));
                    clkM = REG_CLK_SOURCE_SDMMC24_0__CLK_M__PLLC4_OUT2_LJ;
                    n = 0;  // PLLC4_OUT2_LJ 199.68MHz / 1 = 199.68MHz
                    *pOutActualClockFrequencyKHz = 199680;
                #else
                    n = 3;  // PLLP 408MHz / 2.5 = 163.2MHz
                    *pOutActualClockFrequencyKHz = 163200;
                #endif
                break;
            case 208000:    // MaxTargetClockFrequencyKHz
                n = 2;  // PLLP 408MHz / 2 = 204MHz
                *pOutActualClockFrequencyKHz = 204000;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            NN_ABORT_UNLESS_NOT_NULL(s_pRegisters);
            switch (module)
            {
            case Module_Sdmmc1:
                s_pRegisters->clkSourceSdmmc1 = n | clkM;
                break;
            case Module_Sdmmc2:
                s_pRegisters->clkSourceSdmmc2 = n | clkM;
                break;
            case Module_Sdmmc3:
                s_pRegisters->clkSourceSdmmc3 = n | clkM;
                break;
            case Module_Sdmmc4:
                s_pRegisters->clkSourceSdmmc4 = n | clkM;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    #endif
}

void Initialize() NN_NOEXCEPT
{
    uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ClockResetControllerRegistersPhysicalAddress, ClockResetControllerRegistersSize);
    NN_ABORT_UNLESS(registersAddress != 0);
    s_pRegisters = reinterpret_cast<ClockResetControllerRegisters*>(registersAddress);

    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1) && defined(NN_DETAIL_SDMMC_SET_PLLC4_DIRECTLY))
        InitializePllc4();
    #endif
    #if (defined(NN_BUILD_CONFIG_SOC_TEGRA_X1))
        InitializeTmclk();
    #endif
}

bool IsAvailable(Module module) NN_NOEXCEPT
{
    return (IsResetReleased(module) && IsClockEnabled(module));
}

void SetClockFrequencyKHz(uint32_t* pOutActualClockFrequencyKHz, Module module, uint32_t targetClockFrequencyKHz) NN_NOEXCEPT
{
    bool isClockEnabled = IsClockEnabled(module);
    if (isClockEnabled)
    {
        ClearEnableClock(module);
    }

    SetClockSorceSdmmc(pOutActualClockFrequencyKHz, module, targetClockFrequencyKHz);

    if (isClockEnabled)
    {
        SetEnableClock(module);
    }
}

void Reset(Module module) NN_NOEXCEPT
{
    if (IsClockEnabled(module))
    {
        ClearEnableClock(module);
    }
    SetReset(module);
    uint32_t actualSourceClockFrequencyKHz;
    SetClockFrequencyKHz(&actualSourceClockFrequencyKHz, module, MaxTargetClockFrequencyKHz);
    SetEnableClock(module);
    WaitClocks(100, actualSourceClockFrequencyKHz);
    ClearReset(module);
}

} // namespace ClockResetController {

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
