﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <list>
#include <string>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_Argument.h>
#include <nn/os/os_Thread.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nnt.h>


// max number of files to copy from the SD card
static const int MAX_FILE_TO_COPY = 100;


// copy files from the SD card to the host system, or from host system to the SD card
void CopySdCardFiles(const char* pReadDir, const char* pWriteDir)
{
    nn::fs::DirectoryHandle dirHandle;
    nn::Result result;
    nn::fs::DirectoryEntry dirEntry[MAX_FILE_TO_COPY];
    int64_t numFilesRead = 0;
    bool finishedReadingAllFiles = false;

    // mount SD card
    result = nn::fs::MountSdCardForDebug("sd");
    if (result.IsFailure())
    {
        NN_LOG("Failed to mount SD card (Description: %d)\n", result.GetDescription());
        return;
    }

    // mount host file system
    result = nn::fs::MountHostRoot();
    if (result.IsFailure())
    {
        NN_LOG("Failed to mount host (Description: %d)\n", result.GetDescription());
        return;
    }

    result = nn::fs::OpenDirectory(&dirHandle, pReadDir, nn::fs::OpenDirectoryMode_All);
    if (result.IsFailure())
    {
        // failed to open the read directory, exit
        NN_LOG("Failed to open the ready directory (%s)\n", pReadDir);
        nn::fs::Unmount("sd");
        nn::fs::UnmountHostRoot();
        return;
    }

    // create the directory to write to, if it doesn't exist
    // create the directory recursively from the root
    // eg. C:\Windows\temp  or  sd:/NxBinLogs/temp
    // for C:\Windows\temp, create C:\Windows first, then create C:\Windows\temp

    // index for the subdir
    char tempDir[256] = "\0";
    int indexOfFirstSlash = 0;

    for (int i = 0; i < strlen(pWriteDir); i++)
    {
        // copy directory to tempDir to be created later
        tempDir[i] = pWriteDir[i];

        if ((pWriteDir[i] == '/') || (pWriteDir[i] == '\\'))
        {
            if (indexOfFirstSlash == 0)
            {
                // start index of sub dir
                indexOfFirstSlash = i;
            }
            else
            {
                // create directory, if necessary
                nn::fs::CreateDirectory(tempDir);
            }
        }
    }

    // create the final child directory of directory of pWriteDir
    nn::fs::CreateDirectory(pWriteDir);

    // read until all files in the directory have been read
    while (!finishedReadingAllFiles)
    {
        result = nn::fs::ReadDirectory(&numFilesRead, dirEntry, dirHandle, MAX_FILE_TO_COPY);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // each file on the SD card needs to be read, and written to the host
        for (int i = 0; i < numFilesRead; i++)
        {
            if (dirEntry[i].directoryEntryType == nn::fs::DirectoryEntryType_File)
            {
                nn::fs::FileHandle readFileHandle;
                nn::fs::FileHandle writeFileHandle;
                std::string readFileName(pReadDir);
                std::string writeFileName(pWriteDir);

                // read the file on the SD card
                readFileName.append("/");
                readFileName.append(dirEntry[i].name);
                result = nn::fs::OpenFile(&readFileHandle, readFileName.c_str(), nn::fs::OpenMode_Read);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                // copy to a file on the host
                writeFileName.append("/");
                writeFileName.append(dirEntry[i].name);
                nn::fs::DeleteFile(writeFileName.c_str());
                result = nn::fs::CreateFile(writeFileName.c_str(), dirEntry[i].fileSize);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                result = nn::fs::OpenFile(&writeFileHandle, writeFileName.c_str(), nn::fs::OpenMode_Write);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                // read and write the data
                size_t outValue;
                char* fileData = reinterpret_cast<char*>(malloc(static_cast<size_t>(dirEntry[i].fileSize)));
                memset(fileData, 0, static_cast<size_t>(dirEntry[i].fileSize));
                result = nn::fs::ReadFile(&outValue, readFileHandle, 0, fileData, static_cast<size_t>(dirEntry[i].fileSize));
                NN_ASSERT_EQUAL(outValue, dirEntry[i].fileSize);

                result = nn::fs::WriteFile(writeFileHandle, 0, fileData, static_cast<size_t>(dirEntry[i].fileSize), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                free(fileData);
                nn::fs::CloseFile(readFileHandle);
                nn::fs::CloseFile(writeFileHandle);
            }
        }

        // check if we have read all files in the directory
        if (numFilesRead < MAX_FILE_TO_COPY)
        {
            finishedReadingAllFiles = true;
        }
    }

    nn::fs::CloseDirectory(dirHandle);

    // unmount file systems
    nn::fs::Unmount("sd");
    nn::fs::UnmountHostRoot();
}


extern "C" void nnMain()
{
    // input parameters we need
    // eg.  GetSdCardFiles.nsp --read_dir=sd:/NxBinLogs --write_dir=C:/Temp
    const char read_dir[] = "--read_dir";
    const char write_dir[] = "--write_dir";

    // set the default read and write dirs
    std::string strReadDir("sd:/NxBinLogs");
    std::string strWriteDir("C:/Temp");

    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    for (int i = 0; i < argc; i++)
    {
        if (strncasecmp(argv[i], read_dir, strlen(read_dir)) == 0)
        {
            NN_LOG("argv %i    : %s\n", i, argv[i]);
            strReadDir = &(argv[i][strlen(read_dir) + 1]);
        }
        else if (strncasecmp(argv[i], write_dir, strlen(write_dir)) == 0)
        {
            NN_LOG("argv %i    : %s\n", i, argv[i]);
            strWriteDir = &(argv[i][strlen(write_dir) + 1]);
        }
        else
        {
            NN_LOG("argv %s is not recognized\n", argv[i]);
        }
    }

    CopySdCardFiles(strReadDir.c_str(), strWriteDir.c_str());

    return;
}
