﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.NintendoSdkVsExtension.Base;
using Nintendo.NintendoSdkVsExtension.Serialization.V1;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Schema;
using System.Xml.Serialization;

namespace Nintendo.NintendoSdkVsExtension.UnitTest
{
    [TestClass]
    [DeploymentItem(_TestDataXmlPath, "Data")]
    public class SerializarionSdkProjectSettingsTest
    {
        private const string _TestDataXmlPath = @"Data\NintendoSdkProjectSettings.V1.nnsdk.xml";

        [TestMethod]
        public void TestTotalRoundTrip()
        {
            using (var ms = new MemoryStream())
            {
                var serializer = new SdkProjectSettingsSerializer();
                var store1 = CreateStore();

                serializer.Serialize(ms, new SerializationSdkProjectSettings(store1));
                ms.Flush();
                ms.Seek(0, SeekOrigin.Begin);
                var store2 = serializer.Deserialize(ms).ToStore();

                AssertUtil.AssertAreDeepEqualByPublicProperties(store1, store2);
            }
        }

        [TestMethod]
        public void TestRoundTrip()
        {
            using (var ms = new MemoryStream())
            {
                var serializer = new SdkProjectSettingsSerializer();
                var store1 = CreateSerializationStore();

                serializer.Serialize(ms, store1);
                ms.Flush();
                ms.Seek(0, SeekOrigin.Begin);
                var store2 = serializer.Deserialize(ms);

                AssertUtil.AssertAreDeepEqualByPublicProperties(store1, store2);
            }
        }

        [TestMethod]
        public void TestSerialization()
        {
            using (var ms = new MemoryStream())
            {
                var serializer = new SdkProjectSettingsSerializer();
                var store1 = CreateSerializationStore();

                serializer.Serialize(ms, store1);
                ms.Flush();
                ms.Seek(0, SeekOrigin.Begin);
                var xdoc = XDocument.Load(ms);

                var expected = LoadTestData();
                Assert.IsTrue(XNode.DeepEquals(expected, xdoc.Root));
            }
        }

        [TestMethod]
        public void TestDeserialization()
        {
            using (var fs = File.OpenRead(_TestDataXmlPath))
            {
                var serializer = new SdkProjectSettingsSerializer();
                var store1 = CreateSerializationStore();

                var store2 = serializer.Deserialize(fs);

                AssertUtil.AssertAreDeepEqualByPublicProperties(store1, store2);
            }
        }

        private static SdkProjectSettingsStore CreateStore()
        {
            return new SdkProjectSettingsStore(
                new[] {
                    new SdkConfigurationSettingsStore(new ConfigurationPair("Debug|Win32"))
                    {
                        IsSdkEnabled = false,
                        SdkRoot = new SdkRoot(SdkRootKind.DirectProperty, string.Empty),
                        Spec = string.Empty,
                        BuildType = SdkBuildType.Debug,
                    },
                    new SdkConfigurationSettingsStore(new ConfigurationPair("Develop|Win32"))
                    {
                        SdkRoot = new SdkRoot(SdkRootKind.DirectProperty, "$(NINTENDO_SDK_ROOT)"),
                        IsSdkEnabled = true,
                        Spec = "Generic",
                        BuildType = SdkBuildType.Develop,
                    },
                    new SdkConfigurationSettingsStore(new ConfigurationPair("Release|Win32"))
                    {
                        SdkRoot = new SdkRoot(SdkRootKind.NdiEnvironmentName, "Sdk1"),
                        IsSdkEnabled = true,
                        Spec = "Generic",
                        BuildType = SdkBuildType.Release,
                    }
                });
        }

        private static SerializationSdkProjectSettings CreateSerializationStore()
        {
            return new SerializationSdkProjectSettings()
            {
                Version = "1.0",
                ProjectSetting = new SerializationProjectSetting()
                {
                    ConfigurationSettings = new SerializationConfigurationSetting[]
                    {
                        new SerializationConfigurationSetting()
                        {
                            Name = "Debug|Win32",
                            IsSdkEnabled = false,
                            SdkRoot = new SerializationSdkRoot { Kind = SdkRootKind.DirectProperty, Value = string.Empty },
                            Spec = string.Empty,
                            BuildType = SdkBuildType.Debug,
                        },
                        new SerializationConfigurationSetting()
                        {
                            Name = "Develop|Win32",
                            SdkRoot = new SerializationSdkRoot { Kind = SdkRootKind.DirectProperty, Value = "$(NINTENDO_SDK_ROOT)" },
                            IsSdkEnabled = true,
                            Spec = "Generic",
                            BuildType = SdkBuildType.Develop,
                        },
                        new SerializationConfigurationSetting()
                        {
                            Name = "Release|Win32",
                            SdkRoot = new SerializationSdkRoot { Kind = SdkRootKind.NdiEnvironmentName, Value = "Sdk1" },
                            IsSdkEnabled = true,
                            Spec = "Generic",
                            BuildType = SdkBuildType.Release,
                        },
                    }
                }
            };
        }

        private static XElement LoadTestData()
        {
            return XDocument.Load(_TestDataXmlPath).Root;
        }
    }
}
