﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>

#include <nn/nn_Assert.h>

#include "TestAppSimple_WorkContractor.h"

namespace {

    const nn::os::Tick ZeroTick;
}

Work::Work() NN_NOEXCEPT
    : m_LastResult(nn::ResultSuccess())
    , m_Lock(true)
    , m_State(State_NotPrepared)
    , m_Begin(0)
    , m_End(0)
    , m_IsCancelRequested(false)
{
}

Work::~Work() NN_NOEXCEPT
{
}

bool Work::IsPrepared() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    return (m_State == State_Prepared);
}

bool Work::IsExecuted() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    return (m_State == State_Executed);
}

bool Work::IsCancelRequested() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    return m_IsCancelRequested;
}

nn::TimeSpan Work::GetElapsedTime() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    nn::TimeSpan span;
    if (m_State == State_Executed)
    {
        span = nn::os::ConvertToTimeSpan(m_End - m_Begin);
    }
    return span;
}

nn::Result Work::GetLastResult() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    NN_ASSERT_EQUAL(m_State, State_Executed);
    return m_LastResult;
}

void Work::Cleanup() NN_NOEXCEPT
{
    SetStateToNotPrepared();
}

void Work::SetStateToNotPrepared() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    NN_ASSERT_NOT_EQUAL(m_State, State_Executing);
    m_State = State_NotPrepared;
}

void Work::SetStateToPrepared() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    NN_ASSERT_NOT_EQUAL(m_State, State_Executing);
    m_LastResult = nn::ResultSuccess();
    m_Begin = ZeroTick;
    m_End = ZeroTick;
    m_IsCancelRequested = false;
    m_State = State_Prepared;
}

void Work::SetStateToExecuting() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    NN_ASSERT_EQUAL(m_State, State_Prepared);
    m_Begin = nn::os::GetSystemTick();
    m_State = State_Executing;
}

void Work::SetStateToExecuted() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    NN_ASSERT_EQUAL(m_State, State_Executing);
    m_End = nn::os::GetSystemTick();
    m_State = State_Executed;
}

void Work::RequestToCancel() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    m_IsCancelRequested = true;
}

WorkContractor::WorkContractor(void* pStack, size_t stackSize) NN_NOEXCEPT
    : m_Lock(true)
    , m_State(State_NotInitialized)
    , m_pStackBuffer(pStack)
    , m_StackSize(stackSize)
{
}

WorkContractor::~WorkContractor() NN_NOEXCEPT
{
    Cancel();
    RequestToStop();
}

void WorkContractor::Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    NN_ASSERT_EQUAL(m_State, State_NotInitialized);

    nn::os::InitializeEvent(&m_StartEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_FinishEvent, false, nn::os::EventClearMode_AutoClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&m_Thread, ThreadFunc, this, m_pStackBuffer, m_StackSize, nn::os::DefaultThreadPriority) );
    nn::os::StartThread(&m_Thread);

    m_State = State_Runnable;
}

void WorkContractor::Finalize() NN_NOEXCEPT
{
    Cancel();
    if (RequestToStop())
    {
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);
        nn::os::FinalizeEvent(&m_StartEvent);
        nn::os::FinalizeEvent(&m_FinishEvent);
    }
}

void WorkContractor::Start(Work* pWork) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pWork);
    NN_ASSERT(pWork->IsPrepared());

    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    NN_ASSERT_EQUAL(m_State, State_Runnable);

    // 二重登録不可
    NN_ASSERT_EQUAL(std::find(m_Works.begin(), m_Works.end(), pWork), m_Works.end());

    // 登録時は空が必須
    NN_ASSERT(m_Works.empty());
    m_Works.push_back(pWork);

    // 作業開始要求
    nn::os::ClearEvent(&m_FinishEvent);
    nn::os::SignalEvent(&m_StartEvent);
}

void WorkContractor::Start(Work* pWorkArray[], size_t workCount) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pWorkArray);
    NN_ASSERT_GREATER(workCount, 0U);

    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    NN_ASSERT_EQUAL(m_State, State_Runnable);

    // 登録時は空が必須
    NN_ASSERT(m_Works.empty());
    for (size_t i = 0; i < workCount; i++)
    {
        auto pWork = pWorkArray[i];
        NN_ASSERT_NOT_NULL(pWork);
        NN_ASSERT(pWork->IsPrepared());

        // 二重登録不可
        NN_ASSERT_EQUAL(std::find(m_Works.begin(), m_Works.end(), pWork), m_Works.end());

        m_Works.push_back(pWork);
    }

    // 作業開始要求
    nn::os::ClearEvent(&m_FinishEvent);
    nn::os::SignalEvent(&m_StartEvent);
}

void WorkContractor::WaitFinish() NN_NOEXCEPT
{
    nn::os::WaitEvent(&m_FinishEvent);
}

bool WorkContractor::IsFinished() NN_NOEXCEPT
{
    return nn::os::TryWaitEvent(&m_FinishEvent);
}

bool WorkContractor::IsBusy() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    return !m_Works.empty();
}

void WorkContractor::Cancel() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    if (!m_Works.empty())
    {
        for (auto pWork : m_Works)
        {
            pWork->RequestToCancel();
        }
        m_Works.erase(m_Works.begin() + 1, m_Works.end());
    }
}

bool WorkContractor::IsRunnable() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    return (m_State == State_Runnable);
}

bool WorkContractor::RequestToStop() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    bool stop = (m_State == State_Runnable);
    if (stop)
    {
        m_State = State_Stopping;
        nn::os::SignalEvent(&m_StartEvent);
    }
    return stop;
}

Work* WorkContractor::FrontWork() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    auto it = m_Works.begin();
    return (it != m_Works.end() ? *it : nullptr);
}

bool WorkContractor::PopFrontWork(Work* pWork) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    auto it = m_Works.begin();
    if (it != m_Works.end() && *it == pWork)
    {
        m_Works.erase(it);
        return true;
    }
    return false;
}

void WorkContractor::ThreadFunc(void* arg) NN_NOEXCEPT
{
    WorkContractor* pContractor = reinterpret_cast<WorkContractor*>(arg);
    NN_ASSERT(pContractor->IsRunnable());

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::WaitEvent(&(pContractor->m_StartEvent));
        if (!pContractor->IsRunnable())
        {
            break;
        }

        while (NN_STATIC_CONDITION(true))
        {
            auto pWork = pContractor->FrontWork();
            if (pWork == nullptr)
            {
                break;
            }

            pWork->SetStateToExecuting();
            if (!pWork->Exeute())
            {
                pContractor->Cancel();
            }
            pWork->SetStateToExecuted();

            pContractor->PopFrontWork(pWork);
        }
        nn::os::SignalEvent(&(pContractor->m_FinishEvent));
    }
}
