﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/util/util_StringUtil.h>

#include "TestAppSimple_SceneCommon.h"

// static 変数なため、ここに定義しておく
nn::hid::TouchState SceneCommon::m_PreviousTouch;
LaunchStorageInfo SceneCommon::m_LaunchStorageInfo;
SceneCommon::ApplicationInfo SceneCommon::m_ApplicationInfo;

namespace {
    const auto PageButtonMerginX = 25.0f;
    const auto PageButtonPosY = 20.0f;
}

WarningView::WarningView() NN_NOEXCEPT
    : m_Title(""),
    m_TitleColor(Red),
    m_TitleScale(3.0f),
    m_TitlePosX(0.0f),
    m_TitlePosY(150.0f),
    m_MsgScale(2.4f),
    m_MsgPosX(0.0f),
    m_MsgPosY(270.0f),
    m_IsChanged(true),
    m_IsTitleCentering(true),
    m_IsMessageCentering(true)
{
};

void WarningView::SetTitle(const std::string& title) NN_NOEXCEPT
{
    m_Title = title;

    SetChanged(true);
}

void WarningView::SetTitleColor(const nn::util::Unorm8x4& color) NN_NOEXCEPT
{
    m_TitleColor = color;

    SetChanged(true);
}

void WarningView::SetTitleScale(float scale) NN_NOEXCEPT
{
    m_TitleScale = scale;

    SetChanged(true);
}

void WarningView::SetTitlePos(float x, float y) NN_NOEXCEPT
{
    m_TitlePosX = x;
    m_TitlePosY = y;

    SetChanged(true);
}

void WarningView::ClearMessage() NN_NOEXCEPT
{
    m_Messages.clear();

    SetChanged(true);
}

size_t WarningView::PushMessage(const std::string& text, const nn::util::Unorm8x4& color) NN_NOEXCEPT
{
    MesageParam msg;
    msg.text = text;
    msg.color = color;
    m_Messages.push_back(msg);

    SetChanged(true);

    return m_Messages.size() - 1;
}

void WarningView::UpdateMessage(size_t index, const std::string& text) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(index >= 0, true);
    NN_SDK_ASSERT_EQUAL(index < m_Messages.size(), true);

    m_Messages.at(index).text = text;

    SetChanged(true);
}

void WarningView::SetMessageScale(float scale) NN_NOEXCEPT
{
    m_MsgScale = scale;

    SetChanged(true);
}

void WarningView::SetMessagePos(float x, float y) NN_NOEXCEPT
{
    m_MsgPosX = x;
    m_MsgPosY = y;

    SetChanged(true);
}

bool WarningView::IsChanged() NN_NOEXCEPT
{
    return m_IsChanged;
}

void WarningView::SetChanged(bool isChanged) NN_NOEXCEPT
{
    m_IsChanged = isChanged;
}

bool WarningView::IsTitleCentering() NN_NOEXCEPT
{
    return m_IsTitleCentering;
}

void WarningView::SetTitleCentering(bool isCentering) NN_NOEXCEPT
{
    m_IsTitleCentering = isCentering;
}

bool WarningView::IsMessageCentering() NN_NOEXCEPT
{
    return m_IsMessageCentering;
}

void WarningView::SetMessageCentering(bool isCentering) NN_NOEXCEPT
{
    m_IsMessageCentering = isCentering;
}

void WarningView::CenteringTitle(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    if (m_IsTitleCentering == true)
    {
        writer->SetScale(m_TitleScale, m_TitleScale);
        m_TitlePosX = (ScreenWidth - writer->CalculateStringWidth(m_Title.c_str())) / 2.0f;
    }
}

void WarningView::CenteringMessage(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    if (m_IsMessageCentering == true)
    {
        writer->SetScale(m_MsgScale, m_MsgScale);
        std::string text;
        for (auto& msg : m_Messages)
        {
            text += msg.text;
        }

        m_MsgPosX = (ScreenWidth - writer->CalculateStringWidth(text.c_str())) / 2.0f;
    }
}

void WarningView::DrawView(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetScale(m_TitleScale, m_TitleScale);
    writer->SetCursor(m_TitlePosX, m_TitlePosY);
    writer->SetTextColor(m_TitleColor);
    writer->Print(m_Title.c_str());

    writer->SetScale(m_MsgScale, m_MsgScale);
    writer->SetCursor(m_MsgPosX, m_MsgPosY);
    for (auto& msg : m_Messages)
    {
        writer->SetTextColor(msg.color);
        writer->Print(msg.text.c_str());
    }
}

SceneCommon::SceneCommon() NN_NOEXCEPT
    : m_NextAction(NextAction::None), m_IsTouch(false), m_IsCheckingExit(false)
{
}

SceneCommon::~SceneCommon() NN_NOEXCEPT
{
}

void SceneCommon::SetLaunchStorageInfo(LaunchStorageInfo&& inStorageInfo) NN_NOEXCEPT
{
    m_LaunchStorageInfo = std::move(inStorageInfo);
}

void SceneCommon::Setup(nn::gfx::util::DebugFontTextWriter* writer, const nn::hid::TouchScreenState<1>& inTouchState, NextAction ownAction) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    m_PreviousTouch = inTouchState.touches[0];

    m_OwnAction = ownAction;
    this->SetPageInfo(writer, m_OwnAction);

    {
        m_Headline.pos.x = 350.0f;
        m_Headline.pos.y = 660.0f;
        m_Headline.labelStr = "Plus(Start): Headline";
    }

    {
        m_Exit.pos.x = 680.0f;
        m_Exit.pos.y = 660.0f;
        m_Exit.labelStr = "Minus(Select): Exit";
    }

    {
        m_ConfirmCancelRange.pos.x = 250.0f;
        m_ConfirmCancelRange.pos.y = 400.0f;
        m_ConfirmCancelRange.labelStr = "B: Cancel";

        m_ConfirmOkRange.pos.x = 800.0f;
        m_ConfirmOkRange.pos.y = 400.0f;
        m_ConfirmOkRange.labelStr = "A: OK";
    }

    m_ExitWarningView.PushMessage("Do you really want to close the application ?", White);

    this->InternalSetup();
}

void SceneCommon::EnterScene() NN_NOEXCEPT
{
}
void SceneCommon::LeaveScene() NN_NOEXCEPT
{
}

void SceneCommon::ResetState() NN_NOEXCEPT
{
    m_NextAction = NextAction::None;
}

void SceneCommon::HandleNPad() NN_NOEXCEPT
{
    if (m_IsCheckingExit == false)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::L::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::ZL::Mask))
        {
            m_NextAction = m_LeftPage.action;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::R::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::ZR::Mask))
        {
            m_NextAction = m_RightPage.action;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Plus::Mask))
        {
            SetJumpPage(NextAction::Headline);
        }

        this->InternalHandleNPad();
    }

    this->InternalExitNPadHandleProcess();
}

void SceneCommon::HandleTouchScreen(const nn::hid::TouchScreenState<1>& state) NN_NOEXCEPT
{
    if (state.count > 0)
    {
        m_PreviousTouch = state.touches[0];
        m_IsTouch = true;
    }
    else
    {
        if (m_IsTouch == true)
        {
            if (m_IsCheckingExit == false)
            {
                if (m_LeftPage.package.range.IsInRange(m_PreviousTouch) == true)
                {
                    m_NextAction = m_LeftPage.action;
                }
                else if (m_RightPage.package.range.IsInRange(m_PreviousTouch) == true)
                {
                    m_NextAction = m_RightPage.action;
                }
                else if (m_Headline.range.IsInRange(m_PreviousTouch) == true)
                {
                    SetJumpPage(NextAction::Headline);
                }

                this->InternalHandleTouchScreen();
            }
            this->InternalExitHandleProcess();
            m_IsTouch = false;
        }
    }
}

void SceneCommon::InternalExitNPadHandleProcess() NN_NOEXCEPT
{
    if (m_IsCheckingExit == false)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Minus::Mask))
        {
            m_IsCheckingExit = true;
        }
    }
    else
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_NextAction = NextAction::Exit;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            m_IsCheckingExit = false;
        }
    }
}

void SceneCommon::InternalExitHandleProcess() NN_NOEXCEPT
{
    if (m_IsCheckingExit == false)
    {
        if (m_Exit.range.IsInRange(m_PreviousTouch) == true)
        {
            m_IsCheckingExit = true;
        }
    }
    else
    {
        if (m_ConfirmOkRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_NextAction = NextAction::Exit;
        }
        else if (m_ConfirmCancelRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_IsCheckingExit = false;
        }
    }
}

NextAction SceneCommon::Process() NN_NOEXCEPT
{
    this->InternalProcess();
    return m_NextAction;
}

void SceneCommon::DrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    this->SetDefaultScale(writer);

    writer->SetTextColor(White);
    writer->SetCursor(390.0f, 11.0f);
    writer->Print("Test Application");

    {
        writer->SetScale(1.0f, 1.0f);
        writer->SetCursor(380.0f, 42.0f);
        // (SIGLO-78007) ProgramId を表示する (ProgramIndex が設定されていない場合は従来通り ApplicationId が表示される)
        writer->Print("[ %s ]", m_ApplicationInfo.programId.c_str());
    }

    // 起動ストレージ情報の書き出し (ひとまず共有部分中央辺りに表示しておく)
    {
        writer->SetScale(1.2f, 1.2f);

        writer->SetCursor(640.0f, 10.0f);
        writer->Print("LaunchStorage: %s", m_LaunchStorageInfo.launch.c_str());
        writer->SetCursor(648.0f, 32.0f);
        writer->Print("PatchStorage : %s", m_LaunchStorageInfo.patch.c_str());

        this->SetDefaultScale(writer);
    }

#if 0
    // タッチした点(範囲)を描画するデバック用関数
    this->DrowDebugTouchRange(writer);
#endif

    if (m_IsCheckingExit == true)
    {
        // 警告画面の描画
        this->DrawWarningMessage(writer, &m_ExitWarningView);

        // 以降の処理は書き出さない
        return;
    }

    this->InternalDrawDebugText(writer);

    this->SetDefaultScale(writer);

    this->WriteTouchRange(writer, &m_RightPage.package);
    this->WriteTouchRange(writer, &m_LeftPage.package);
    this->WriteTouchRange(writer, &m_Exit);

    writer->SetTextColor(Gray);
    m_Headline.range.isTouchable = m_OwnAction != NextAction::Headline;
    this->WriteTouchRange(writer, &m_Headline);
}

void SceneCommon::SetDefaultScale(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetScale(1.5f, 1.5f);
}

void SceneCommon::DrowDebugTouchRange(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    if (m_IsTouch == false)
    {
        return;
    }

    // タッチ範囲のデバック用表示処理
    writer->SetTextColor(Aqua);

    const float barOffsetX = writer->CalculateStringWidth("|") / 2;
    const float barOffsetY = writer->CalculateStringHeight("|") / 2;
    const float minusOffsetX = writer->CalculateStringWidth("-") / 2;
    const float minusOffsetY = writer->CalculateStringHeight("-") / 2;
    const float plusOffsetX = writer->CalculateStringWidth("+") / 2;
    const float plusOffsetY = writer->CalculateStringHeight("+") / 2;

    const nn::hid::TouchState& state = m_PreviousTouch;
    writer->SetCursor(state.x - barOffsetX - 45,
        state.y - barOffsetY);
    writer->Print("|");
    writer->SetCursor(state.x - barOffsetX + 45,
        state.y - barOffsetY);
    writer->Print("|");
    writer->SetCursor(state.x - minusOffsetX,
        state.y - minusOffsetY - 45);
    writer->Print("-");
    writer->SetCursor(state.x - minusOffsetX,
        state.y - minusOffsetY + 45);
    writer->Print("-");
    writer->SetCursor(state.x - plusOffsetX - 45,
        state.y - plusOffsetY - 45);
    writer->Print("+");
    writer->SetCursor(state.x - plusOffsetX - 45,
        state.y - plusOffsetY + 45);
    writer->Print("+");
    writer->SetCursor(state.x - plusOffsetX + 45,
        state.y - plusOffsetY - 45);
    writer->Print("+");
    writer->SetCursor(state.x - plusOffsetX + 45,
        state.y - plusOffsetY + 45);
    writer->Print("+");

    writer->SetCursor(state.x - 43.0f, state.y - 39.0f);
    writer->SetScale(0.8f, 0.8f);
    writer->Print("(%d, %d)", state.x, state.y);

    this->SetDefaultScale(writer);
    writer->SetTextColor(White);
}

void SceneCommon::WriteTouchRange(nn::gfx::util::DebugFontTextWriter* writer, const Position& inPos, const char* inStr, TouchRange* inoutRange, bool isAddBracket) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    if (inoutRange->isSetting == false)
    {
        const float height = writer->CalculateStringHeight(inStr);
        const float width = writer->CalculateStringWidth(inStr);

        inoutRange->beginPos = inPos;
        inoutRange->endPos.x = inPos.x + width;
        inoutRange->endPos.y = inPos.y + height + 25;

        inoutRange->isSetting = true;
        inoutRange->labelStr = inStr;
    }

    writer->SetCursor(inPos.x, inPos.y);
    if (isAddBracket)
    {
        writer->SetTextColor(White);
        writer->Print("[ ");
    }

    if (inoutRange->isTouchable == true)
    {
        writer->SetTextColor(LinkBlue);
    }
    writer->Print(inStr);

    if (isAddBracket)
    {
        writer->SetTextColor(White);
        writer->Print(" ]");
    }
}

void SceneCommon::WriteTouchRange(nn::gfx::util::DebugFontTextWriter* writer, RangePackage* inPackage, bool isAddBracket) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    this->WriteTouchRange(writer, inPackage->pos, inPackage->labelStr.c_str(), &inPackage->range, isAddBracket);
}

void SceneCommon::SetPageInfo(nn::gfx::util::DebugFontTextWriter* writer, NextAction inSetAction) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    int leftIndex = 0;
    int rightIndex = 0;
    for (int i = 0; i < PageCountMax; i++)
    {
        if (ActionPairList[i].Action == inSetAction)
        {
            leftIndex = (i + PageCountMax - 1) % PageCountMax;
            rightIndex = (i + 1) % PageCountMax;

            break;
        }
    }

    m_LeftPage.action = ActionPairList[leftIndex].Action;
    m_LeftPage.package.labelStr.append("L: ").append(ActionPairList[leftIndex].Label);
    m_LeftPage.package.pos.x = PageButtonMerginX;
    m_LeftPage.package.pos.y = PageButtonPosY;

    m_RightPage.action = ActionPairList[rightIndex].Action;
    m_RightPage.package.labelStr.append("R: ").append(ActionPairList[rightIndex].Label);
    // 右ボタンは文字長を計り画面右端から指定マージンにボタン右端を配置する
    this->SetDefaultScale(writer);
    const auto width = writer->CalculateStringWidth(m_RightPage.package.labelStr.c_str());
    m_RightPage.package.pos.x = ScreenWidth - width - PageButtonMerginX;
    m_RightPage.package.pos.y = PageButtonPosY;
}

void SceneCommon::DrawWarningMessage(nn::gfx::util::DebugFontTextWriter* writer, WarningView* warningView) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);
    NN_ABORT_UNLESS_NOT_NULL(warningView);

    if (warningView->IsChanged() == true)
    {
        warningView->SetChanged(false);
        warningView->CenteringTitle(writer);
        warningView->CenteringMessage(writer);
    }

    warningView->DrawView(writer);

    writer->SetScale(3.5f, 3.5f);
    this->WriteTouchRange(writer, &m_ConfirmCancelRange);
    this->WriteTouchRange(writer, &m_ConfirmOkRange);
}

void SceneCommon::SetJumpPage(NextAction targetPage) NN_NOEXCEPT
{
    m_NextAction = targetPage;
}
