﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

// nn::mii 系モジュールは Generic では扱わない
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nn/mii/mii_Applet.h>
#include <nn/mii/mii_Result.h>
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

#include "TestAppSimple_MiiScene.h"

namespace {
    const float Label_X_Pos = 60.0f;
    const float Label_Y_Pos = 130.0f;
    const float Label_Y_Width = 90.0f;
};

MiiScene::MiiScene() NN_NOEXCEPT
    : m_State(State_None), m_SelectedLauchType(LaunchType_None),
    m_CurrentSelectPos(0), m_CurrentMaxPosNum(MaxItemCount - 1)
{
}

void MiiScene::InternalSetup() NN_NOEXCEPT
{
    // 表示するラベルと対応する呼び出し API の enum 値をまとめて設定
    {
        struct labelPair
        {
            const char* label;
            LaunchType  type;
        } labelList[MaxItemCount] = {
            { "AppendMii", LaunchType_AppendMii },
        };

        for (int i = 0; i < MaxItemCount; ++i)
        {
            m_LaunchRangeList[i].range.pos.x = Label_X_Pos;
            m_LaunchRangeList[i].range.pos.y = Label_Y_Pos + (Label_Y_Width * i);
            m_LaunchRangeList[i].range.labelStr = labelList[i].label;
            m_LaunchRangeList[i].type = labelList[i].type;
        }
    }
}

void MiiScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State != State_None)
    {
        // State_None 以外は受け付けないようにしておく
        return;
    }

    if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
    {
        m_State = State_Launching;
        m_SelectedLauchType = m_LaunchRangeList[m_CurrentSelectPos].type;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
    {
        if (m_CurrentSelectPos < m_CurrentMaxPosNum)
        {
            ++m_CurrentSelectPos;
        }
        else
        {
            m_CurrentSelectPos = 0;
        }
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
    {
        if (m_CurrentSelectPos > 0)
        {
            --m_CurrentSelectPos;
        }
        else
        {
            m_CurrentSelectPos = m_CurrentMaxPosNum;
        }
    }
}

void MiiScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State != State_None)
    {
        // State_None 以外は受け付けないようにしておく
        return;
    }

    int count = 0;
    for (auto& lr : m_LaunchRangeList)
    {
        if (lr.range.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_Launching;
            m_SelectedLauchType = lr.type;
            m_CurrentSelectPos = count;
            break;
        }
        ++count;
    }
}

void MiiScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_Launching)
    {
        // nn::mii 系モジュールは Generic では扱わない
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        nn::Result result;
        switch (m_SelectedLauchType)
        {
        case LaunchType_AppendMii:
            {
                int index; // この変数は特に利用しない
                // Mii 追加のアプレットを呼び出す
                result = nn::mii::AppendMii(&index);
            }
            break;
        case LaunchType_None:
        default:
            // ここに来ることはないと思うが何もしない
            break;
        }

        if (result.IsFailure())
        {
            if (nn::mii::ResultCanceled::Includes(result) == false)
            {
                NN_LOG("[Error] MiiScene::InternalProcess() Launch Applet Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
            }
        }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        m_State = State_None;
    }
}

void MiiScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[Mii]");

    writer->SetScale(1.8f, 1.8f);
    {
        writer->SetTextColor(White);
        writer->SetCursor(Label_X_Pos - 30.0f, Label_Y_Pos + (Label_Y_Width * (m_CurrentSelectPos)));
        writer->Print("*");
    }

    for (auto& lr : m_LaunchRangeList)
    {
        this->WriteTouchRange(writer, &lr.range);
    }

    if (m_State == State_None)
    {
        writer->SetTextColor(White);
        writer->SetCursor(100.0f, 600.0f);
        writer->Print("A: Launch Applet");
    }
}
